package jp.kirikiri.tjs2;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public abstract class BinaryStream {

	public static final int READ = 0;
	public static final int WRITE = 1;
	public static final int APPEND = 2;
	public static final int UPDATE = 3;

	public static final int ACCESS_MASK = 0x0f;

	public static final int SEEK_SET = 0;
	public static final int SEEK_CUR = 1;
	public static final int SEEK_END = 2;

	/**
	 * if error, position is not changed
	 * @throws TJSException
	 */
	public abstract long seek( long offset, int whence ) throws TJSException;

	/**
	 * returns actually read size
	 */
	public abstract int read( ByteBuffer buffer );

	/**
	 * returns actually written size
	 */
	public abstract int write( final ByteBuffer buffer );

	/**
	 * close stream
	 */
	public abstract void close();

	/**
	 * the default behavior is raising a exception
	 * if error, raises exception
	 */
	public void setEndOfStorage() throws TJSException {
		throw new TJSException( Error.WriteError );
	}

	/**
	 * should re-implement for higher performance
	 * @throws TJSException
	 */
	public long getSize() throws TJSException {
		long orgpos = getPosition();
		long size = seek(0, SEEK_END );
		seek( orgpos, SEEK_SET );
		return size;
	}

	public long getPosition() throws TJSException {
		return seek( 0, SEEK_CUR );
	}

	public void setPosition( long pos ) throws TJSException {
		if( pos != seek( pos, SEEK_SET ) ) {
			throw new TJSException( Error.SeekError );
		}
	}

	public void readBuffer( ByteBuffer buffer ) throws TJSException {
		if( read( buffer ) != -1 ) {
			throw new TJSException( Error.ReadError );
		}
		buffer.flip();
	}
	public void writeBuffer( final ByteBuffer buffer ) throws TJSException {
		if( write( buffer ) != -1 ) {
			throw new TJSException( Error.WriteError );
		}
	}

	// reads little-endian integers
	public long readI64LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(8);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getLong();
	}
	public int readI32LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getInt();
	}
	public short readI16LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getShort();
	}
}

