/**
 * @file ntlibc.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * A tiny libc like module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "hal.h"

int ntlibc_strlen(const char *s)
{
    const char *p = s;
    int cnt = 0;
    while (*p) {
        cnt++;
        p++;
    }
    return cnt;
}

char *ntlibc_strcpy(char *des, const char *src)
{
    char *d = des;
    const char *s = src;
    while (*s) {
        *d = *s;
        d++;
        s++;
    }
    *d = '\0';
    return des;
}

char *ntlibc_strcat(char *des, const char *src)
{
    char *d = des;
    const char *s = src;
    while (*d) {
        d++;
    }
    while (*s) {
        *d = *s;
        d++;
        s++;
    }
    *d = '\0';
    return des;
}

int ntlibc_strcmp(const char *s1, const char *s2)
{
    char *p1 = (char *)s1;
    char *p2 = (char *)s2;
    while (*p1 || *p2) {
        if (*p1 != *p2) {
            return (*p1 < *p2) ? -1 : 1;
        }
        p1++;
        p2++;
    }
    if (*p1 == *p2) {
        return 0;
    } else {
        return (*p1 < *p2) ? -1 : 1;
    }
}

int ntlibc_strncmp(const char *s1, const char *s2, int n)
{
    char *p1 = (char *)s1;
    char *p2 = (char *)s2;
    int len = 0;
    while (*p1 || *p2) {
        if (n <= len) {
            break;
        }
        if (*p1 != *p2) {
            return (*p1 < *p2) ? -1 : 1;
        }
        p1++;
        p2++;
        len++;
    }
    return 0;
}

int ntlibc_putc(int c)
{
  return hal_putc(c);
}

int ntlibc_getc(void)
{
  return hal_getc();
}

int ntlibc_puts(const char *s)
{
  char *p = (char *)s;
  while (*p) {
    ntlibc_putc(*p);
    p++;
  }
  return 0;
}

char *ntlibc_gets(char *s, int size)
{
  char *p = s;
  int i;
  for (i = 0; i < size; i++) {
    int c = ntlibc_getc();
    if ((c == '\0') || (c == '\r') || (c == '\n')) {
      break;
    }
    *p = c;
    p++;
  }
  *p = '\0';
  return s;
}

int ntlibc_isdigit(int c)
{
  if (('0' <= c) && (c <= '9')) {
    return 1;
  }
  return 0;
}

int ntlibc_isalpha(int c)
{
  if (('A' <= c) && (c <= 'Z')) {
    return 1;
  }
  if (('a' <= c) && (c <= 'z')) {
    return 1;
  }
  return 0;
}

int ntlibc_iscntrl(int c)
{
  if (c == 0x07) { return 0; }
  if (c == 0x08) { return 0; }
  if (c == 0x09) { return 0; }
  if (c == 0x0a) { return 0; }
  if (c == 0x0b) { return 0; }
  if (c == 0x0c) { return 0; }
  if (c == 0x0d) { return 0; }
  if ((0x00 <= c) && (c <= 0x1f)) {
    return 1;
  }
  return 0;
}

int ntlibc_toupper(int c)
{
  if (('a' <= c) && (c <= 'z')) {
    int diff = 'a' - 'A';
    return c - diff;
  }
  return c;
}

int ntlibc_tolower(int c)
{
  if (('A' <= c) && (c <= 'Z')) {
    int diff = 'a' - 'A';
    return c + diff;
  }
  return c;
}

int ntlibc_atoi(const char *nptr)
{
    const int base = 10;
    const int len = ntlibc_strlen(nptr);
    int work = 0;
    int bnum = 1;
    int i;

    char *p = (char *)&nptr[len - 1];
    for (i = 0; i < len; i++, p--) {
        char c = *p;
        switch (c) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                work += bnum * (c - '0');
                bnum = bnum * base;
                break;
        }
    }

    return work;
}

char *ntlibc_strchr(const char *s, int c)
{
  char *p = (char *)s;
  while (*p) {
    if (*p == c) {
      return p;
    }
    p++;
  }
  return 0;
}

