//
//  AHKCALCsvExporter.m
//  iCal to KS2
//
//  Created by FUJIDANA on 05/05/20.
//  Copyright 2005 FUJIDANA. All rights reserved.
//
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "AHKCALCsvExporter.h"
#import "ICSCalendar.h"
#import "ICSEvent.h"
#import "ICSTodo.h"

@interface AHKCALCsvExporter (Private)

+ (int)indexWithColor:(NSColor *)color;
+ (NSString *)quotedFormWithString:(NSString *)string;
- (NSString *)header;
- (NSString *)csvRepresentationWithCalendar:(ICSCalendar *)calendar;
- (NSString *)csvRepresentationWithEvent:(ICSEvent *)event colorIndex:(int)colorIndex;
- (NSString *)csvRepresentationWithTodo:(ICSTodo *)todo colorIndex:(int)colorIndex;
- (NSArray *)eventsByDividingCrossDayEvents:(NSArray *)events;

@end

@implementation AHKCALCsvExporter

- (id)initWithCalendars:(NSArray *)array
{
	self = [super init];
	if (self != nil) {
		calendars = [array retain];
	}
	return self;
}

- (void) dealloc
{
	[calendars release];
	[self setStartingDateForExport:nil];
	[self setEndingDateForExport:nil];
	[super dealloc];
}

#pragma mark accessor methods
- (NSArray *)calendars
{
	return calendars;
}

- (NSCalendarDate *)startingDateForExport
{
	return startingDateForExport;
}

- (void)setStartingDateForExport:(NSCalendarDate *)value
{
	if (startingDateForExport != value) {
		[startingDateForExport release];
		startingDateForExport = [value copy];
	}
}

- (NSCalendarDate *)endingDateForExport
{
	return endingDateForExport;
}

- (void)setEndingDateForExport:(NSCalendarDate *)value
{
	if (endingDateForExport != value) {
		[endingDateForExport release];
		endingDateForExport = [value copy];
	}
}

- (BOOL)exportsColor
{
	return exportsColor;
}
- (void)setExportsColor:(BOOL)flag
{
	exportsColor = flag;
}

- (BOOL)omitsTimeOfAllDayEvents
{
	return omitsTimeOfAllDayEvents;
}
- (void)setOmitsTimeOfAllDayEvents:(BOOL)flag
{
	omitsTimeOfAllDayEvents = flag;
}

- (BOOL)dividesCrossDayEvents
{
	return dividesCrossDayEvents;
}
- (void)setDividesCrossDayEvents:(BOOL)flag
{
	dividesCrossDayEvents = flag;
}

- (BOOL)omitsInterludeOfLongLastingEvents
{
	return omitsInterludeOfLongLastingEvents;
}
- (void)setOmitsInterludeOfLongLastingEvents:(BOOL)flag
{
	omitsInterludeOfLongLastingEvents = flag;
}

- (int)maxDaysForNotOmittingInterlude
{
	return maxDaysForNotOmittingInterlude;
}
- (void)setMaxDaysForNotOmittingInterlude:(int)value
{
	maxDaysForNotOmittingInterlude = value;
}

- (BOOL)expandsRecurrentEvents
{
	return expandsRecurrentEvents;
}
- (void)setExpandsRecurrentEvents:(BOOL)flag
{
	expandsRecurrentEvents = flag;
}

- (BOOL)exportsTodos
{
	return exportsTodos;
}
- (void)setExportsTodos:(BOOL)flag
{
	exportsTodos = flag;
}

- (BOOL)exportsCompletedTodos
{
	return exportsCompletedTodos;
}
- (void)setExportsCompletedTodos:(BOOL)flag
{
	exportsCompletedTodos = flag;
}

- (BOOL)asignsDueDatesOfTermlessTodos
{
	return asignsDueDatesOfTermlessTodos;
}
- (void)setAsignsDueDatesOfTermlessTodos:(BOOL)flag
{
	asignsDueDatesOfTermlessTodos = flag;
}

- (NSCalendarDate *)dueDateOfTermlessTodos
{
	return dueDateOfTermlessTodos;
}
- (void)setDueDateOfTermlessTodos:(NSCalendarDate *)value
{
	if (dueDateOfTermlessTodos != value) {
		[dueDateOfTermlessTodos release];
		dueDateOfTermlessTodos = [value copy];
	}
}


#pragma mark other mehods

- (NSString *)csvRepresentation
{
	NSMutableString	*csvRepresentation	= [NSMutableString stringWithCapacity:1024];
	[csvRepresentation appendString:[self header]];
	
	NSEnumerator	*enumerator			= [calendars objectEnumerator];
	ICSCalendar		*calendar;
	
	while (calendar = [enumerator nextObject]) {
		NSString *string = [self csvRepresentationWithCalendar:calendar];
		if (string) {
			[csvRepresentation appendString:string];
		}
	}
	
	return csvRepresentation;
}

#pragma mark private methods
+ (int)indexWithColor:(NSColor *)color
{
	int index = 0;
	if (color) {
		float red, green, blue, alpha;
		
		[color getRed:&red 
				green:&green
				 blue:&blue 
				alpha:&alpha];
		
		if (red == (float)0xF5 / 0xFF && 
			green == (float)0x78 / 0xFF && 
			blue == (float)0x02 / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 1; // color representation in iCal is orange.
			
		} else if (red == (float)0xE5 / 0xFF && 
			green == (float)0x17 / 0xFF && 
			blue == (float)0x17 / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 2; // color representation in iCal is red.
			
		} else if (red == (float)0x02 / 0xFF && 
			green == (float)0x52 / 0xFF && 
			blue == (float)0xD4 / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 3; // color representation in iCal is blue.
			
		} else if (red == (float)0x2C / 0xFF && 
			green == (float)0xA1 / 0xFF && 
			blue == (float)0x0B / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 4; // color representation in iCal is green.
			
		} else if (red == (float)0x49 / 0xFF && 
			green == (float)0x2B / 0xFF && 
			blue == (float)0xA1 / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 5; // color representation in iCal is purple.
			
		} else if (red == (float)0xB0 / 0xFF && 
			green == (float)0x27 / 0xFF && 
			blue == (float)0xAE / 0xFF && 
			alpha == (float)0xFF / 0xFF) {
			
			index = 6; // color representation in iCal is teal.
		}
	}
	return index;
}

+ (NSString *)quotedFormWithString:(NSString *)string
{
	if (string == nil || [string isEqualToString:@""]) {
		return @"";
	}
	
	NSArray	*array		= [string componentsSeparatedByString:@"\""];
	NSString *newString	= [array componentsJoinedByString:@"\"\""];
	return [NSString stringWithFormat:@"\"%@\"", newString];
}


- (NSString *)header
{
	NSMutableString	*string = [NSMutableString stringWithCapacity:32];
	
	[string appendString:NSLocalizedString(@"default header", nil)];
	
	if ([self exportsColor]) {
		[string appendString:NSLocalizedString(@"color", nil)];
	}
	
	if ([self exportsTodos] ) {
		[string appendString:NSLocalizedString(@"priority", nil)];
	}
	
	[string appendString:@"\r\n"];
	return string;
}

- (NSString *)csvRepresentationWithCalendar:(ICSCalendar *)calendar;
{
	int				colorIndex		= [AHKCALCsvExporter indexWithColor:[calendar color]];
	
	NSMutableString	*resultString	= [NSMutableString stringWithCapacity:1024];
	NSArray			*events			= [calendar events];
	
	NSEnumerator	*eventsEnumerator;
	ICSEvent		*currentEvent;
	
	// -- expand recurrent events --
	if ([self expandsRecurrentEvents]) {
		eventsEnumerator = [events objectEnumerator];
		NSMutableArray *tempEvents = [NSMutableArray arrayWithCapacity:[events count]];
		
		while (currentEvent = [eventsEnumerator nextObject]) {
			NSArray *eventsByExpandingCurrentEvent = [currentEvent eventsByExpandingRecurrenceUntilDate:[self endingDateForExport]];
			if (eventsByExpandingCurrentEvent) {
				[tempEvents addObjectsFromArray:eventsByExpandingCurrentEvent];
			}
		}
		events = tempEvents;
	}
	
	// -- remove events whose date is out of range --
	eventsEnumerator = [events objectEnumerator];
	NSMutableArray *tempArray = [NSMutableArray arrayWithCapacity:[events count]];
	
	while (currentEvent = [eventsEnumerator nextObject]) {
		if (([self endingDateForExport] != nil && [[currentEvent startDate] timeIntervalSinceDate:[self endingDateForExport]] > 0) ||
			([self startingDateForExport] != nil && [[currentEvent endDate] timeIntervalSinceDate:[self startingDateForExport]] < 0)) {
			continue;
		}
		[tempArray addObject:currentEvent];
	}
	events = tempArray;
	
	// -- divide events which cross over multiple days --
	if ([self dividesCrossDayEvents]) {
		events = [self eventsByDividingCrossDayEvents:events];
	}
	
	// -- export events in CSV format --
	eventsEnumerator = [events objectEnumerator];
	while (currentEvent = [eventsEnumerator nextObject]) {
		NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
		NSString *tempString = [self csvRepresentationWithEvent:currentEvent colorIndex:colorIndex];
		if (tempString) {
			[resultString appendString:tempString];
		}
		[pool release];
	}
	
	// -- export todos in CSV format --
	if ([self exportsTodos]) {
		NSEnumerator	*todosEnumerator	= [[calendar todos] objectEnumerator];
		ICSTodo			*todo;
		while (todo = [todosEnumerator nextObject]) {
			if ([todo completionDate] != nil && [self exportsCompletedTodos] == NO) {
				continue;
			}
			
			if ([self asignsDueDatesOfTermlessTodos] && [todo dueDate] == nil) {
				[todo setDueDate:[self dueDateOfTermlessTodos]];
			}
			
			NSString *tempString = [self csvRepresentationWithTodo:todo colorIndex:colorIndex];
			if (tempString) {
				[resultString appendString:tempString];
			}
		}
	}
	
	return resultString;
	
}

- (NSString *)csvRepresentationWithEvent:(ICSEvent *)event colorIndex:(int)colorIndex
{
	if (event == nil) {
		return nil;
	}
	
	NSMutableString *string			= [NSMutableString stringWithCapacity:64];
	NSString		*emptyString	= @"";
	NSString *summary, *startDateString, *startTimeString, *endDateString, *endTimeString, *location, *eventNotes;
	
	if ((summary = [event summary]) == nil) {
		summary = emptyString;
	}
	
	if ([event startDate] == nil) {
		startDateString = emptyString;
		startTimeString = emptyString;
	} else {
		startDateString = [[event startDate] descriptionWithCalendarFormat:@"%Y/%m/%d"];
		if ([event isAllDayEvent] && [self omitsTimeOfAllDayEvents]) {
			startTimeString = emptyString;
		} else {
			startTimeString = [[event startDate] descriptionWithCalendarFormat:@"%H:%M:%S"];
		}
	}
	
	if ([event endDate] == nil) {
		endDateString = emptyString;
		endTimeString = emptyString;
	} else {
		endDateString = [[event endDate] descriptionWithCalendarFormat:@"%Y/%m/%d"];
		if ([event isAllDayEvent] && [self omitsTimeOfAllDayEvents]) {
			endDateString = [[[event endDate] dateByAddingYears:0
														 months:0
														   days:-1
														  hours:0
														minutes:0
														seconds:0] descriptionWithCalendarFormat:@"%Y/%m/%d"];
			endTimeString = emptyString;
			
		} else {
			endTimeString = [[event endDate] descriptionWithCalendarFormat:@"%H:%M:%S"];
		}
	}
	
	if ((location = [event location]) == nil) {
		location = emptyString;
	}
	
	if ((eventNotes = [event notes]) == nil) {
		eventNotes = emptyString;
	}
	
	[string appendFormat:@"%@,%@,%@,%@,%@,%@,%@",
		[AHKCALCsvExporter quotedFormWithString:summary], 
		startDateString, 
		startTimeString, 
		endDateString, 
		endTimeString, 
		[AHKCALCsvExporter quotedFormWithString:location], 
		[AHKCALCsvExporter quotedFormWithString:eventNotes]];
	
	if ([self exportsColor]) {
		[string appendFormat:@",%d", colorIndex];
	}
	if ([self exportsTodos]) {
		[string appendFormat:@",%d", 0];
	}
	[string appendString:@"\r\n"];
	
	return string;
}

- (NSString *)csvRepresentationWithTodo:(ICSTodo *)todo colorIndex:(int)colorIndex
{
	if (todo == nil) {
		return nil;
	}
	
	NSMutableString *string			= [NSMutableString stringWithCapacity:64];
	NSString		*emptyString	= @"";
	NSString *summary, *startDateString, *startTimeString, *endDateString, *endTimeString, *location, *eventNotes;
	
	if ((summary = [todo summary]) == nil) {
		summary = emptyString;
	}
	
	if ([todo dueDate] == nil) {
		startDateString = emptyString;
		startTimeString = emptyString;
	} else {
		startDateString = [[todo dueDate] descriptionWithCalendarFormat:@"%Y/%m/%d"];
		if ([self omitsTimeOfAllDayEvents] == NO) {
			startTimeString = [[todo dueDate] descriptionWithCalendarFormat:@"%H:%M:%S"];
		} else {
			startTimeString = emptyString;
		}
	}
	
	endDateString = emptyString;
	endTimeString = emptyString;
	location = emptyString;
	
	if ((eventNotes = [todo notes]) == nil) {
		eventNotes = emptyString;
	}
	
	[string appendFormat:@"%@,%@,%@,%@,%@,%@,%@",
		[AHKCALCsvExporter quotedFormWithString:summary], 
		startDateString, 
		startTimeString, 
		endDateString, 
		endTimeString, 
		[AHKCALCsvExporter quotedFormWithString:location], 
		[AHKCALCsvExporter quotedFormWithString:eventNotes]];
	
	if ([self exportsColor]) {
		[string appendFormat:@",%d", colorIndex];
	}
	
	int priority;
	if ([todo priority] == AHKHighPriority) {
		priority = 1;
	} else if ([todo priority] == AHKMediumPriority) {
		priority = 2;
	}  else {
		priority = 3;
	}
	
	[string appendFormat:@",%d", priority];
	[string appendString:@"\r\n"];
	
	return string;
}

- (NSArray *)eventsByDividingCrossDayEvents:(NSArray *)events
{
	NSMutableArray		*resultArray	= [NSMutableArray arrayWithCapacity:[events count]];
	
	NSEnumerator		*enumerator		= [events objectEnumerator];
	ICSEvent			*event;
	
	while (event = [enumerator nextObject]) {
		NSAutoreleasePool	*pool		= [[NSAutoreleasePool alloc] init];
		NSCalendarDate		*startDate	= [event startDate];
		NSCalendarDate		*endDate	= [event endDate];
		
		if ([event isAllDayEvent]) {
			endDate = [[event endDate] dateByAddingYears:0
												  months:0
													days:-1
												   hours:0
												 minutes:0
												 seconds:0];
		}
		
		NSCalendarDate	*startDateWithoutTime	= [startDate dateByAddingYears:0
																		months:0
																		  days:0
																		 hours:- [startDate hourOfDay]
																	   minutes:- [startDate minuteOfHour]
																	   seconds:- [startDate secondOfMinute]];
		NSCalendarDate	*endDateWithoutTime		= [endDate dateByAddingYears:0
																	  months:0
																		days:0
																	   hours:- [endDate hourOfDay]
																	 minutes:- [endDate minuteOfHour]
																	 seconds:- [endDate secondOfMinute]];
		int span;
		[endDateWithoutTime years:NULL
						   months:NULL
							 days:&span
							hours:NULL
						  minutes:NULL
						  seconds:NULL
						sinceDate:startDateWithoutTime];
		
		if (span == 0) {
			// -- single day event --
			[resultArray addObject:event];
		
		} else if (span >= [self maxDaysForNotOmittingInterlude] && [self omitsInterludeOfLongLastingEvents]) {
			// -- multiple days event whose span is so long that whose interludes are omitted --
			// -- creates an event at the beginning date --
			ICSEvent *tempEvent = [event copy];
			[tempEvent setSummary:[NSString stringWithFormat:@"%@ [to %d.%d.%d]", [tempEvent summary], [endDate yearOfCommonEra], [endDate monthOfYear], [endDate dayOfMonth]]];
			if ([tempEvent isAllDayEvent]) {
				[tempEvent setDuration:24 * 60 * 60];
			} else {
				int hours	= [[tempEvent startDate] hourOfDay];
				int minutes	= [[tempEvent startDate] minuteOfHour] + hours * 60;
				int seconds	= [[tempEvent startDate] secondOfMinute] + minutes * 60;
				[tempEvent setDuration:24 * 60 * 60 - seconds - 1];
			}
			[resultArray addObject:tempEvent];
			[tempEvent release];
			
			// -- creates an event at the end date --
			tempEvent = [event copy];
			[tempEvent setSummary:[NSString stringWithFormat:@"%@ [from %d.%d.%d]", [tempEvent summary], [startDate yearOfCommonEra], [startDate monthOfYear], [startDate dayOfMonth]]];
			[tempEvent setStartDate:endDateWithoutTime];
			if ([tempEvent isAllDayEvent]) {
				[tempEvent setDuration:24 * 60 * 60];
			} else {
				int hours	= [[event endDate] hourOfDay];
				int minutes	= [[event endDate] minuteOfHour] + hours * 60;
				int seconds	= [[event endDate] secondOfMinute] + minutes * 60;
				[tempEvent setDuration:seconds];
			}
			[resultArray addObject:tempEvent];
			[tempEvent release];
		} else {
			// -- multiple days event whose span is not long enough that whose interludes are not omitted --
			int i;
			for (i = 0; i < (span + 1); i++) {
				ICSEvent *tempEvent = [event copy];
				[tempEvent setSummary:[NSString stringWithFormat:@"%@ [%d/%d]", [tempEvent summary], (i + 1), (span + 1)]];
				
				if ([tempEvent isAllDayEvent]) {
					if (i == 0) {
						[tempEvent setDuration:24 * 60 * 60];
					} else {
						[tempEvent setStartDate:[startDateWithoutTime dateByAddingYears:0
																				 months:0
																				   days:i
																				  hours:0
																				minutes:0
																				seconds:0]];
						[tempEvent setDuration:24 * 60 * 60];
					}
				} else {
					if (i == 0) {
						int startHourOfDay		= [startDate hourOfDay];
						int startMinuteOfDay	= [startDate minuteOfHour] + startHourOfDay * 60;
						int startSecondOfDay	= [startDate secondOfMinute] + startMinuteOfDay * 60;
						
						[tempEvent setDuration:24 * 60 * 60 - startSecondOfDay - 1];
					} else if (i == span) {
						int endHourOfDay	= [endDate hourOfDay];
						int endMinuteOfDay	= [endDate minuteOfHour] + endHourOfDay * 60;
						int endSecondOfDay	= [endDate secondOfMinute] + endMinuteOfDay * 60;
						
						[tempEvent setStartDate:[startDateWithoutTime dateByAddingYears:0
																				 months:0
																				   days:span
																				  hours:0
																				minutes:0
																				seconds:0]];
						[tempEvent setDuration:endSecondOfDay];
					} else {
						[tempEvent setStartDate:[startDateWithoutTime dateByAddingYears:0
																				 months:0
																				   days:i
																				  hours:0
																				minutes:0
																				seconds:0]];
						[tempEvent setDuration:(24 * 60 * 60 - 1)];
					}
				}
				[resultArray addObject:tempEvent];
				[tempEvent release];
			}
		}
		[pool release];
	}
	return resultArray;
}

@end
