//
//  FJNCoreDataArrayController.m
//  FJNCoreData
//
//  Created by FUJIDANA on 06/08/17.
//  Copyright 2006-2008 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "FJNCoreDataArrayController.h"


NSString *FJNObjectIDsPboardType = @"FJNObjectIDsPboardType";


@interface FJNCoreDataArrayController ()

- (BOOL)isSortedUsingDefaultDescriptors;
- (void)reorderByMovingObjects:(NSArray *)movedObjects toArrangedObjectIndex:(unsigned)index;

@end


@implementation FJNCoreDataArrayController

#pragma mark Methods overriding super class

- (void)awakeFromNib
{
	if ([NSArrayController instancesRespondToSelector:@selector(awakeFromNib)])
	{
		[super awakeFromNib];
	}
	
	[mainTableView setDelegate:self];
	[mainTableView setDataSource:self];
	
	[self setSortDescriptors:[self defaultSortDescriptors]];
	[self setClearsFilterPredicateOnInsertion:YES];
	
	[mainTableView registerForDraggedTypes:[self readablePasteboardTypes]];
	[mainTableView setDraggingSourceOperationMask:NSDragOperationCopy forLocal:YES];
}

- (void)dealloc
{
	[_defaultSortDescriptors release], _defaultSortDescriptors = nil;
	[self setLocalizedEntityName:nil];
	
	[super dealloc];
}

#pragma mark Accessor methods

- (NSString *)sortKey
{
	return @"order";
}

- (NSTableView *)tableView
{
	return mainTableView;
}

- (NSArray *)defaultSortDescriptors
{
	if (_defaultSortDescriptors == nil)
	{
		NSSortDescriptor *sortDescriptor = [[NSSortDescriptor alloc] initWithKey:[self sortKey] ascending:YES];
		_defaultSortDescriptors = [[NSArray alloc] initWithObjects:sortDescriptor, nil];
		[sortDescriptor release];
	}
	return _defaultSortDescriptors;
}

- (NSString *)localizedEntityName
{
	return (_localizedEntityName) ? _localizedEntityName : [self entityName];
}

- (void)setLocalizedEntityName:(NSString *)value
{
	if (_localizedEntityName != value)
	{
		[_localizedEntityName release];
		_localizedEntityName = [value copy];
	}
}

- (NSArray *)readablePasteboardTypes
{
	return [NSArray arrayWithObject:FJNObjectIDsPboardType];
}

- (BOOL)supportsDragAndDrop
{
	return _supportsDragAndDrop;
}

- (void)setSupportsDragAndDrop:(BOOL)value
{
	_supportsDragAndDrop = value;
}

/* Different from ``content'' method defined in NSObjectController in the 
 point that this method retuns all objects of entity the array controller manages. 
 (``content'' method returns only objects specified by contentArray, 
 or contentSet binding, if it is set.) */
- (NSArray *)allObjects
{
	NSManagedObjectContext *context      = [self managedObjectContext];
	NSFetchRequest         *fetchRequest = [[NSFetchRequest alloc] init];
	NSEntityDescription    *entity       = [NSEntityDescription entityForName:[self entityName]
													   inManagedObjectContext:context];
	[fetchRequest setEntity:entity];
	[fetchRequest setSortDescriptors:[self defaultSortDescriptors]];
	
	NSArray *result = [context executeFetchRequest:fetchRequest error:NULL];
	[fetchRequest release];
	return result;
}

#pragma mark Actions

- (void)selectNext:(id)sender
{
	[super selectNext:sender];
//	[mainTableView scrollRowToVisible:[self selectionIndex]];
}

- (void)selectPrevious:(id)sender
{
	[super selectPrevious:sender];
//	[mainTableView scrollRowToVisible:[self selectionIndex]];
}

- (IBAction)deselectSort:(id)sender
{
	[self setSortDescriptors:[self defaultSortDescriptors]];
}

#pragma mark Methods overriding NSArrayController

- (void)insertObject:(id)object atArrangedObjectIndex:(unsigned int)index
{
	[super insertObject:object atArrangedObjectIndex:index];
	
	[self reorderByMovingObjects:[NSArray arrayWithObject:object] toArrangedObjectIndex:index];
}

- (void)insertObjects:(NSArray *)objects atArrangedObjectIndexes:(NSIndexSet *)indexes
{
	[super insertObjects:objects atArrangedObjectIndexes:indexes];
	
	[self reorderByMovingObjects:objects toArrangedObjectIndex:[indexes firstIndex]];
}

- (BOOL)validateUserInterfaceItem:(id <NSValidatedUserInterfaceItem>)item
{
#ifdef NSAppKitVersionNumber10_4
	if ([item action] == @selector(deselectSort:)) return ![self isSortedUsingDefaultDescriptors];
	return [super validateUserInterfaceItem:item];
#else
	if ([item action] == @selector(add:)) return [self canAdd];
	else if ([item action] == @selector(insert:)) return [self canInsert];
	else if ([item action] == @selector(remove:)) return [self canRemove];
	else if ([item action] == @selector(selectNext:)) return [self canSelectNext];
	else if ([item action] == @selector(selectPrevious:)) return [self canSelectPrevious];
	else if ([item action] == @selector(deselectSort:)) return ![self isSortedUsingDefaultDescriptors];
	return YES;
#endif
}

#pragma mark Private Methods

- (BOOL)isSortedUsingDefaultDescriptors
{
	return [[self defaultSortDescriptors] isEqualToArray:[self sortDescriptors]];
}

// Auxiliary methods that helps drag and drop
- (void)reorderByMovingObjects:(NSArray *)movedObjects toArrangedObjectIndex:(unsigned)index
{
	if ([self isSortedUsingDefaultDescriptors])
	{
		NSArray *arrangedObjects = [self arrangedObjects];
		
		// Get order value at insertion point.
		long order = 0;
		if (index > 0)
		{
			NSManagedObject *objectJustAboveInsertion = [arrangedObjects objectAtIndex:index - 1];
			order = [[objectJustAboveInsertion valueForKey:[self sortKey]] longValue] + 1;
		}
		
		NSEnumerator *enumerator = [movedObjects objectEnumerator];
		id object;
		
		// set the sort index of movedObjects incrementally.
		while (object = [enumerator nextObject])
		{
			[object setValue:[NSNumber numberWithLong:order++] forKey:[self sortKey]];
		}
		
		// set the sort index properties of objects below the index.
		unsigned int i;
		for (i = index; i < [arrangedObjects count]; i++)
		{
			NSManagedObject *object = [arrangedObjects objectAtIndex:i];
			
			// if object is a member of movedObjects, do nothing (since sort index is already set.)
			if ([movedObjects containsObject:object])
			{
				continue;
			}
			
			// else, set sort index property incrementally.
			[object setValue:[NSNumber numberWithLong:order++] forKey:[self sortKey]];
		}
	}
	else
	{
		// If controller is not sorted in the correct order, simply add the inserted item to last of the array.
		NSArray *sortedArray = [[self content] sortedArrayUsingDescriptors:[self defaultSortDescriptors]];
		
		long order = 0;
		if ([sortedArray count] > 0)
		{
			order = [[[sortedArray lastObject] valueForKey:[self sortKey]] longValue] + 1;
		}
		
		NSEnumerator *enumerator = [movedObjects objectEnumerator];
		NSManagedObject *object;
		while (object = [enumerator nextObject])
		{
			[object setValue:[NSNumber numberWithLong:order++] forKey:[self sortKey]];
		}
	}
}

#pragma mark Methods implementing NSTableDataSource informal protocol

- (BOOL)tableView:(NSTableView *)aTableView writeRowsWithIndexes:(NSIndexSet *)rowIndexes toPasteboard:(NSPasteboard *)pboard
{
	if (aTableView == mainTableView && [self supportsDragAndDrop])
	{
		// Write URL representations of selected object to pasteboard.
		// Declare types
		[pboard declareTypes:[NSArray arrayWithObject:FJNObjectIDsPboardType] owner:nil];
		
		// Get dragged objects and convert each object to URIRepresentation (NSString)
		NSArray *objects = [[self arrangedObjects] objectsAtIndexes:rowIndexes];
		NSArray *URIReps = [self URIRepresentationsOfManagedObjects:objects];
		if (URIReps == nil || [URIReps count] == 0)
		{
			return NO;
		}
		// Set URI representations
		[pboard setPropertyList:URIReps forType:FJNObjectIDsPboardType];
		
		return YES;
	}
	return NO;
}

- (NSDragOperation)tableView:(NSTableView *)aTableView validateDrop:(id <NSDraggingInfo>)info proposedRow:(int)row proposedDropOperation:(NSTableViewDropOperation)operation
{
	if (operation == NSTableViewDropAbove &&
		aTableView == mainTableView &&
		[info draggingSource] == mainTableView &&
		[self filterPredicate] == nil &&
		[[[info draggingPasteboard] types] containsObject:FJNObjectIDsPboardType] &&
		[self isSortedUsingDefaultDescriptors])
	{
		// Dragged between itemsTableView... (This means dragged items is being reordered.)
		return NSDragOperationMove;
	}
	return NSDragOperationNone;
}

- (BOOL)tableView:(NSTableView *)aTableView acceptDrop:(id <NSDraggingInfo>)info row:(int)row dropOperation:(NSTableViewDropOperation)operation
{
	if (operation == NSTableViewDropAbove &&
		aTableView == mainTableView &&
		[info draggingSource] == mainTableView &&
		[self filterPredicate] == nil &&
		[[[info draggingPasteboard] types] containsObject:FJNObjectIDsPboardType] &&
		[self isSortedUsingDefaultDescriptors])
	{
		// Dragged between itemsTableView... (This means dragged items is being reordered.)
		NSArray *URIReps = [[info draggingPasteboard] propertyListForType:FJNObjectIDsPboardType];
		if (URIReps == nil || [URIReps isKindOfClass:[NSArray class]] == NO || [URIReps count] == 0) return NO;
		NSArray *objects = [self managedObjectsOfURIRepresentations:URIReps];
		
		[self reorderByMovingObjects:objects toArrangedObjectIndex:row];
		
		[self rearrangeItemsWithUndoRegistration];
		
		return YES;
	}
	return NO;
}

#pragma mark Methods to support drag and drop operation.

// --- get NSArray of URI Representations from NSArray of managed objects. ---
- (NSArray *)URIRepresentationsOfManagedObjects:(NSArray *)objects
{
	NSMutableArray  *URIReps    = [NSMutableArray arrayWithCapacity:[objects count]];
	NSEnumerator    *enumerator = [objects objectEnumerator];
	NSManagedObject *object;
	
	while (object = [enumerator nextObject])
	{
		// Get URI representations coresponding to current object (NSManagedObject).
		NSString* URIRep = [[[object objectID] URIRepresentation] absoluteString];
		
		if (URIRep == nil) continue;
		
		// Add URI representation
		[URIReps addObject:URIRep];
	}
	return URIReps;
}

// --- get NSArray of managed objects from NSArray of URI Representations. ---
- (NSArray *)managedObjectsOfURIRepresentations:(NSArray *)URIReps
{
	NSManagedObjectContext       *context     = [self managedObjectContext];
	NSPersistentStoreCoordinator *coordinator = [context persistentStoreCoordinator];
	
	NSMutableArray               *objects     = [NSMutableArray arrayWithCapacity:[URIReps count]];
	NSEnumerator                 *enumerator  = [URIReps objectEnumerator];
	NSString                     *URIRep;
	NSManagedObjectID            *objectID;
	NSManagedObject              *object;
	
	while (URIRep = [enumerator nextObject])
	{
		// get an object ID coresponding to current object (NSString representing URI Representaion).
		objectID = [coordinator managedObjectIDForURIRepresentation:[NSURL URLWithString:URIRep]];
		if (objectID == nil) continue;
		
		// get an object corespponding to the object ID obtained above.
		object = [context objectWithID:objectID];
		if (object == nil) continue;
		
		// add object into array.
		[objects addObject:object]; 
	}
	return objects;
}

/* send self to rearrangeObjects while registering undo manager.
   invoked when user changed ``order'' property of items by drag and drop.
*/
- (void)rearrangeItemsWithUndoRegistration
{
	NSUndoManager *undoManager = [[self managedObjectContext] undoManager];
	[[undoManager prepareWithInvocationTarget:self] rearrangeItemsWithUndoRegistration];
	[self rearrangeObjects];
}

@end
