/*
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package cron;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import com.l2jserver.gameserver.ThreadPoolManager;

/**
 * @author JOJO
 * 
 * How to use:
 *   Add "cron/ServerRestartTask.java" into scripts.cfg
 * 
 * Option:
 * 
 *   Write BACKUP.BAT yourself. using mysqldump etc...
 *   And patch startGameServer.bat
-------------------------------------   
  if ERRORLEVEL 2 goto restart
  if ERRORLEVEL 1 goto error
  goto end
  :restart
  echo.
  echo Admin Restart ...
  echo.
+ IF EXIST __backup__ CALL BACKUP.BAT
  goto start
-------------------------------------   
 * 
 */
public class ServerRestartTask
{
	protected static final String TASK_NAME = ServerRestartTask.class.getSimpleName();
	private static final int SECONDS_SHUT = 180;
	private static final long MINIMUM_DELAY = 300000;	// safety guard time.
	
	private static final String BACKUP_FLAG = "__backup__";
	
	public ServerRestartTask()
	{
		long[] timeTable = new long[]{
				calcDelay(MINIMUM_DELAY, Calendar.SUNDAY, 12, 0),
				calcDelay(MINIMUM_DELAY, Calendar.WEDNESDAY, 23, 30),
				calcDelay(MINIMUM_DELAY, Calendar.FRIDAY, 9, 0),
		};
		Arrays.sort(timeTable);
		final long delay = timeTable[0] - System.currentTimeMillis();
		
		new File(BACKUP_FLAG).delete();	// remove flag file.
		
		ThreadPoolManager.getInstance().scheduleGeneral(new Runnable()
		{
			@Override
			public void run()
			{
			//	com.l2jserver.gameserver.Announcements.getInstance().announceToAll("*****");
				com.l2jserver.gameserver.Shutdown.getInstance().startTelnetShutdown(TASK_NAME, SECONDS_SHUT, true);
				try
				{
					new File(BACKUP_FLAG).createNewFile();	// create flag file.
				}
				catch (IOException e)
				{
					e.printStackTrace();
				}
			}
		}, delay);
		
		System.out.println(TASK_NAME + ": Scheduled " + new Date(timeTable[0]).toString());
	}
	
	private long calcDelay(long minimumDelay, int dayOfWeek, int hourOfDay, int miniute)
	{
		long now = System.currentTimeMillis();
		GregorianCalendar reset = new GregorianCalendar();
		reset.setTimeInMillis(now);
		reset.set(Calendar.MILLISECOND, 0);
		reset.set(Calendar.SECOND, 0);
		reset.set(Calendar.MINUTE, miniute);
		reset.set(Calendar.HOUR_OF_DAY, hourOfDay);
		while (reset.getTimeInMillis() - now < minimumDelay)
			reset.add(Calendar.DATE, 1);
		while (reset.get(Calendar.DAY_OF_WEEK) != dayOfWeek)
			reset.add(Calendar.DATE, 1);
		return reset.getTimeInMillis();
	}
	
	public static void main(String args[])
	{
		new ServerRestartTask();
	}
}
