/*
 * Colapi - Code Colorer for Java API 
 * Public Domain (latest version at http://javolution.org/colapi.jar)
 */
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.HashSet;

public class Colapi {

    public static final String VERSION = "1.2";

    private static final String USAGE = "\n"
            + "Code Colorer for Java API - Version " + VERSION + "\n"
            + "Tool to colorize/format code between [code]...[/code]\n" + "\n"
            + "Usage:" + "    java -jar file.html [properties]\n"
            + "               (color one html file)\n"
            + "            or java -jar directory [properties]\n"
            + "               (color all html files in directory)\n" + "\n"
            + "where properties include:\n"
            + "   -Dencoding=<value> File encoding (default \"UTF-8\")\n"
            + "   -DkeywordColor=<value> Keyword color (default \"#7F0055\")\n"
            + "   -DcommentColor=<value> Comment color (default \"#3F7F5F\")\n"
            + "   -DstringColor=<value> String color (default \"#0000A0\")\n"
            + "\n";

    private static final String ENCODING = System.getProperty("encoding",
            "UTF-8");

    private static final String KEYWORD_COLOR = System.getProperty(
            "keywordColor", "#7F0055");

    private static final String COMMENT_COLOR = System.getProperty(
            "commentColor", "#3F7F5F");

    private static final String STRING_COLOR = System.getProperty(
            "stringColor", "#0000A0");

    private static int Processed = 0;

    private static int Modified = 0;

    public static void main(String[] args) throws Exception {
        if (args.length == 0) {
            System.out.println(USAGE);
            return;
        }
        File file = new File(args[0]);
        if (!file.exists()) {
            System.err.println(args[0] + " is not a file or directory.");
            return;
        }
        if (file.isDirectory()) {
            processDirectory(file);
        } else {
            processFile(file);
        }
        System.out.println("Colapi processed " + Colapi.Processed
                + " files and modified " + Colapi.Modified);
    }

    private static void processDirectory(File dir) throws Exception {
        File[] files = dir.listFiles(new FileFilter() {
            public boolean accept(File pathname) {
                return pathname.isFile()
                        && pathname.getName().endsWith(".html");
            }
        });
        for (int i = 0; i < files.length; i++) {
            processFile(files[i]);
        }
        File[] dirs = dir.listFiles(new FileFilter() {
            public boolean accept(File pathname) {
                return pathname.isDirectory();
            }
        });
        for (int i = 0; i < dirs.length; i++) {
            processDirectory(dirs[i]);
        }
    }

    private static void processFile(File file) throws Exception {
        Colapi.Processed++;
        BufferedReader in = new BufferedReader(new InputStreamReader(
                new FileInputStream(file), ENCODING));
        StringBuffer doc = new StringBuffer(10000);
        int start = -1;
        int state = DATA;
        boolean modified = false;
        for (int read = in.read(); read != -1; read = in.read()) {

            // Escape substitution.
            if (state != DATA) {
                if (read == '<') {
                    doc.append("&lt;");
                } else if (read == '>') {
                    doc.append("&gt;");
                } else if (read == '&') {
                    doc.append("&amp;");
                } else {
                    doc.append((char) read);
                }
            } else {
                doc.append((char) read);
            }
            switch (state) {
                case DATA:
                    if ((read == ']')
                            && (doc.length() > 5)
                            && doc.substring(doc.length() - 6)
                                    .equalsIgnoreCase("[code]")) {
                        doc.setLength(doc.length() - 6);
                        doc.append("<code><pre>");
                        modified = true;
                        state = CODE;
                    }
                    break;

                case CODE:
                    if (Character.isJavaIdentifierPart((char)read)) {
                        state = IDENTIFIER;
                        start = doc.length() - 1;
                    } else if (read == '"') {
                        state = STRING_LITERAL;
                        doc.insert(doc.length() - 1, "<font color=\""
                                + STRING_COLOR + "\">");
                    } else if ((read == '/')
                            && (doc.charAt(doc.length() - 2) == '/')) {
                        state = COMMENT;
                        doc.insert(doc.length() - 2, "<font color=\""
                                + COMMENT_COLOR + "\">");
                    }
                    break;

                case STRING_LITERAL:
                    if ((read == '"') && (doc.charAt(doc.length() - 2) != '\\')) {
                        doc.append("</font>");
                        state = CODE;
                    }
                    break;

                case IDENTIFIER:
                    if ((read == ']')
                            && // code identifier.
                            doc.substring(doc.length() - 7).equalsIgnoreCase(
                                    "[/code]")) {
                        doc.setLength(doc.length() - 7);
                        doc.append("</pre></code>");
                        state = DATA;
                    } else if (!Character.isJavaIdentifierPart((char)read)) { // End of identifier.
                        String name = doc.substring(start, doc.length() - 1);
                        if (IDENTIFIERS.contains(name)) { // Identifier found.
                            doc.insert(start + name.length(), "</b></font>");
                            doc.insert(start, "<font color=\"" + KEYWORD_COLOR
                                    + "\"><b>");
                        }
                        state = CODE;
                    }
                    break;

                case COMMENT:
                    if ((read == '\n') || (read == '\r')) {
                        doc.insert(doc.length() - 1, "</font>");
                        state = CODE;
                    }
                    break;
            }
        }
        in.close();

        if (modified) {
            Colapi.Modified++;
            OutputStreamWriter out = new OutputStreamWriter(
                    new FileOutputStream(file), ENCODING);
            out.write(doc.toString());
            out.close();
        }
    }

    private static final int DATA = 0;

    private static final int CODE = 1;

    private static final int IDENTIFIER = 2;

    private static final int COMMENT = 3; // Can only be end of line comments.

    private static final int STRING_LITERAL = 4;

    private static final String[] KEYWORDS = { "abstract", "continue", "for",
            "new", "switch", "assert", "default", "if", "package",
            "synchronized", "boolean", "do", "goto", "private", "this",
            "break", "double", "implements", "protected", "throw", "byte",
            "else", "import", "public", "throws", "case", "enum", "instanceof",
            "return", "transient", "catch", "extends", "int", "short", "try",
            "char", "final", "interface", "static", "void", "class", "finally",
            "long", "strictfp", "volatile", "const", "float", "native",
            "super", "while" };

    private static final HashSet IDENTIFIERS = new HashSet();
    static {
        for (int i = 0; i < KEYWORDS.length; i++) {
            IDENTIFIERS.add(KEYWORDS[i]);
        }
    }
}