<?php

    //=================================================================================================================================================
    /**
     * メール送信クラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2010 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2010/2/26
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class EMail{
        //ヘッダ
        private $_toAddr        = '';
        private $_subject       = '';
        private $_fromAddr      = '';
        //追加ヘッダ
        private $_ccAddr        = '';
        private $_bccAddr       = '';

        //リターンパス
        private $_returnPath    = '';

        //使用MTA
        private $_mta           = '';

        //エンコード関連
        private $_encoding      = array();

        //子クラスでセットするヘッダ配列
        private $_mailText      = '';       //メール本文文字列
        private $_mailHeader    = array();
        private $_attachments   = array();  //添付ファイルリスト(ファイル名 => ファイルパス)

        //--------------------------------------------------------------------------------------------------------------------------------
        /**    
         * コンストラクタ/デストラクタ
         * 
         * @param string    $mailHeader     メールヘッダ
         * @param string    $toAddr     宛先
         * @param string    $fromAddr   送信元
         * @param string    $subject    タイトル
         * @param string    $ccAddr     CC:
         * @param string    $bccAddr    BC:
         * @param string    $mta        使用するMTAの種類：'postfix' / (その他)
         *                              postfix では改行コードが、'\n'だが通常改行コードは'\r\n'なので2行改行されてしまうのでそのための処理
         *                              postfixでは1行が989バイトより長い行があると勝手に改行を入れてしまう
         */
        //--------------------------------------------------------------------------------------------------------------------------------
        function __construct($toAddr = '', $fromAddr = '', $subject = '', $ccAddr = '',$bccAddr = '', $returnPath = '', $mta = 'postfix'){
            $this->_toAddr      = $toAddr;
            $this->_fromAddr    = $fromAddr;
            $this->_subject     = $subject;
            $this->_ccAddr      = $ccAddr;
            $this->_bccAddr     = $bccAddr;
            $this->_returnPath  = $returnPath;
            $this->_mta         = strtolower($mta);

            //デフォルトのエンコード設定
            $this->setEncoding(array('TO'=>'JIS', 'FROM'=>'UTF-8'));
        }

        //-----------------------------------------------------------------------------------------------------------------------------------
        /**
         * エンコード設定
         * mb_send_mail()→mail()を使用するように変更したことにより
         * 明示的にエンコード指定する必要があるため追加（デフォルトは SJIS-WIN）
         * 
         * @param string    $encoding   'TO'=>変換先・'FROM'=>変換元キャラクタセットを格納した連想配列
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        function setEncoding($encoding){
            $this->_encoding    = $encoding;
            switch(strtoupper($encoding['TO'])){
                case 'JIS':
                    $this->_headerEncoding = 'iso-2022-jp';
                    $this->_headerBit      = '7';
                    break;
                case 'UTF-8':
                    $this->_headerEncoding = 'UTF-8';
                    $this->_headerBit      = '8';
                    break;
                default:
                    $this->_headerEncoding = 'iso-2022-jp';
                    $this->_headerBit      = '7';
            }
        }

        //-----------------------------------------------------------------------------------------------------------------------------------
        /** 
         * テンプレートよりメール送信
         * 
         * @param string    $path       テンプレートファイルのパス
         * @param string    $elements   変換文字列（タグ名 => 文字列 の連想配列）
         * @param string    $mailHeader     メールヘッダ（ヘッダ名 => ヘッダ値 の連想配列）
         * @param string    $attachment 添付ファイルがある場合は添付ファイルの連想配列(ファイル名 => ファイルパス)、
         *                              無い場合はNULL
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        function sendFromTemplate($path, &$elements, $mailHeader = NULL, $attachment = NULL){
            //メール本文生成
            $wkMailText        = $this->buildMail($path, $elements);

            //送信
            $this->send($wkMailText, $mailHeader, $attachment);
        }

        //-----------------------------------------------------------------------------------------------------------------------------------
        /**
         * メール送信
         * 
         * @param string    $mailText   メール本文を格納した文字列 or 配列
         * @param string    $mailHeader     メールヘッダ（ヘッダ名 => ヘッダ値 の連想配列）
         * @param string    $attachment 添付ファイルがある場合は添付ファイルの連想配列(ファイル名 => ファイルパス)、
         *                              無い場合はNULL
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        function send($mailText = '', $mailHeader = NULL, $attachment = NULL){
            if ($mailText !== ''){
                $this->_mailText    = $mailText;
            }
            if ($mailHeader !== NULL){
                $this->_mailHeader  = $mailHeader;
            }
            if ($attachment !== NULL){
                $this->_attachments = $attachment;
            }
            $rc    = $this->_setMailHeader();
            if (!$rc){return false;}

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //ヘッダ編集
            $wkHeader       = array();
            $wkHeader[]     = "From:".$this->_fromAddr;

            if (strlen(trim($this->_ccAddr)) > 0){
                $wkHeader[] = "Cc:".$this->_ccAddr;
            }
            if (strlen(trim($this->_bccAddr)) > 0){
                $wkHeader[] = "Bcc:".$this->_bccAddr;
            }

            $wkReturnPath   = (strlen(trim($this->_returnPath)) > 0 ? $this->_returnPath : $this->_fromAddr);
            $wkHeader[]     = "Return-Path:".$wkReturnPath;

            //添付ファイルがある場合は MultiPart にする
            if ($this->_attachments == null){
                $wkHeader[] = "Content-Type: text/plain; charset=".$this->_headerEncoding;
                $wkHeader[] = 'MIME-Version: 1.0; Content-Transfer-Encoding: '.$this->_headerBit.'bit';
            }
            else{
                $wkHeader[] = 'MIME-Version: 1.0';
                $wkHeader[]    = 'Content-Type: multipart/mixed; boundary="__BOUNDARY__--"';
                $wkHeader[] = "";
            }

            $wkMailHeader   = implode("\r\n", $wkHeader);

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //メール本文編集
            if (is_array($this->_mailText)){
                $wkMailText = implode("", $this->_mailText);
            }
            else{
                $wkMailText = $this->_mailText;
            }

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //メール本文エンコード
            $wkMailText     = mb_convert_encoding($wkMailText, $this->_encoding['TO'], $this->_encoding['FROM']);

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //本文編集
            if ($this->_attachments != null){
                //本文
                $wkMultiParts       = array();
                $wkMultiParts[]     = '--__BOUNDARY__--';
                $wkMultiParts[]     = 'Content-Type: text/plain; charset='.$this->_headerEncoding;
                $wkMultiParts[]     = 'MIME-Version: 1.0; Content-Transfer-Encoding: '.$this->_headerBit.'bit';
                $wkMultiParts[]     = '';
                $wkMultiParts[]     = $wkMailText;
                $wkMultiParts[]     = '--__BOUNDARY__--';

                //添付ファイル
                foreach ($this->_attachments as $fileName => $filePath) {
                    $wkMultiParts[] = 'Content-Type: application/octetstream;';
                    $wkMultiParts[] = 'Content-Transfer-Encoding: base64';
                    $wkMultiParts[] = 'Content-Disposition: attachment; filename="'.$fileName.'"';
                    $wkMultiParts[] = '';
                    $wkFile     = fopen($filePath, "r") or die("error"); //ファイルの読み込み
                    $wkContents = fread($wkFile, filesize($filePath));
                    fclose($wkFile);
                    $wkMultiParts[] = chunk_split(base64_encode($wkContents));
                    $wkMultiParts[] = '--__BOUNDARY__--';
                }

                $wkMailText         = implode("\n", $wkMultiParts);
            }

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //パラメータ編集
            $wkParam    = " -f".$wkReturnPath;

            //-----------------------------------------------------------------------------------------------------------------------------------------------------------
            //メール送信
            mail($this->_toAddr,
                 "=?".strtoupper($this->_headerEncoding)."?B?".base64_encode(mb_convert_encoding($this->_subject, $this->_encoding['TO'], $this->_encoding['FROM']))."?=",
                 $wkMailText,
                 $wkMailHeader,
                 $wkParam);
        }

        //-----------------------------------------------------------------------------------------------------------------------------------
        /** 
         * メールテンプレート変換
         *
         * @param string    $path       テンプレートファイルのパス
         * @param array     $elements   変換文字列（タグ名 => 文字列 の連想配列）
         * @return string   メール本文の文字列
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        function buildMail($path, &$elements){
            $wkTemplate        = array();
            $wkMailText        = array();

            $wkTemplate        = file($path);
            foreach ($wkTemplate as $line){
                //postfix改行コードの対処処理
                if ($this->_mta    == 'postfix'){
                    $wkLine        = str_replace("\r\n", "\n", $line);
                }

                //変換処理
                foreach ($elements as $elementID => $value){
                    $wkLine    = str_replace(TPL_START.$elementID.TPL_END, $value, $wkLine);
                }
                $wkMailText[]    = $wkLine;
            }
            return $wkMailText;
        }
        
        //-----------------------------------------------------------------------------------------------------------------------------------
        /** 
         * メール本文文字列セット
         * 文字列を直接メールする場合にメール本文をセットする
         * 
         * @param string    $mailText   メール本文文字列
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        public function setMailText($mailText){
            $this->_mailText    = $mailText;
        }
        
        //-----------------------------------------------------------------------------------------------------------------------------------
        /** 
         * メールヘッダー追加
         * 
         * @param string    $name       メールヘッダ名
         * @param string    $value      メールヘッダ値
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        public function addMailHeader($name, $value){
            $this->_mailHeader[$name]   = $value;
        }
                
        //-----------------------------------------------------------------------------------------------------------------------------------
        /** 
         * 添付ファイル追加
         * 
         * @param string    $name       添付ファイル
         * @param string    $value      添付ファイルフルパス
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        public function addAttachedFile($name, $path){
            $this->_attachments[$name] = $path;
        }
 
        //-----------------------------------------------------------------------------------------------------------------------------------
        /**
         * パラメータセット
         * 
         * @return boolean  $this->_mailHeaderのキーに保存されたパラメータ名が存在しない場合FALSE
         */
        //-----------------------------------------------------------------------------------------------------------------------------------
        private function _setMailHeader(){
            //セット無しの場合処理無し
            if (!isset($this->_mailHeader)){
                return true;
            }

            //セット
            foreach ($this->_mailHeader as $key => $value){
                switch($key){
                    //ヘッダ
                    case 'toAddr':
                        $this->_toAddr      = $value;
                        break;
                    case 'fromAddr':
                        $this->_fromAddr    = $value;
                        break;
                    case 'subject':
                        $this->_subject     = $value;
                        break;
                    //追加ヘッダ
                    case 'ccAddr':
                        $this->_ccAddr      = $value;
                        break;
                    case 'bccAddr':
                        $this->_bccAddr     = $value;
                        break;
                    //リターンパス
                    case 'returnPath':
                        $this->_returnPath  = $value;
                        break;
                    default:
                        return FALSE;
                }
            }

            //正常終了
            return TRUE;
        }
    }

?>
