<?php
    //========================================================================================================================
    /**
     * イベント管理クラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2010 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2010/8/26
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class EventManager{
        private $_eventList = array();
        private $_timerList = array();

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * コンストラクタ/デストラクタ
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function __construct(){

        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * イベント登録
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function addEvent($event){
            $this->_eventList[]   = $event;
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * タイマー登録
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function addTimerEvent($aTimer){
            $this->_timerList[]   = $aTimer;
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 登録イベント数取得
         * @return integer  登録イベント数
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function count(){
            return count($this->_eventList);
        }
        
        public function timerCount(){
            return count($this->_timerList);
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * イベント登録スクリプト作成
         * @return string   イベント登録情報リスト文字列
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function script(){
           return sprintf("var LBEvents = %s;", $this->items());
        }

        public function timerScript(){
           return sprintf("var LBTimers = %s;", $this->Timers());
        }

        public function items(){
            //イベントリスト作成
            $eventItems     = array();
            foreach ($this->_eventList as $_event){
                $eventItems[$_event->elementID()][$_event->eventType()]  = array($_event->type(), 
                                                                                 $_event->getTargetList(), 
                                                                                 $_event->getHandlerList(), 
                                                                                 $_event->getServerList(), 
                                                                                 $_event->getColumnList(), 
                                                                                 $_event->groupNames());
            }
            return json_encode($eventItems);
        }

        public function timers(){
            //イベントリスト作成
            $wTimerItems    = array();
            foreach ($this->_timerList as $_timer){
                $wTimerItems[$_timer->timerID()]  = array($_timer->timerType(),
                                                          $_timer->interval(), 
                                                          $_timer->getSW(),
                                                          $_timer->getHandlerList(), 
                                                          $_timer->getTargetList(), 
                                                          $_timer->getExecParams());
            }
            return json_encode($wTimerItems);
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 送信対象エレメントリスト文字列作成
         * @param array     $targetList         Ajax送信対象エレメントの連想配列
         * @return string   送信対象エレメントリスト文字列
         */
        //-------------------------------------------------------------------------------------------------------------------
        private function _targetElements($targetList){
        }
    }
    
    //========================================================================================================================
    /**
     * イベントクラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2010 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2010/8/26
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class Event{
        private $_elementID;
        private $_type;
        private $_eventType;
        private $_groupNames;
        private $_targetList    = array(AJAX_ITEM_COUNT => '');
        private $_handlerList   = array(AJAX_ITEM_COUNT => '');
        private $_execParamList = array(AJAX_ITEM_COUNT => '');
        private $_columnList    = array();
    
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * コンストラクタ/デストラクタ
         * @param string $elementID     エレメントID
         * @param string $eventType     イベント種類
         * @param array  $handlerList   イベントハンドラ名配列
         * @param array  $targetList    通信対象エレメントID配列
         * @param array  $serverList    サーバサイド起動情報配列
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function __construct($elementID, $eventType, $type, $handlerList, $targetList, $serverList, $groupNames){
            $this->_elementID       = $elementID;
            $this->_eventType       = $eventType;
            $this->_type            = $type;
            $this->_groupNames      = $groupNames;
            $this->_handlerList     = array_merge($this->_handlerList, $handlerList);
            $this->_targetList      = array_merge($this->_targetList, $targetList);
            $this->_execParamList   = array_merge($this->_execParamList, $serverList);
        }
        
        public function elementID(){
            return $this->_elementID;
        }
        
        public function eventType(){
            return $this->_eventType;
        }
        
        public function type(){
            return $this->_type;
        }
        
        public function groupNames(){
             return $this->_groupNames;
        }
        
        public function getHandlerList(){
            $this->_handlerList[AJAX_ITEM_COUNT]    = count($this->_handlerList) - 1;
            return $this->_handlerList;
        }
        
        public function getTargetList(){
            //送信エレメント無し
            if ($this->_targetList === NULL){
                $elements   = array(); 
            }
            //Ajax送信対象のエレメント群の配列
            else if (is_array($this->_targetList)){
                $this->_targetList[AJAX_ITEM_COUNT]    = count($this->_targetList) - 1;
                return $this->_targetList;
            }
            //コンテナエレメントとみなし、実行時に従属するエレメントを解析しAjax送信対象とする
            else{
                return  $this->_targetList;
            }
        }
        
        public function getServerList(){
            $this->_execParamList[AJAX_ITEM_COUNT]    = count($this->_execParamList) - 1;
            return $this->_execParamList;
        }
        
        public function getColumnList(){
            return $this->_columnList;
        }

        public function __call($name, $params){
            switch ($name){
                case 'sendElement':
                    $wTargetList            = (is_array($params[0]) ? $params[0] : array($params[0]));
                    $this->_targetList      = array_merge($this->_targetList, $wTargetList);
                    break;

                case 'handler':
                    $wHandlerList           = (is_array($params[0]) ? $params[0] : array($params[0] => $params[1]));
                    $this->_handlerList     = array_merge($this->_handlerList, $wHandlerList);
                    break;

                case 'execParam':
                    $wExecParamList         = (is_array($params[0]) ? $params[0] : array($params[0] => $params[1]));
                    $this->_execParamList   = array_merge($this->_execParamList, $wExecParamList);
                    break;

                case 'gridColumn':
                    $wColumnList            = (is_array($params[0]) ? $params[0] : array($params[0]));
                    $this->_columnList      = array_merge($this->_columnList, $wColumnList);
                     break;
            }
        }
        
    }
    
    //========================================================================================================================
    /**
     * タイマークラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2011 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2011/08/01
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class Timer{
        private $_timerID;
        private $_timerType;
        private $_interval;
        private $_targetList    = array(AJAX_ITEM_COUNT => '');
        private $_handlerList;
        private $_execParams;
        private $_sw;
    
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * コンストラクタ/デストラクタ
         * @param string  $aTimerID     タイマーID
         * @param integer $aInterval    タイマーイベント発生間隔
         * @param string  $aTimerType   タイマー種類('U':汎用/'A':Ajax)
         * @param array   $aHandler     イベントハンドラ名配列('U'の場合:イベントハンドラ(文字列)/'A'の場合:ポイント毎ハンドラリスト(連想配列))
         * @param array   $aTargetList  通信対象エレメントID配列
         * @param array   $aExecParams  サーバサイド起動情報配列('U'の場合:イベントハンドラ引数(文字列)/'A'の場合:Ajaxリクエストパラメータリスト(連想配列))
         * @param string  $aSW          起動時タイマーON/OFF
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function __construct($aTimerID, $aTimerType, $aInterval, $aHandler, $aTargetList, $aExecParams, $aSW = 'ON'){
            $this->_timerID         = $aTimerID;
            $this->_timerType       = $aTimerType;
            $this->_interval        = $aInterval;
            if ($aTimerType == 'U'){
                $this->_handlerList = $aHandler;
                $this->_execParams  = $aExecParams;
            }
            else{
                $this->_handlerList = array_merge(array(AJAX_ITEM_COUNT => ''), $aHandler);
                $this->_execParams  = array_merge(array(AJAX_ITEM_COUNT => ''), $aExecParams);
            }
            $this->_targetList      = array_merge($this->_targetList, $aTargetList);
            $this->_sw              = $aSW;
        }

        /**
         * 起動時タイマーON/OFF
         * @param boolean $aSW      TRUE/FALSE→ON/OFF
         */
        public function sw($aSW){
            $this->_sw  = ($aSW ? 'ON' : 'OFF');
        }

        //-----------------------------------------------------------------------------
        //アクセサ
        public function timerID(){
            return $this->_timerID;
        }
        
        public function timerType(){
            return $this->_timerType;
        }

        public function interval(){
            return $this->_interval;
        }
        
        public function getHandlerList(){
            if ($this->timerType() == 'A'){
                $this->_handlerList[AJAX_ITEM_COUNT]    = count($this->_handlerList) - 1;
            }
            return $this->_handlerList;
        }
        
        public function getTargetList(){
            $this->_targetList[AJAX_ITEM_COUNT]     = count($this->_targetList) - 1;
            return $this->_targetList;
        }
        
        public function getExecParams(){
            switch($this->_timerType){
                case 'U':
                    return $this->_execParams;

                case 'A':
                    if (!array_key_exists('method', $this->_execParams)){
                        $this->_execParams['method']    = 'onTimer_'.$this->timerID();
                    }
                    $this->_execParams[AJAX_ITEM_COUNT]     = count($this->_execParams) - 1;
                    return $this->_execParams;
            }
        }

        public function getSW(){
            return $this->_sw;
        }

    }

?>
