<?php
    
    //========================================================================================================================
    /**
     * SVGビルダークラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2010 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2010/2/26
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class SvgBuilder {
        //属性データ
        private $_width         = 0;
        private $_height        = 0;
        private $_id            = "";
        
        //描画データ
        private $_shapeList     = array();
        private $_svgText       = array();
     
        //-----------------------------------------------------------------------------------------------------------------------------------------
        /**
         * コンストラクタ/デストラクタ
         * 
         * @param integer   $width      キャンバスサイズ(幅)
         * @param integer   $height     キャンバスサイズ(高)
         */
        //-----------------------------------------------------------------------------------------------------------------------------------------
        function __construct($aWidth, $aHeight, $aID = null){
            $this->_width           = $aWidth;
            $this->_height          = $aHeight;
            $this->_id              = $aID;
        }

        public function __destruct(){
        
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * レスポンスデータ作成
         *
         * @param int x             ViewBox座標X
         * @param int y             ViewBox座標Y
         * @param int viewWidth     ViewBox幅
         * @param int viewHeight    ViewBox高
         * @return string  SVGタグ文字列データ
         */  
        //-------------------------------------------------------------------------------------------------------------------
        public function build($x, $y, $viewWidth, $viewHeight){
            //zxc('draw start:'.date('Y/m/d H:i:s').' => ', 'magick.html');
            
            $this->_buildBody($x, $y, $viewWidth, $viewHeight);            
            return implode("\n", $this->_svgText);

            //zx('draw end:'.date('Y/m/d H:i:s'), 'magick.html');
        }     
       
        public function buildToFile($aToFile, $x, $y, $viewWidth, $viewHeight){
            //zxc('draw start:'.date('Y/m/d H:i:s').' => ', 'magick.html');
            
            $this->_buildBody($x, $y, $viewWidth, $viewHeight);            
            file_put_contents($aToFile, $this->_svgText);
                                 
            //zx('draw end:'.date('Y/m/d H:i:s'), 'magick.html');
        }

        private function _buildBody($x, $y, $viewWidth, $viewHeight){
            $this->_svgText     = array();
            $wID    = ($this->_id == null ? "" : ' id="'.$this->_id.'" ');
            $this->_svgText[]   = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"'.$wID;
            $this->_svgText[]   = 'viewBox="'.$x.' '.$y.' '.$viewWidth.' '.$viewHeight.'" width="'.$this->_width.'" height="'.$this->_height.'" >';

            foreach($this->_shapeList as $_shape) {
                $this->_svgText[] = $_shape->draw();
            }

            $this->_svgText[]   = '</svg>';
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 描画シェイプ追加
         * 
         * @param object    $shape  シェイプオブジェクト
         */  
        //-------------------------------------------------------------------------------------------------------------------
        public function addShape($shape){
            $this->_shapeList[] = $shape;
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 四角形シェイプ追加
         * 
         * @param integer   $x              開始位置：横座標(ピクセル)
         * @param integer   $y              開始位置：縦座標(ピクセル)
         * @param integer   $width          横サイズ(ピクセル)
         * @param integer   $height         縦サイズ(ピクセル)
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)
         * @param integer   $strokeWidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)
         * @param string    $fillColor      塗りつぶし色指定文字列(色名 or #000000 のRGB形式)
         * @param array     $options        その他オプションリスト
         */  
        //-------------------------------------------------------------------------------------------------------------------
        public function addRectangle($x, $y, $width, $height, $strokeColor = 'black', $strokeWidth = 1, $fillColor = 'transparent', $options = array()){
            $wShape     = new Rectangle($x, $y, $width, $height, $strokeColor, $strokeWidth, $fillColor, $options);
            $this->addShape($wShape);
            return $wShape;
        }
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 直線シェイプ
         * 
         * @param integer   $x              開始位置：横座標(ピクセル)
         * @param integer   $y              開始位置：縦座標(ピクセル)
         * @param integer   $distanceX      起点からのX軸方向距離(ピクセル)
         * @param integer   $distanceY      起点からのY軸方向距離(ピクセル)
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)
         * @param integer   $strokeWidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)
         * @param array     $options        その他オプションリスト
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function addLine($x, $y, $distanceX, $distanceY, $strokeColor = 'black', $strokeWidth = 1, $options = array()){
            $wShape     = new Line($x, $y, $distanceX, $distanceY, $strokeColor, $strokeWidth, $options);
            $this->addShape($wShape);
            return $wShape;
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * テキストシェイプ
         * 
         * @param integer   $x              開始位置：横座標(ピクセル)
         * @param integer   $y              開始位置：縦座標(ピクセル)
         * @param integer   $angle          軸傾斜角度
         * (@param integer   $angleY        Y軸傾斜角度)
         * @param string    $text           描画文字列
         * @param string    $pointSize      文字サイズ(ポイント)
         * @param string    $font           使用フォント(日本語TTFを使用する場合はフルパスで指定する 例:FONT_DIR.'/kochi-gothic-subst.ttf')
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)：nullの場合セットしない
         * @param integer   $strokewidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)：nullの場合セットしない
         * @param string    $fillColor      塗りつぶし色指定文字列(色名 or #000000 のRGB形式)
         * @param array     $options        その他オプションリスト
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function addText($x, $y, $text, $color = 'black', $font = null, $options = array()){
            $wShape     = new Text($x, $y, $text, $color, $font, $options);
            $this->addShape($wShape);
            return $wShape;
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 折れ線シェイプ追加
         * 
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)
         * @param integer   $strokewidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)
         * @param string    $fillColor      塗りつぶし色指定文字列(色名 or #000000 のRGB形式)
         * @param array     $options        その他オプションリスト
         * @return object   シェイプオブジェクト
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function addPolyline($strokeColor = 'black', $strokeWidth = 1, $fillColor = 'white', $options = array()){
            $wShape     = new Polyline($strokeColor, $strokeWidth, $fillColor, $options);
            $this->addShape($wShape);
            return $wShape;
        }

    }
    
    //========================================================================================================================
    /**
     * SVGビルダー用シェイプ基本クラス
     *
     * LICENSE: This source file is subject to GNU GENERAL PUBLIC LICENSE Version 3
     * http://www.gnu.org/licenses/gpl-3.0.txt.
     *
     * @package    LabbitBox
     * @subpackage core
     * @copyright  2010 LabbitBox Development Team.
     * @license    GPL v3 http://www.gnu.org/licenses/gpl-3.0.txt
     * @author     neobaba <neobaba@labbitbox.org>
     * @version    v0.1
     * @since      2010/2/26
     * @link       http://labbitbox.org/
     * @see        http://labbitbox.org/
     */
    //========================================================================================================================
    class SvgShape{
        //プロパティ
        public $type;                   //シェイプタイプ(offset/rectangle/line/text)
        private $_styleList = array();
        private $_attrList  = array();
        
        //クラス変数
        protected $_id    = null;
        protected $_options = array();
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * シェイプ基本クラス
         * 
         * @param string    $type       シェイプタイプ
         */  
        //-------------------------------------------------------------------------------------------------------------------
       function __construct($type, $options = null){
            $this->type     = $type;
            $this->_options = $options;
        }

        public function id($aID){
            $this->_id  = $aID;
            return $this;
        }

        public function style($aStyles, $aValue = null){
            $this->_setStyle($aStyles, $aValue);
            return $this;
        }
       
        protected function _setStyle($aStyles, $aValue = null){
            if (is_array($aStyles)){
                $this->_styleList   = array_merge($this->_styleList, $aStyles);
            }
            else{
                $this->_styleList[$aStyles] = $aValue;
            }
        }

        protected function _getStyle(){
            if (count($this->_styleList) == 0){
                return "";
            } 

            $wEdit  = array();
            foreach($this->_styleList as $_style => $_value){
                $wEdit[]    = $_style.':'.$_value.'';    
            } 
            return 'style="'.implode(';', $wEdit).'"';
        }

        public function attribute($aAttributes, $aValue = null){
            $this->_setAttribute($aAttributes, $aValue);
            return $this;
        }

        protected function _setAttribute($aAttributes, $aValue = null){
            if (is_array($aAttributes)){
                $this->_attrList    = array_merge($this->_attrList, $aAttributes);
            }
            else{
                $this->_attrList[$aAttributes]  = $aValue;
            }
        }

        protected function _getAttribute(){
            if (count($this->_attrList) == 0){
                return "";
            }

            $wEdit  = array();
            foreach($this->_attrList as $_attr => $_value){
                $wEdit[]    = $_attr.'="'.$_value.'"';
            }
            return " ".implode(' ', $wEdit);
        }

    }

    //========================================================================================================================
    /**
     * SVGビルダー用シェイプ:Rectangleクラス
     */
    //========================================================================================================================
    class Rectangle extends SvgShape{
        private $_x             = 0;
        private $_y             = 0;
        private $_width         = 0;
        private $_height        = 0;
        private $_fill          = "";

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 四角形シェイプ
         * @param integer   $x              開始位置：横座標(ピクセル)
         * @param integer   $y              開始位置：縦座標(ピクセル)
         * @param integer   $width          横サイズ(ピクセル)
         * @param integer   $height         縦サイズ(ピクセル)
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)
         * @param integer   $strokeWidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)
         * @param string    $fillColor      塗りつぶし色指定文字列(色名 or #000000 のRGB形式)
         * @param array     $options        その他オプションリスト
         */
        //-------------------------------------------------------------------------------------------------------------------
        function __construct($x, $y, $width, $height, $strokeColor = 'black', $strokeWidth = 1, $fillColor = "transparent", $attr=array(), $options = array()){
            $this->_x               = $x;
            $this->_y               = $y;
            $this->_width           = $width;
            $this->_height          = $height;
            $this->_fill            = ($fillColor == "transparent" ? null : $fillColor);
            $this->_setAttribute($attr);

            $this->_setStyle(
                array(
                    'stroke'        => $strokeColor,
                    'stroke-width'  => $strokeWidth,
                )
            );

            parent::__construct('RECTANGLE', $options);
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 四角形編集
         * @return string   drawパラメータ文字列
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function draw(){
            $wID    = ($this->_id == null ? "" : ' id="'.$this->_id.'"');
            $wSvgTag    = array();
            $wSvgTag[]  = "<rect".$wID;
            $wSvgTag[]  = 'x="'.$this->_x.'"';
            $wSvgTag[]  = 'y="'.$this->_y.'"';
            $wSvgTag[]  = 'width="'.$this->_width.'"';
            $wSvgTag[]  = 'height="'.$this->_height.'"';
            if ($this->_fill !== null){
                $wSvgTag[]  = 'fill="'.$this->_fill.'"';
            }
            if (($wStyles = $this->_getStyle()) !== ""){
                $wSvgTag[]  = $wStyles;
            }
            $wSvgTag[]  = $this->_getAttribute();
            $wSvgTag[]  = "/>";

            return implode(" ", $wSvgTag);
        }
    }
            
    //========================================================================================================================
    /**
     * SVGビルダー用シェイプ:Lineクラス
     */
    //========================================================================================================================
    class Line extends SvgShape{
        private $_sx            = 0;
        private $_sy            = 0;
        private $_ex            = 0;
        private $_ey            = 0;
        
        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 四角形シェイプ
         * @param integer   $sx             開始位置：X座標(ピクセル)
         * @param integer   $sy             開始位置：Y座標(ピクセル)
         * @param integer   $ex             終了位置：X座標(ピクセル)
         * @param integer   $ey             終了位置：Y座標(ピクセル)
         * @param string    $strokeColor    線色指定文字列(色名 or #000000 のRGB形式)
         * @param integer   $strokeWidth    線幅(ピクセル：0の場合はデフォルトもしくは設定済み幅値)
         * @param array     $attr           その他属性リスト
         * @param array     $options        その他オプションリスト
         */  
        //-------------------------------------------------------------------------------------------------------------------
        function __construct($sx, $sy, $ex, $ey, $strokeColor = 'black', $strokeWidth = 1, $attr=array(), $options = array()){
            $this->_sx      = $sx;
            $this->_sy      = $sy;
            $this->_ex      = $ex;
            $this->_ey      = $ey;
            $this->_setAttribute($attr);
            
            $this->_setStyle(
                array(
                    'stroke'        => $strokeColor,
                    'stroke-width'  => $strokeWidth,
                )
            );
           
            parent::__construct('LINE', $options);    
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 直線編集
         * @return string   drawパラメータ文字列
         */  
        //-------------------------------------------------------------------------------------------------------------------
        public function draw(){
            $wID    = ($this->_id == null ? "" : ' id="'.$this->_id.'"');
            $wSvgTag    = array();
            $wSvgTag[]  = "<line".$wID;
            $wSvgTag[]  = 'x1="'.$this->_sx.'"';
            $wSvgTag[]  = 'y1="'.$this->_sy.'"';
            $wSvgTag[]  = 'x2="'.$this->_ex.'"';
            $wSvgTag[]  = 'y2="'.$this->_ey.'"';
            if (($wStyles = $this->_getStyle()) !== ""){
                $wSvgTag[]  = $wStyles;
            }
            $wSvgTag[]  = $this->_getAttribute();
            $wSvgTag[]  = "/>";

            return implode(" ", $wSvgTag);
        }
    }

    //========================================================================================================================
    /**
     * SVGビルダー用シェイプ:Textクラス
     */
    //========================================================================================================================
    class Text extends SvgShape{
        private $_x         = 0;
        private $_y         = 0;
        private $_text      = "";
        private $_color     = "";
        private $_font      = array();

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 四角形シェイプ
         * @param integer   $x              開始位置：横座標(ピクセル)
         * @param integer   $y              開始位置：縦座標(ピクセル)
         * @param string    $text           描画文字列
         * @param string    $color          文字色 
         * @param array     $font           フォントスタイル
         * @param array     $attr           その他属性リスト
         * @param array     $options        その他オプションリスト
         */
        //-------------------------------------------------------------------------------------------------------------------
        function __construct($x, $y, $text, $color = 'black', $font = null, $attr=array(), $options = array()){
            $this->_x       = $x;
            $this->_y       = $y;
            $this->_text    = $text;
            $this->_color   = $color;
            $this->_font    = $font;
            $this->_setAttribute($attr);

            parent::__construct('TEXT', $options);
        }

        //-------------------------------------------------------------------------------------------------------------------
        /**
         * 直線編集
         * @return string   drawパラメータ文字列
         */
        //-------------------------------------------------------------------------------------------------------------------
        public function draw(){
            $wID    = ($this->_id == null ? "" : ' id="'.$this->_id.'"');
            $wSvgTag    = array();
            $wSvgTag[]  = "<text".$wID;
            $wSvgTag[]  = 'x="'.$this->_x.'"';
            $wSvgTag[]  = 'y="'.$this->_y.'"';
            $wSvgTag[]  = 'fill="'.$this->_color.'"';
            foreach($this->_font as $_attr => $_value){
                $wSvgTag[]  = $_attr.'="'.$_value.'"';
            }
            $wSvgTag[]  = $this->_getAttribute();
            $wSvgTag[]  = ">";

            return implode(" ", $wSvgTag).$this->_text."</text>";
        }

    }
    
?>
