/*
 * created  2008/07/26 
 * Copyright 2008 yhj All rights reserved.
 */
package jp.que.ti.yhj.less.than.more;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

/**
 * {@link java.io.InputStream} で実現できない逆方向ファイル読み込みを実現するためのクラスです
 * 
 * @author yhj
 * @author $Author$ (last modified)
 * @version $Revision$
 */
public class ForwardByteReader extends ByteReaderBase {
	private static final int DEFAULT_BUFFERE_LENGTH = 3 * 1024;
	private File file;
	private int bufferSize;

	private final int getBufferSize() {
		return bufferSize;
	}

	private final void setBufferSize(int bufferSize) {
		this.bufferSize = bufferSize;
	}

	@Override
	public final File getFile() {
		return file;
	}

	private final void setFile(File file) {
		this.file = file;
	}

	private static FileInputStream initFileInputStream(File file) {
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
		} catch (FileNotFoundException e) {
			throw new RuntimeException(e.getMessage(), e);
		}
		return fis;
	}

	private long position = 0;

	BufferedInputStream bufferedInputStream;

	/**
	 * コンストラクタ
	 * 
	 * @param file
	 *            このクラスが使用するファイル
	 */
	public ForwardByteReader(File file) {
		this(file, DEFAULT_BUFFERE_LENGTH);
	}

	/**
	 * コンストラクタ
	 * 
	 * @param file
	 *            このクラスが使用するファイル
	 * @param bufferSize
	 *            読み込みに使用するバッファサイズ
	 */
	public ForwardByteReader(File file, int bufferSize) {
		setFile(file);
		setBufferSize(bufferSize);
		setBufferedInputStream(new BufferedInputStream(
				initFileInputStream(file), bufferSize));
	}

	@Override
	public void close() throws IOException {
		getBufferedInputStream().close();

	}

	private final BufferedInputStream getBufferedInputStream() {
		return bufferedInputStream;
	}

	@Override
	public long getPosition() {
		return this.position;
	}

	/**
	 * 前方方向であることを意味する true を返却します。
	 * 
	 * @return turu を返却します。
	 * @see jp.que.ti.yhj.less.than.more.ByteReader#isForward()
	 */
	@Override
	final public boolean isForward() {
		return true;
	}

	@Override
	public int read() {
		try {
			this.position++;
			return bufferedInputStream.read();
		} catch (IOException e) {
			throw new RuntimeException("読み込みに失敗しました", e);
		}
	}

	private final void setBufferedInputStream(
			BufferedInputStream bufferedInputStream) {
		this.bufferedInputStream = bufferedInputStream;
	}

	/**
	 * 次回呼び出される {@link #read()} により取得するアドレスを設定します。
	 * 
	 * @see jp.que.ti.yhj.less.than.more.ByteReader#setPosition(long)
	 * @throws IllegalArgumentException
	 *             現在のアドレスよりも前のアドレスを引数 position に指定した場合スローします。
	 */
	@Override
	public void setPosition(long position) {
		long gap = position - getPosition();
		long skipSize = gap;
		if (gap == 0) {
			return;
		} else if (gap > 0) { // 現在の BufferedInputStream そのまま使用可能
		} else if (gap < 0) { // 現在の BufferedInputStream を初期化
			try {
				getBufferedInputStream().close();
			} catch (IOException e) {
				throw new RuntimeException(
						"ファイルの読み込み位置をクリアするため、現在使用している InputStream をクローズしようとしましたが、失敗しました。",
						e);
			}
			final FileInputStream fis = initFileInputStream(getFile());
			final BufferedInputStream bis = new BufferedInputStream(fis,
					getBufferSize());
			setBufferedInputStream(bis);
			skipSize = position;
		}
		try {
			getBufferedInputStream().skip(skipSize);
		} catch (IOException e) {
			throw new RuntimeException("読み込み位置をスキップすることができませんでした", e);
		}
		this.position = position;

	}

	/**
	 * 引数 startPosition から、改行文字までのbyte 配列を返却します
	 * 
	 * @param startPosition
	 *            読み込み先頭位置
	 * @return 引数 startPosition から、改行文字までのbyte 配列
	 */
	@Override
	public byte[] readLine(long startPosition) {
		ByteReader reader = null;
		byte[] rtn = null;
		try {
			reader = new ForwardByteReader(getFile());
			rtn = readToCrLf(startPosition);
		} finally {
			try {
				reader.close();
			} catch (IOException e) {
				throw new RuntimeException("ファイルクローズに失敗しました", e);
			}
		}
		return rtn;
	}
}
