/***********************************************************
	ar.c -- main file
***********************************************************/

static char *usage =
    "ar -- compression archiver -- written by Haruhiko Okumura\n"
    "  PC-VAN:SCIENCE        CompuServe:74050,1022\n"
    "  NIFTY-Serve:PAF01022  INTERNET:74050.1022@compuserve.com\n"
    "Usage: ar command archive [file ...]\n"
    "Commands:\n"
    "   a: Add files to archive (replace if present)\n"
    "   x: Extract files from archive\n"
    "   r: Replace files in archive\n"
    "   d: Delete files from archive\n"
    "   p: Print files on standard output\n"
    "   l: List contents of archive\n"
    "If no files are named, all files in archive are processed,\n"
    "   except for commands 'a' and 'd'.\n"
    "You may copy, distribute, and rewrite this program freely.\n";

/***********************************************************

Structure of archive block (low order byte first):
-----preheader
 1	basic header size
		= 25 + strlen(filename) (= 0 if end of archive)
 1	basic header algebraic sum (mod 256)
-----basic header
 5	method ("-lh0-" = stored, "-lh5-" = compressed)
 4	compressed size (including extended headers)
 4	original size
 4	not used
 1	0x20
 1	0x01
 1	filename length (x)
 x	filename
 2	original file's CRC
 1	0x20
 2	first extended header size (0 if none)
-----first extended header, etc.
-----compressed file

***********************************************************/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "ar.h"

struct lzh_header {
    char filename[1024];
    int  namelen;
    char method[5];
    int compsize;
    int origsize;
    int ftime;
    int file_crc;
    char os_id;
};

struct lha_opts opts;

init_opts()
{
    opts.nocompress = 0;
    opts.outdir = NULL;
    opts.quiet = 0;
}

#define FNAME_MAX (255 - 25)    /* max strlen(filename) */

int unpackable;                 /* global, set in io.c */
ulong compsize, origsize;       /* global */

static uchar headersize;
static char *temp_name;

static void
print_usage()
{
    puts("usage: ...");
    exit(0);
}

static uint
ratio(ulong a, ulong b)
{                               /* [(1000a + [b/2]) / b] */
    int i;

    for (i = 0; i < 3; i++)
        if (a <= ULONG_MAX / 10)
            a *= 10;
        else
            b /= 10;
    if ((ulong) (a + (b >> 1)) < a) {
        a >>= 1;
        b >>= 1;
    }
    if (b == 0)
        return 0;
    return (uint) ((a + (b >> 1)) / b);
}

static void
put_to_header(char *buf, int i, int n, ulong x)
{
    while (--n >= 0) {
        buf[i++] = (uchar) ((uint) x & 0xFF);
        x >>= 8;
    }
}

static ulong
get_from_header(char *buf, int i, int n)
{
    ulong s;

    s = 0;
    while (--n >= 0)
        s = (s << 8) + buf[i + n];   /* little endian */
    return s;
}

static uint
calc_headersum(char *buf, int size)
{
    int i;
    uint s;

    s = 0;
    for (i = 0; i < size; i++)
        s += buf[i];
    return s & 0xFF;
}

int
get_byte(char *buf)
{
    return *(unsigned char*)buf;
}

uint16_t
get_word(char *buf)
{
    return get_byte(buf) | (get_byte(buf+1) << 8);
}

uint32_t
get_dword(char *buf)
{
    return get_byte(buf) |
        (get_byte(buf+1) << 8) |
        (get_byte(buf+2) << 16) |
        (get_byte(buf+3) << 24);
}

void
get_char(char *buf, char *p, size_t size)
{
    memcpy(p, buf, size);
}

void
put_byte(char *buf, int c)
{
    *buf = (unsigned char)(c & 0xff);
}

void
put_word(char *buf, uint16_t c)
{
    put_byte(buf,   c);
    put_byte(buf+1, c>>8);
}

void
put_dword(char *buf, uint32_t c)
{
    put_byte(buf, c);
    put_byte(buf+1, c>>8);
    put_byte(buf+2, c>>16);
    put_byte(buf+3, c>>24);
}

void
put_char(char *buf, char *p, size_t size)
{
    memcpy(buf, p, size);
}

static int
read_header(FILE *fp, struct lzh_header *h)
{
    int headersize;
    int headersum;
    char buf[4096];
    int ext_headersize;

    headersize = (uchar) fgetc(fp);
    if (headersize == 0)
        return 0;               /* end of archive */
    headersum = (uchar) fgetc(fp);
    fread_crc(buf, headersize, fp);     /* CRC not used */
    if (calc_headersum(buf, headersize) != headersum)
        error("Header sum error");

    get_char(&buf[0], h->method, 5);
    h->compsize = get_dword(&buf[5]);
    h->origsize = get_dword(&buf[9]);
    h->ftime    = get_dword(&buf[13]);
    /* attrib   = get_byte(&buf[17]); */
    /* level    = get_byte(&buf[18]); */         /* level */
    h->namelen = get_byte(&buf[19]);
    get_char(&buf[20], h->filename, h->namelen);
    h->filename[h->namelen] = 0;
    h->file_crc = get_word(&buf[20+h->namelen]);
    h->os_id    = get_byte(&buf[20+h->namelen+2]);

    ext_headersize = get_word(&buf[20+h->namelen+3]);

    while (ext_headersize != 0) {
        fprintf(stderr, "There's an extended header of size %u.\n",
                ext_headersize);
        h->compsize -= ext_headersize;

        /* skip ext header */
        if (fseek(arcfile, ext_headersize - 2, SEEK_CUR))
            error("Can't read");
        ext_headersize = fgetc(arcfile);
        ext_headersize += (uint) fgetc(arcfile) << 8;
    }

    return 1;                   /* success */
}


void
write_header(FILE *fp, int headersize, struct lzh_header *h)
{
    char buf[4096], *p = buf;
    int sum;

    put_byte(&buf[0], headersize);
    put_byte(&buf[1], 0); /* dummy */
    put_char(&buf[2], h->method, 5);
    put_dword(&buf[7], h->compsize);   /* packed size */
    put_dword(&buf[11], h->origsize);   /* original size */
    put_dword(&buf[15], h->ftime);   /* ftime */
    put_byte(&buf[19], 0x20);   /* attribute */
    put_byte(&buf[20], 1);  /* level */
    put_byte(&buf[21], h->namelen); /* length of pathname */
    put_char(&buf[22], h->filename, h->namelen);
    put_word(&buf[22+h->namelen], h->file_crc);
    put_byte(&buf[22+h->namelen+2], 'M');
    put_word(&buf[22+h->namelen+3], 0x0000); /* next header size */

    sum = calc_headersum(buf+2, headersize);
    put_byte(&buf[1], sum);

    fwrite_crc(buf, headersize+2, fp);
}

static void
skip(FILE *fp, struct lzh_header *h)
{
    fseek(fp, h->compsize, SEEK_CUR);
}

static void
copy(FILE *arcfile, FILE *outfile, struct lzh_header *h)
{
    uint n;
    uchar buffer[DICSIZ];

    write_header(outfile, headersize, h);
    while (h->compsize != 0) {
        n = (uint) ((h->compsize > DICSIZ) ? DICSIZ : h->compsize);
        if (fread((char *) buffer, 1, n, arcfile) != n)
            error("Can't read");
        if (fwrite((char *) buffer, 1, n, outfile) != n)
            error("Can't write");
        h->compsize -= n;
    }
}

static void
store(void)
{
    uint n;
    uchar buffer[DICSIZ];

    origsize = 0;
    crc = INIT_CRC;
    while ((n = fread((char *) buffer, 1, DICSIZ, infile)) != 0) {
        fwrite_crc(buffer, n, outfile);
        origsize += n;
    }
    compsize = origsize;
}

static int
add(int replace_flag, char *filename)
{
    long headerpos, arcpos;
    uint r;
    struct lzh_header h;

    if ((infile = fopen(filename, "rb")) == NULL) {
        fprintf(stderr, "Can't open %s\n", filename);
        return 0;               /* failure */
    }
    if (replace_flag) {
        if (opts.quiet < 2)
            printf("Replacing %s ", filename);
        skip(arcfile, &h);
    }
    else {
        if (opts.quiet < 2)
            printf("Adding %s ", filename);
    }

    strcpy(h.filename, filename);
    h.namelen = strlen(filename);

    headersize = 25 + h.namelen;
    memcpy(h.method, "-lh5-", 5);  /* compress */

    headerpos = ftell(outfile);
    write_header(outfile, headersize, &h);
    arcpos = ftell(outfile);

    origsize = compsize = 0;
    crc = INIT_CRC;
    if (opts.nocompress) {
        unpackable = 1;
    }
    else {
        unpackable = 0;
        encode();
    }

    if (unpackable) {
        h.method[3] = '0';        /* store */
        rewind(infile);
        fseek(outfile, arcpos, SEEK_SET);
        store();
    }
    h.file_crc = crc ^ INIT_CRC;
    fclose(infile);

    h.compsize = compsize;
    h.origsize = origsize;

    fseek(outfile, headerpos, SEEK_SET);
    write_header(outfile, headersize, &h);
    fseek(outfile, 0L, SEEK_END);
    r = ratio(compsize, origsize);
    if (opts.quiet < 2)
        printf(" %d.%d%%\n", r / 10, r % 10);
    return 1;                   /* success */
}

int
get_line(char *s, int n)
{
    int i, c;

    i = 0;
    while ((c = getchar()) != EOF && c != '\n')
        if (i < n)
            s[i++] = (char) c;
    s[i] = '\0';
    return i;
}

static void
extract(int to_file, struct lzh_header *h)
{
    int n, method;
    uint ext_headersize;
    uchar buffer[DICSIZ];

    if (to_file) {
        while ((outfile = fopen(h->filename, "wb")) == NULL) {
            fprintf(stderr, "Can't open %s\nNew filename: ", h->filename);
            if (get_line(h->filename, FNAME_MAX) == 0) {
                fprintf(stderr, "Not extracted\n");
                skip(arcfile, h);
                return;
            }
            h->namelen = strlen(h->filename);
        }
        if (opts.quiet < 2)
            printf("Extracting %s ", h->filename);
    }
    else {
        outfile = stdout;
        if (opts.quiet < 2)
            printf("===== %s =====\n", h->filename);
    }
    crc = INIT_CRC;
    method = h->method[3];      /* -lh*5*- */
    h->method[3] = ' ';
    if (!strchr("045", method) || memcmp("-lh -", h->method, 5)) {
        fprintf(stderr, "Unknown method: %u\n", method);
        skip(arcfile, h);
    }
    else {
        crc = INIT_CRC;
        if (method != '0')
            decode_start();
        while (h->origsize != 0) {
            n = (uint) ((h->origsize > DICSIZ) ? DICSIZ : h->origsize);
            if (method != '0')
                decode(n, buffer);
            else if (fread((char *) buffer, 1, n, arcfile) != n)
                error("Can't read");
            fwrite_crc(buffer, n, outfile);
            if (outfile != stdout && opts.quiet < 1) {
                putc('.', stdout);
            }
            h->origsize -= n;
        }
    }
    if (to_file)
        fclose(outfile);
    else
        outfile = NULL;
    if ((crc ^ INIT_CRC) != h->file_crc)
        fprintf(stderr, "CRC error\n");
}

static void
list_start(void)
{
    if (opts.quiet < 2)
        printf("Filename         Original Compressed Ratio CRC Method\n");
}

static void
list(struct lzh_header *h)
{
    uint r;

    printf("%-14.*s", h->namelen, h->filename);
    if (h->namelen > 14)
        printf("\n              ");
    r = ratio(h->compsize, h->origsize);
    printf(" %10lu %10lu %u.%03u %04X %5.5s\n",
           h->origsize, h->compsize, r / 1000, r % 1000, h->file_crc, h->method);
}

static int
match(char *s1, char *s2)
{
    for (;;) {
        while (*s2 == '*' || *s2 == '?') {
            if (*s2++ == '*')
                while (*s1 && *s1 != *s2)
                    s1++;
            else if (*s1 == 0)
                return 0;
            else
                s1++;
        }
        if (*s1 != *s2)
            return 0;
        if (*s1 == 0)
            return 1;
        s1++;
        s2++;
    }
}

static int
search(int argc, char *argv[], struct lzh_header *h)
{
    int i;

    if (argc == 0)
        return 1;
    for (i = 0; i < argc; i++)
        if (match(h->filename, argv[i]))
            return 1;
    return 0;
}

static void
exitfunc(void)
{
    fclose(outfile);
    remove(temp_name);
}

#include "getopt_long.h"

int
parse_args(int argc, char **argv)
{
    int c;

    for (;;) {
        int this_option_optind = optind ? optind : 1;
        int option_index = 0;

        enum {
            LHA_OPT_HELP = 128,
        };

        static struct option long_options[] = {
            /* name, has_arg, *flag, val */
            /* has_arg:
               no_argument (0)
               required_argument (1)
               optional_argument (2)
               flag:
               NULL: getopt_long() return val
               non-NULL: getopt_long() return 0, and *flag set val.
            */
            {"help", no_argument, NULL, LHA_OPT_HELP},
            {0, 0, 0, 0}
        };

        c = getopt_long(argc, argv, "q[012]w:z", long_options, &option_index);

        if (c == -1) break;

        switch (c) {
        case 0:                 /* set vallue by long option */
            break;
        case 'q':               /* quiet mode */
            opts.quiet = 2;     /* level 2 */
            if (optarg)
                opts.quiet = *optarg - '0';
            break;
        case 'w':               /* extract directory */
            if (!optarg)
                error("extract directory does not specified for `-w'");
            if (*optarg == '=')
                optarg++;

            opts.outdir = optarg;
            break;
        case 'z':               /* no compress */
            opts.nocompress = 1;
            break;
        case LHA_OPT_HELP:
            print_usage();
            break;
        default:
            break;
        }
    }
}

FILE *
open_tempfile()
{
    temp_name = tmpnam(NULL);
    outfile = fopen(temp_name, "wb");
    if (outfile == NULL)
        error("Can't open temporary file");
    atexit(exitfunc);

    return outfile;
}

int
main(int argc, char *argv[])
{
    int i, j, cmd, count, nfiles, found, done;
    char *archive_file;
    struct lzh_header h;

    if (argv[1] == 0)
        print_usage();

    /*take a command character */
    {
        char *arg1;

        arg1 = argv[1];
        if (arg1[0] == '-')
            arg1++;
        if (arg1[0] == 0)
            print_usage();

        cmd = *arg1;
        if (arg1[1] == 0) {
            /* -<cmd> -<opts> ... */
            argv++;
            argc--;
        }
        else {
            /* -<cmd><opts> => -<opts> */
            *arg1 = '-';
        }
    }

    parse_args(argc, argv);
    argv += optind;
    argc -= optind;

    archive_file = argv[0];

    argv++;
    argc--;

    temp_name = NULL;

    make_crctable();
    count = done = 0;

    switch (cmd) {
    case 'a':
    case 'c':
        outfile = open_tempfile();
        if (*argv == 0)
            error("archived files are not specified.");
        for (i = 0; i < argc; i++) {
            add(0, argv[i]);
        }

        fputc(0, outfile);      /* end of archive */
        if (ferror(outfile) || fclose(outfile) == EOF)
            error("Can't write");
        remove(archive_file);
        fclose(arcfile);
        if (rename(temp_name, archive_file) == -1)
            error("fail to rename()");

        exit(0);
        break;
    case 'r':
    case 'd':
        outfile = open_tempfile();
    case 'x':
    case 'p':
    case 'l':
    case 'v':
        /* Open archive. */
        arcfile = fopen(archive_file, "rb");
        if (arcfile == NULL)
            error("Can't open archive '%s'", argv[2]);

        break;
    default:
        print_usage();
        break;
    }

    /* change directory to extract dir */
    if (cmd == 'x') {
        struct stat *stbuf;

        if (opts.outdir) {
            if (mkdir(opts.outdir, 0777) == -1) {
                if (errno != EEXIST)
                    error("cannot make directory \"%s\"", opts.outdir);
            }

            if (chdir(opts.outdir) == -1)
                error("cannot change directory \"%s\"", opts.outdir);
        }
    }

    while (!done && read_header(arcfile, &h)) {

        compsize = h.compsize;
        origsize = h.origsize;

        found = search(argc, argv, &h);
        switch (cmd) {
        case 'r':
            if (found) {
                if (add(1, *argv))
                    count++;
                else
                    copy(arcfile, outfile, &h);
            }
            else
                copy(arcfile, outfile, &h);
            break;
        case 'a':
        case 'd':
            if (found) {
                count += (cmd == 'D');
                skip(arcfile, &h);
            }
            else
                copy(arcfile, outfile, &h);
            break;
        case 'x':
        case 'p':
            if (found) {
                extract(cmd == 'x', &h);
                if (++count == nfiles)
                    done = 1;
            }
            else
                skip(arcfile, &h);
            break;
        case 'l':
        case 'v':
            if (found) {
                if (count == 0)
                    list_start();
                list(&h);
                if (++count == nfiles)
                    done = 1;
            }
            skip(arcfile, &h);
            break;
        }
    }

    if (cmd != 'p') {
        if (opts.quiet < 2)
            printf("  %d files\n", count);
    }
    return EXIT_SUCCESS;
}
