/***********************************************************
	ar.c -- main file
***********************************************************/

static char *version = "0.01";

static char *usage =
    "ar -- compression archiver -- written by Haruhiko Okumura\n"
    "  PC-VAN:SCIENCE        CompuServe:74050,1022\n"
    "  NIFTY-Serve:PAF01022  INTERNET:74050.1022@compuserve.com\n"
    "Usage: ar command archive [file ...]\n"
    "Commands:\n"
    "   a: Add files to archive (replace if present)\n"
    "   x: Extract files from archive\n"
    "   r: Replace files in archive\n"
    "   d: Delete files from archive\n"
    "   p: Print files on standard output\n"
    "   l: List contents of archive\n"
    "If no files are named, all files in archive are processed,\n"
    "   except for commands 'a' and 'd'.\n"
    "You may copy, distribute, and rewrite this program freely.\n";

/***********************************************************

Structure of archive block (low order byte first):
-----preheader
 1	basic header size
		= 25 + strlen(filename) (= 0 if end of archive)
 1	basic header algebraic sum (mod 256)
-----basic header
 5	method ("-lh0-" = stored, "-lh5-" = compressed)
 4	compressed size (including extended headers)
 4	original size
 4	not used
 1	0x20
 1	0x01
 1	filename length (x)
 x	filename
 2	original file's CRC
 1	0x20
 2	first extended header size (0 if none)
-----first extended header, etc.
-----compressed file

***********************************************************/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <time.h>
#include <utime.h>
#include <unistd.h>
#include "ar.h"

struct lha_method methods[] = {
    /* id, dicbit, pbit, maxmatch */
    /* note: dicbit == 0 means no compress */
    {"-lh0-", 0,  0,  0},  /* 0: no compress */
    {"-lh1-", 12, 0, 60},  /* 1: 2^12 =  4KB dynamic huffman (LHarc) */
    {"-lh2-", 13, 0,256},  /* 2: 2^13 =  8KB dynamic huffman */
    {"-lh3-", 13, 0,256},  /* 3: 2^13 =  8KB static huffman */
    {"-lh4-", 12, 4,256},  /* 4: 2^12 =  4KB static huffman (pos and len)*/
    {"-lh5-", 13, 4,256},  /* 5: 2^13 =  8KB static huffman (pos and len)*/
    {"-lh6-", 15, 5,256},  /* 6: 2^15 = 32KB static huffman (pos and len)*/
    {"-lh7-", 16, 5,256},  /* 7: 2^16 = 64KB static huffman (pos and len)*/
    {"-lzs-", 11, 0, 17},  /* 8: 2^11 =  2KB (LArc) */
    {"-lz5-", 12, 0, 17},  /* 9: 2^12 =  4KB (LArc) */
    {"-lz4-", 0,  0,  0},  /*10: no compress (LArc) */
    {"-lhd-", 0,  0,  0},  /*11: directory */
};

struct lha_opts opts;

struct lha_method *
which_method(char *id)
{
    int i;

    for (i = 0; i < sizeof(methods)/sizeof(methods[0]); i++) {
        if (strncmp(id, methods[i].id, sizeof(methods[0].id)) == 0) {
            return &methods[i];
        }
    }
    return NULL;
}

static char *temp_name = NULL;

static void
print_usage()
{
    printf("%s", usage);
    exit(0);
}

static void
print_version()
{
    printf("version %s\n", version);
    exit(0);
}

uint
ratio(ulong a, ulong b)
{                               /* [(1000a + [b/2]) / b] */
    int i;

    for (i = 0; i < 3; i++)
        if (a <= ULONG_MAX / 10)
            a *= 10;
        else
            b /= 10;
    if ((ulong) (a + (b >> 1)) < a) {
        a >>= 1;
        b >>= 1;
    }
    if (b == 0)
        return 0;
    return (uint) ((a + (b >> 1)) / b);
}

void
skip(FILE *fp, struct lzh_header *h)
{
    int i;
    if (opts.archive_to_stdio)
        for (i = 0; i < h->compsize; i++)
            fgetc(fp);
    else
        fseek(fp, h->compsize, SEEK_CUR);
}

static void
copy(FILE *arcfile, FILE *outfile, struct lzh_header *h)
{
    uint n;
    uchar buffer[MAXDICSIZ];

    write_header(outfile, h);
    while (h->compsize != 0) {
        n = (uint) ((h->compsize > sizeof(buffer)) ? sizeof(buffer) : h->compsize);
        if (fread((char *) buffer, 1, n, arcfile) != n)
            error("Can't read");
        if (fwrite((char *) buffer, 1, n, outfile) != n)
            error("Can't write");
        h->compsize -= n;
    }
}

int
get_line(char *s, int n)
{
    int i, c;

    i = 0;
    while ((c = getchar()) != EOF && c != '\n')
        if (i < n)
            s[i++] = (char) c;
    s[i] = '\0';
    return i;
}

static int
match(char *s1, char *s2)
{
    for (;;) {
        while (*s2 == '*' || *s2 == '?') {
            if (*s2++ == '*')
                while (*s1 && *s1 != *s2)
                    s1++;
            else if (*s1 == 0)
                return 0;
            else
                s1++;
        }
        if (*s1 != *s2)
            return 0;
        if (*s1 == 0)
            return 1;
        s1++;
        s2++;
    }
}

static int
search(int argc, char *argv[], struct lzh_header *h)
{
    int i;

    if (argc == 0)
        return -1;
    for (i = 0; i < argc; i++)
        if (argv[i] && match(h->filename, argv[i]))
            return i+1;
    return 0;
}

static void
exitfunc(void)
{
    if (temp_name)
        remove(temp_name);
}

#include "getopt_long.h"

void
parse_args(int argc, char **argv)
{
    int c;

    for (;;) {
        /* int this_option_optind = optind ? optind : 1; */
        int option_index = 0;

        enum {
            LHA_OPT_HELP = 128,
            LHA_OPT_VERSION,
        };

        static struct option long_options[] = {
            /* name, has_arg, *flag, val */
            /* has_arg:
               no_argument (0)
               required_argument (1)
               optional_argument (2)
               flag:
               NULL: getopt_long() return val
               non-NULL: getopt_long() return 0, and *flag set val.
            */
            {"help", no_argument, NULL, LHA_OPT_HELP},
            {"version", no_argument, NULL, LHA_OPT_VERSION},
            {0, 0, 0, 0}
        };

        c = getopt_long(argc, argv, "012fgo[567]q[012]vw:z",
                        long_options, &option_index);

        if (c == -1) break;     /* end of parsing options */

        switch (c) {
        case '?':
            print_usage();
            break;
        case 0:
            /* set value by long option */
            break;
        case '0': case '1': case '2':
            /* header level */
            opts.header_level = c - '0';
            break;
        case 'f':
            opts.force_extract = 1;
            break;
        case 'g':
            opts.generic = 1;
            opts.header_level = 0;
            break;
        case 'o':
            /* compress method */
            {
                int idx = 1;    /* -o means -lh1- method */

                if (optarg)
                    idx = *optarg - '0'; /* -lh[567]- method */

                opts.method   = &methods[idx];
            }
            break;
        case 'q':
            /* quiet mode */
            opts.quiet = 2;     /* -q is equivalent to -q2 */
            if (optarg)
                opts.quiet = *optarg - '0';
            break;
        case 'v':
            /* verbose mode */
            opts.verbose = 1;
            break;

        case 'w':
            /* extract directory */
            if (!optarg)
                error("extract directory does not specified for `-w'");
            if (*optarg == '=')
                optarg++;

            opts.outdir = optarg;
            break;
        case 'z':               /* no compress */
            opts.nocompress = 1;
            break;
        case LHA_OPT_HELP:
            print_usage();
            break;
        case LHA_OPT_VERSION:
            print_version();
            break;
        default:
            break;
        }
    }
}

FILE *
open_tempfile()
{
    FILE *outfile;

    temp_name = tmpnam(NULL);
    outfile = fopen(temp_name, "wb");
    if (outfile == NULL)
        error("Can't open temporary file");
    atexit(exitfunc);

    return outfile;
}

int
main(int argc, char *argv[])
{
    int i, cmd, count, nfiles, found, done;
    char *archive_file;
    struct lzh_header h;
    int arc_count;
    struct lzh_istream r, *rp;
    struct lzh_ostream w, *wp;
    FILE *arcfile = NULL;
    FILE *outfile = NULL;

    rp = &r;
    wp = &w;

    INITIALIZE_OPTS(opts);

    if (argv[1] == 0)
        print_usage();

    /*take a command character */
    {
        char *arg1;

        arg1 = argv[1];
        if (arg1[0] == '-')
            arg1++;
        if (arg1[0] == 0)
            print_usage();

        cmd = *arg1;
        if (arg1[1] == 0) {
            /* -<cmd> -<opts> ... */
            argv++;
            argc--;
        }
        else {
            /* -<cmd><opts> => -<opts> */
            *arg1 = '-';
        }
    }

    parse_args(argc, argv);
    argv += optind;
    argc -= optind;

    archive_file = argv[0];

    if (strcmp(archive_file, "-") == 0)
        opts.archive_to_stdio = 1;

    argv++;
    argc--;

    make_crctable();
    count = done = nfiles = 0;

    switch (cmd) {
    case 'a':
    case 'u':
    case 'c':
        if (opts.archive_to_stdio)
            opts.quiet = 2;

        outfile = open_tempfile();
        wp->fp = outfile;
        if (*argv == 0)
            error("archived files are not specified.");

        if (!opts.archive_to_stdio && (cmd == 'a' || cmd == 'u')) {
            if (file_exists(archive_file)) {
                arcfile = fopen(archive_file, "rb");
                if (arcfile == NULL)
                    error("Can't open archive '%s'", archive_file);

                break;
            }
        }
        for (i = 0; i < argc; i++) {
            add(wp, 0, argv[i], strlen(argv[i]));
        }

        fputc(0, outfile);      /* end of archive */
        if (ferror(outfile))
            error("Can't write");
        fclose(outfile);
        if (opts.archive_to_stdio) {
            if (move_file_to_stream(temp_name, stdout) == -1)
                error("fail to move_file_to_stream(): %s -> %s",temp_name,"stdout");
        }
        else {
            unlink(archive_file);
            if (xrename(temp_name, archive_file) == -1)
                error("fail to rename(): %s -> %s",temp_name,archive_file);
        }
        exit(0);
        break;
    case 'r':
    case 'd':
        if (argc == 0) {
            message("No files given in argument, do nothing.");
            exit(0);
        }
        outfile = open_tempfile();
    case 'x':
    case 'p':
    case 'l':
    case 'v':
        /* Open archive. */
        if (opts.archive_to_stdio) {
            arcfile = stdin;
        }
        else {
            arcfile = fopen(archive_file, "rb");
        }
        if (arcfile == NULL)
            error("Can't open archive '%s'", archive_file);

        break;
    default:
        print_usage();
        break;
    }

    /* change directory to extract dir */
    if (cmd == 'x') {
        if (opts.outdir) {
            if (mkdir(opts.outdir, 0777) == -1) {
                if (errno != EEXIST)
                    error("cannot make directory \"%s\"", opts.outdir);
            }

            if (chdir(opts.outdir) == -1)
                error("cannot change directory \"%s\"", opts.outdir);
        }
    }

    arc_count = 0;

    while (!done && read_header(arcfile, &h)) {

        arc_count++;

        found = search(argc, argv, &h);
        switch (cmd) {
        case 'a':
        case 'u':
            if (found>0) {
                argv[found-1] = 0;

                if (cmd == 'u') {
                    time_t mtime;

                    if (file_mtime(h.filename, &mtime) == -1 || h.mtime > mtime) {
                        copy(arcfile, outfile, &h);
                        break;
                    }
                }

                if (add(wp, 1, h.filename, h.namelen)) {
                    skip(arcfile, &h);
                    count++;
                }
                else
                    copy(arcfile, outfile, &h);
            }
            else
                copy(arcfile, outfile, &h);
            break;
        case 'd':
            if (found) {
                count++;
                message("'%s' deleted", h.filename);
                skip(arcfile, &h);
            }
            else
                copy(arcfile, outfile, &h);
            break;
        case 'x':
        case 'p':
            if (found != 0) {
                rp->fp = arcfile;
                rp->compsize = h.compsize;
                extract(rp, cmd == 'x', &h);
                if (++count == nfiles)
                    done = 1;
            }
            else
                skip(arcfile, &h);
            break;
        case 'l':
        case 'v':
            if (found != 0) {
                if (count == 0)
                    list_start();
                list(&h);
                if (++count == nfiles)
                    done = 1;
            }
            skip(arcfile, &h);
            break;
        }
    }

    if (cmd == 'a' || cmd == 'u') {
        for (i = 0; i < argc; i++) {
            if (argv[i]) {
                count++;
                add(wp, 0, argv[i], strlen(argv[i]));
            }
        }
    }

    if (cmd != 'p') {
        if (opts.quiet < 2)
            printf("  %d files\n", count);
    }

    if (count > 0 && (cmd == 'd' || cmd == 'a' || cmd == 'u')) {
        fputc(0, outfile);      /* end of archive */
        if (ferror(outfile))
            error("Can't write");
        if (!opts.archive_to_stdio)
            unlink(archive_file);
        fclose(outfile);
        fclose(arcfile);
        if (cmd == 'd') {
            if (arc_count > count) {
                if (xrename(temp_name, archive_file) == -1)
                    error("fail to rename(): %s -> %s",temp_name,archive_file);
            }
            else {
                message("The archive file \"%s\" was removed because it would be empty.", archive_file);
            }
        }
        else {
            if (xrename(temp_name, archive_file) == -1)
                error("fail to rename(): %s -> %s",temp_name,archive_file);
        }
        exit(0);
    }
    return EXIT_SUCCESS;
}
