/*
 * LiveML - LiveML is screen(graphic) controling library using XML.
 *
 * LGPL License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/liveml/
 * All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#include "livemlrunner.h"

// parameter types
bool LMLParameter::has(size_t index)
{
  DCODE(printf("LMLParameter::has(%d)\n", index);)
  return param->has(index);
}

XMLParameter* LMLParameter::get(size_t index)
{
  DCODE(printf("LMLParameter::getTag(%d)\n", index);)
  return param->get(index);
}

// decode variables
fixed_float LiveMLRunner::calcuDecode(const char **input, void* work)
{
  LMLParameter &param = *(LMLParameter*)work;

  const char *data = *input;
  fixed_float v;
  variable_size variable;
  calcu_decode_data result;

  switch (*data++)
  {
  case CALCU_VARIABLE:
    variable = *(variable_size*)data;
    v = param.getVariable(variable)->getFixedFloat(variable);
    data += sizeof(variable_size);
    break;
  case CALCU_NUMERIC:
    v = *(fixed_float*)data;
    data += sizeof(fixed_float);
    break;
  case CALCU_ELEMENT:
    {
      XMLElement **el = (XMLElement**)data;
      data += sizeof(XMLElement*);
      v = param.runner->runCommand(param, *el);
    }
    break;
  default:
    data--;
  }

  *input = data;

  return v;
}
const char* LiveMLRunner::variableStringDecoder(variable_size id, void *work)
{
  LMLParameter &param = *(LMLParameter*)work;
  DCODE(printf("LiveMLRunner::variableString(%d)\n", id);)

  return param.getVariable(id)->getString(id);
}

namespace lmlCommand
{
  LML_CMDFUNC(action)
  {
    if (param.close)
    {
      param.action->current()->tag = NULL;
      return LML_CMDRTN_NULL;
    }

    return LML_CMDRTN_CHILD;
  }

  LML_CMDFUNC(endAction)
  {
    if (!param.has(0))
    {
      param.action->close();
      return LML_CMDRTN_STOP | LML_CMDRTN_NEXT;
    }

    param.obj->endAction(param.get(0)->getString());

    return LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(event)
  {
    if (!param.close)
    {
      if (!param.has(1) || param.get(1)->getInteger())
      {
        return LML_CMDRTN_CHILD;
      }
    }

    param.action->current()->tag = NULL;

    return LML_CMDRTN_NULL;
  }

  LML_CMDFUNC(print)
  {
    printf(param.get(0)->getString());
    printf("\n");

    return LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(active)
  {
    DCODE(printf("active [%s]\n", param.get(0)->getString());)
    LMLActorObject *obj = param.runner->addActor(param.get(0)->getString());
    if (param.has(1))
    {
      variable_size id = ((XMLVar*)param.get(1))->id;
      size_t objId = param.runner->getActorObjectId(*obj);

      param.getVariable(id)->setFixedFloat(id, int_to_fixed_float((int)objId));
    }

    return LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(activeAction)
  {
    DCODE(printf("activeAction: %s\n", param.get(0)->getString());)

    if (!param.obj->addAction(param.get(0)->getString()))
    {
      ActionParser *ap = param.runner->parser->actionContainer[param.get(0)->getString()];
      param.obj->addAction(ap);
    }

    return LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(repeat)
  {
    if (!param.close) *param.action->repeatCounter.push_back() = 0;

    unsigned int *counter = param.action->repeatCounter.back();

    if (param.has(0))
    {
      if (*counter >= param.get(0)->getInteger())
      {
        param.action->repeatCounter.remove(counter);

        return LML_CMDRTN_NEXT;
      }
      (*counter)++;
    }

    return LML_CMDRTN_CHILD;
  }

  LML_CMDFUNC(rcontinue)
  {
    XMLElement **t = &param.action->current()->tag;
    *t = *param.action->current()->resumeTagStack.current();

    return LML_CMDRTN_NULL;
  }

  LML_CMDFUNC(rbreak)
  {
    if (param.action->repeatCounter.back() == NULL) return LML_CMDRTN_NEXT;
    (*param.action->repeatCounter.back()) = 0;

    return rcontinue(param);
  }

  LML_CMDFUNC(wait)
  {
    if (param.action->waitCounter) param.action->waitCounter--;
    else param.action->waitCounter = param.get(0)->getInteger();

    return param.action->waitCounter ? LML_CMDRTN_STOP : LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(exit)
  {
    param.runner->dropActor(*param.obj);

    return LML_CMDRTN_STOP;
  }

  LML_CMDFUNC(var)
  {
    variable_size id = ((XMLVar*)param.get(0))->id;
    DCODE(printf("var[%d] = %d\n", id, param.getVariable(id)->getFixedFloat(id));)
    param.getVariable(id)->setFixedFloat(id, ((XMLNumeric*)param.get(1))->getFixedFloat());
    DCODE(printf("var[%d] = %d\n", id, param.getVariable(id)->getFixedFloat(id));)

    return LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(callAction)
  {
    DCODE(printf("callAction: %s\n", param.get(0)->getString());)

    if (!param.obj->addAction(param.get(0)->getString()))
    {
      ActionParser *ap = param.runner->parser->actionContainer[param.get(0)->getString()];
      param.obj->addAction(ap);
    }

    return LML_CMDRTN_STOP | LML_CMDRTN_NEXT | LML_CMDRTN_REACTION;
  }

  LML_CMDFUNC(callEvent)
  {
    DCODE(printf("callEvent: %d\n", param.get(0)->getUInteger());)

    if (param.obj->activeActionStack.current() == param.action)
    {
      param.runner->callEvent(*param.obj, param.get(0)->getUInteger());
      return LML_CMDRTN_NEXT;
    }

    param.obj->setEvent(param.get(0)->getUInteger());

    return LML_CMDRTN_STOP | LML_CMDRTN_NEXT | LML_CMDRTN_REACTION;
  }

  LML_CMDFUNC(rint)
  {
    return int_to_fixed_float(param.get(0)->getInteger());
  }

  LML_CMDFUNC(count)
  {
    return int_to_fixed_float(*param.action->repeatCounter.back());
  }

  LML_CMDFUNC(rif)
  {
    if (param.close) return LML_CMDRTN_NEXT;

    return param.get(0)->getInteger() ? LML_CMDRTN_CHILD : LML_CMDRTN_NEXT;
  }

  LML_CMDFUNC(focus)
  {
    LMLActorObject *obj = param.runner->getActorObject(param.get(0)->getUInteger());

    obj->addAction(param.action->current()->action);
    obj->activeActionStack.current()->current()->tag = (XMLElement*)param.action->current()->tag->child;

    LMLParameter p(NULL, param.runner, obj, obj->activeActionStack.current());
    LML_CMDRTN result = param.runner->runAction(p);

    return LML_CMDRTN_NEXT;
  }
}

LiveMLRunner::LiveMLRunner(LiveMLParser *p)
{
  DCODE(printf("LiveMLRunner::LiveMLRunner() begin.\n");)
  parser = p;
  setMaxActors(1000);  // unauthorised
  text_decode_init(1024); // unauthorised
  calcu_decode_callback(calcuDecode);

  _commandFuncs.resize(parser->countTagType());
  registerCommand("print", lmlCommand::print);
  registerCommand("active", lmlCommand::active);
  registerCommand("action", lmlCommand::action);
  registerCommand("activeAction", lmlCommand::activeAction);
  registerCommand("endAction", lmlCommand::endAction);
  registerCommand("event", lmlCommand::event);
  registerCommand("repeat", lmlCommand::repeat);
  registerCommand("count", lmlCommand::count);
  registerCommand("continue", lmlCommand::rcontinue);
  registerCommand("break", lmlCommand::rbreak);
  registerCommand("if", lmlCommand::rif);
  registerCommand("wait", lmlCommand::wait);
  registerCommand("exit", lmlCommand::exit);
  registerCommand("var", lmlCommand::var);
  registerCommand("callAction", lmlCommand::callAction);
  registerCommand("callEvent", lmlCommand::callEvent);
  registerCommand("focus", lmlCommand::focus);
  registerCommand("int", lmlCommand::rint);

  DCODE(printf("LiveMLRunner::LiveMLRunner() end.\n");)
}

LiveMLRunner::~LiveMLRunner()
{
  text_decode_release();
}

bool LiveMLRunner::registerCommand(
  const char* name,
  LML_CMDRTN (*func)(LMLParameter&)
)
{
  tag_type id = parser->getTagTypeId(name);
  if (id == UNameContainer<TagType>::NOTFOUND) return false;

  _commandFuncs[id].func = func;

  return true;
}

LML_CMDRTN LiveMLRunner::runCommand(LMLParameter &param, XMLElement *tag)
{
  LML_CMDRTN (*func)(LMLParameter&) = NULL;
  TagType *rt = tag->type;

  DCODE(printf("LiveMLRunner::runCommand()\n");)

  LML_CMDRTN result = LML_CMDRTN_NEXT;

  func = _commandFuncs[rt->id].func;
  if (func != NULL)
  {
    DCODE(printf("execute command: %s\n", rt->getName().c_str());)
    param.param = &tag->param;
    result = (*func)(param);
    DCODE(printf("finish command: %s\n", rt->getName().c_str());)
  }

  return result;
}

LML_CMDRTN LiveMLRunner::runChildAction(LMLParameter& param)
{
  LML_CMDRTN result = LML_CMDRTN_NULL;
  ChildAction *ca = param.action->current();

  DCODE(printf("LiveMLRunner::runChildAction()\n");)

  if (ca == NULL) return result;

  while (ca->tag != NULL)
  {
    calcu_decode_workarea((void*)&param);

    param.close = ca->resumeTagStack.current() != NULL;
    if (param.close) param.close = ca->tag == *ca->resumeTagStack.current();
    if (param.close) ca->resumeTagStack.close();

    result = runCommand(param, ca->tag);

    if (result & LML_CMDRTN_CHILD)
    {
      *ca->resumeTagStack.add() = ca->tag;
      ca->tag = (XMLElement*)ca->tag->child;
    }
    if (result & LML_CMDRTN_NEXT)
    {
      ca->tag = (XMLElement*)ca->tag->next;
      if (ca->tag == NULL)
      {
        if (ca->resumeTagStack.current() != NULL)
        {
          ca->tag = *ca->resumeTagStack.current();
        }
      }
    }
    if (result & LML_CMDRTN_STOP) break;
  }

  return result;
}

LML_CMDRTN LiveMLRunner::runAction(LMLParameter& param)
{
  LML_CMDRTN result = LML_CMDRTN_NULL;
  ActiveAction *aa = param.action;

  DCODE(printf("LiveMLRunner::runAction()\n");)

  if (aa == NULL) return result;

  while (ChildAction *ca = aa->current())
  {
    result = runChildAction(param);
    if (ca->tag == NULL) aa->close();
    if (result & LML_CMDRTN_STOP && !(result & LML_CMDRTN_REACTION)) break;
  }
  
  return result;
}

void LiveMLRunner::callEvent(LMLActorObject &obj, const char *name)
{
  callEvent(obj, (event_type)parser->eventTypeContainer.getId(name));
}

void LiveMLRunner::callEvent(LMLActorObject &obj, event_type id)
{
  obj.activeEventStack.clear();
  if (!obj.setEvent(id)) return;

  LMLParameter param(NULL, this, &obj, obj.activeEventStack.current());

  while (param.action != NULL)
  {
    LML_CMDRTN result = runAction(param);
    param.action = list_prev(param.action);
  }
}

void LiveMLRunner::callEvent(event_type id)
{
  LMLActorObject *actor = _actorList.front();
  while (actor)
  {
    LMLActorObject *actorNext = list_next(actor);
    callEvent((LMLActorObject&)*actor, id);
    actor = actorNext;
  }
}

bool LiveMLRunner::runActiveActions(LMLActorObject &obj)
{
  LMLParameter param(NULL, this, &obj, obj.activeActionStack.current());

  ActiveAction *prev;

  while (param.action != NULL)
  {
    LML_CMDRTN result = runAction(param);

    prev = list_prev(param.action);
    if (!param.action->has()) obj.activeActionStack.close(param.action);
    
    param.action = prev;
  }

  return obj.activeActionStack.has();
}

bool LiveMLRunner::runObject(LMLActorObject &obj)
{
  return runActiveActions(obj);
}

LMLActorObject* LiveMLRunner::addActor(const char *name)
{
  DCODE(printf("add actor: start\n");)
  LMLActorObject *obj;
  size_t type = 0;

  if (parser->actorContainer[name]->tag->param.has(1))
  {
    type = parser->actorContainer[name]->tag->param.getUInteger(1);
  }

  obj = createActor(type);
  obj->parser = parser->actorContainer[name];
  obj->addAction((action_type)0);
  callEvent(*obj, (event_type)0);
  DCODE(printf("add actor: end[%x]\n", obj);)

  return obj;
}

void LiveMLRunner::dropActor(LMLActorObject &obj)
{
  DCODE(printf("LiveMLRunner::dropActor(%x)\n", &obj);)
  callEvent(obj, (event_type)1);
  obj.release();
  _actorList.remove(&obj);
}

bool LiveMLRunner::run(void)
{
  bool running = false;

  DCODE(printf("LiveMLRunner::run()\n");)

  LMLActorObject *actor = _actorList.front();
  LMLActorObject *actorPrev, *actorNext;
  while (actor)
  {
    DCODE(printf("execute actor: start [%x]\n", actor);)
    actorPrev = list_prev(actor);
    actorNext = list_next(actor);
    running |= runObject(*actor);
    if (NULL == actorNext)
    {
      if ((LMLActorObject*)_actorList.allocator()->get_free() == actor)
      {
        actor = actorPrev;
      }
      if (NULL != actor)
      {
        actorNext = list_next(actor);
      }
    }
    DCODE(printf("execute actor: end [%x]\n", actor);)
    actor = actorNext;
  }

  return running;
}

LMLActorObject* LiveMLRunner::getActorObject(size_t id)
{
  return (*_actorList.allocator())[id];
}

size_t LiveMLRunner::getActorObjectId(LMLActorObject &obj)
{
  return _actorList.allocator()->id(&obj);
}

void LiveMLRunner::setMaxActors(size_t size)
{
  _actionAllocator.resize(size * 10);
  _repeatAllocator.resize(_actionAllocator.size() * 10);
  _childActionAllocator.resize(_actionAllocator.size() * 10);
  _tagPtrAllocator.resize(_actionAllocator.size() * 10);
  _numericVariableAllocator.resize(size * VAR_SCOPE_MAX * 50);

  variable.numericList.allocator(&_numericVariableAllocator);

  ChildAction *ca;
  while (ca = (ChildAction*)_childActionAllocator.add())
  {
    ca->resumeTagStack.list.allocator(&_tagPtrAllocator);
    ca->variable.numericList.allocator(&_numericVariableAllocator);
  }
  _childActionAllocator.clear();

  ActiveAction *aa;
  while (aa = (ActiveAction*)_actionAllocator.add())
  {
    aa->repeatCounter.allocator(&_repeatAllocator);
    aa->childActionStack.list.allocator(&_childActionAllocator);
    aa->clear();
  }
  _actionAllocator.clear();

  _actorList.resize(size);
  while (true)
  {
    LMLActorObject *obj = _actorList.push_back();
    if (obj == NULL) break;

    setObjectAllocators(*obj);
  }
  _actorList.clear(true);
}

void LiveMLRunner::setObjectAllocators(LMLActorObject &obj)
{
  DCODE(printf("LiveMLRunner::setObjectAllocators(%x)\n", &obj);)
  obj.setVariableAllocator(&_numericVariableAllocator);
  obj.activeActionStack.list.allocator(&_actionAllocator);
  obj.activeEventStack.list.allocator(&_actionAllocator);
  obj.release();
}

LMLActorObject* LiveMLRunner::createActor(size_t type)
{
  LMLActorObject *obj;

  switch (type)
  {
  case 0:
    obj = _actorList.push_back(false);
    break;
  case 1:
    obj = _actorList.front();
    if (NULL != obj && obj->parser->tag->param.has(1) && 1 == obj->parser->tag->param.getUInteger(1))
    {
      dropActor(*obj);
    }
    obj = _actorList.push_front(false);
    break;
  }

  return obj;
}

LMLVariable* LMLParameter::getVariable(variable_size id)
{
  DCODE(printf("LiveMLRunner::getVariable(%d[%d])\n", id, VAR_SCOPE_TYPE(id));)
  LMLVariable *v = &runner->variable;
  switch (VAR_SCOPE_TYPE(id))
  {
  case VAR_SCOPE_LOCAL:
    DCODE(printf("is local variable\n");)
    v = &action->current()->variable;
    break;
  case VAR_SCOPE_MEMBER:
    DCODE(printf("is member variable\n");)
    v = &obj->variable;
    break;
  }

  return v;
}
