/**
 * text - original text format.
 *
 * MIT License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/c-utils/
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copiGes or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIAGBILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#include "text.h"
#include <string.h>

enum
{
  TEXT_VARIABLE = 'v',
  TEXT_STRING = 's',
  TEXT_EOD = '\0'
};

static char *decode_buf = NULL;
static char *decode_buf_cur = NULL;
static size_t decode_buf_size = 0;

void text_decode_init(size_t buf_size)
{
  decode_buf = malloc(buf_size);
  decode_buf_size = buf_size;
  text_decode_clear();
}

void text_decode_clear(void)
{
  decode_buf_cur = decode_buf;
}

void text_decode_release(void)
{
  free(decode_buf);
  decode_buf_size = 0;
}

const char* text_decode(const char* data, const char* (*func)(variable_size, void*), void* work)
{
  const char *front_decode_buf = decode_buf_cur;

  if (decode_buf_cur - decode_buf >= decode_buf_size) return NULL;
  *decode_buf_cur = '\0';

  while (*data != TEXT_EOD)
  {
    const char *str;
    size_t str_size;

    switch (*data++)
    {
    case TEXT_VARIABLE:
      str = func(*(variable_size*)data, work);
      data += sizeof(variable_size);
      break;
    case TEXT_STRING:
    default: /* text support */
      str = data;
      data += strlen(data) + 1;
    }

    /* copy string */
    str_size = strlen(str);
    if (str_size)
    {
      memcpy(decode_buf_cur, str, str_size + 1);
      decode_buf_cur += str_size;
    }
  }

  decode_buf_cur++;

  return front_decode_buf;
}

char* text_encode(char* text, variable_size (*func)(const char*, void*), void *work)
{
  char *encode_buf;
  char *temp_buf = malloc(strlen(text) * 4); /* maybe suffices... */
  char *temp_buf_cur = temp_buf;
  char *str = text;
  size_t encode_size;

  for (; *text != '\0'; text++)
  {
    /* variable */
    if (text[0] == '$' && text[1] == '{')
    {
      char *name;

      /* save string */
      *temp_buf_cur = TEXT_STRING;
      temp_buf_cur++;
      *text = '\0';
      strcpy(temp_buf_cur, (const char*)str);
      temp_buf_cur += strlen(str) + 1;
      *text = '$';

      /* save variable */
      name = text += 2;
      for (; *text != '}' && *text != '\0'; text++);
      *text = '\0';
      *temp_buf_cur = TEXT_VARIABLE;
      temp_buf_cur++;
      *(variable_size*)temp_buf_cur = func(name, work);
      temp_buf_cur += sizeof(variable_size);
      *text = '}';

      str = text + 1;
    }
  }

  /* save string */
  if (*str != '\0')
  {
    *temp_buf_cur = TEXT_STRING;
    temp_buf_cur++;
    strcpy(temp_buf_cur, (const char*)str);
    temp_buf_cur += strlen(str) + 1;
  }

  // end of data
  *temp_buf_cur = TEXT_EOD;
  temp_buf_cur++;

  /* release temp buffer */
  encode_size = temp_buf_cur - temp_buf;
  encode_buf = malloc(encode_size);
  memcpy(encode_buf, temp_buf, encode_size);
  free(temp_buf);

  return encode_buf;
}

int text_data_size(const char *data)
{
  const char *cur = data;

  for (;*cur != TEXT_EOD; cur++)
  {
    switch (*cur)
    {
    case TEXT_VARIABLE:
      cur += sizeof(variable_size);
      break;
    case TEXT_STRING:
      cur += strlen(cur) + 1;
      break;
    }
  }

  return cur - data + 1;
}
