/**
 * LiveML - LiveML is screen(graphic) controling library using XML.
 *
 * LGPL License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/liveml/
 * All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#ifndef __XMLPARSER_H__
#define __XMLPARSER_H__

#include <vector>
#include <string>
#include "list.hpp"
using namespace std;

#ifdef _DEBUG
#define DCODE(code) code
#else
#define DCODE(code)
#endif

#define PARAMFUNC(name) XMLParameter* name(const char *value, XMLParser &parser)
#define LML_TAGFUNC(name) XML_TAGRTN name(Parameter &param, XMLParser &parser)

typedef enum
{
  XML_TAGRTN_NULL = 0,
  XML_TAGRTN_LEVEL,
  XML_TAGRTN_STOP,
  XML_TAGRTN_SKIP
} XML_TAGRTN;

typedef size_t tag_type;
typedef size_t param_type;

class XMLNode;
class XMLParser;
class TagType;

class ParameterOption {
protected:
  struct ReplaceValue {
    string value;
    string to;
  };
  size_t _index;
  string _name;
  param_type _type;
  vector<ReplaceValue> _replaces;

public:
  ~ParameterOption()
  {
  }

  void set(size_t index, param_type type, const char *name = NULL)
  {
    if (name != NULL) _name = name;
    _index = index;
    _type = type;
  }

  string& getName(void)
  {
    return _name;
  }

  size_t getIndex(void)
  {
    return _index;
  }

  param_type getType(void)
  {
    return _type;
  }

  void addReplaceValue(const char* value, const char *to)
  {
    ReplaceValue rep = {value, to};
    _replaces.push_back(rep);
  }
  const char* getReplaceValue(const char* value)
  {
    for (size_t i = 0; i < _replaces.size(); i++)
    {
      if (_replaces[i].value == value) return _replaces[i].to.c_str();
    }
    return value;
  }
};

class XMLParameter
{
public:
  virtual int getInteger(void) { return 0; }
  virtual unsigned int getUInteger(void) { return 0; }
  virtual float getFloat(void) { return 0; }
  virtual const char* getString(void) { return ""; }
};

class XMLString : public XMLParameter
{
public:
  const char *text;

  XMLString(const char *text) : XMLParameter()
  {
    char *buf = (char*)malloc(strlen(text) + 1);
    strcpy(buf, text);

    this->text = buf;
  }

  ~XMLString()
  {
    free((void*)text);
  }

  const char* getString(void) { return text; }
};

class XMLSizeT : public XMLParameter
{
  size_t _size;
public:
  XMLSizeT(size_t size) : XMLParameter()
  {
    _size = size;
  }

  int getInteger(void) { return _size; }
  unsigned int getUInteger(void) { return _size; }
};

class Parameter
{
  vector<XMLParameter*> _list;
public:
  Parameter(void)
  {
  }

  ~Parameter()
  {
    for (size_t i = 0; i < _list.size(); i++) delete _list[i];
  }

  void unset(size_t index)
  {
    if (_list[index] == NULL) return;
    delete _list[index];
    _list[index] = NULL;
  }

  bool has(size_t index)
  {
    if (_list.size() < index + 1) return false;
    return _list[index] != NULL;
  }

  XMLParameter* get(size_t index)
  {
    if (_list.size() < index + 1) return NULL;
    return _list[index];
  }

  void set(size_t index, XMLParameter *node)
  {
    if (_list.size() < index + 1) _list.resize(index + 1);
    else unset(index);
    _list[index] = node;
  }

  int getInteger(size_t index)
  {
    return (*_list[index]).getInteger();
  }

  unsigned int getUInteger(size_t index)
  {
    return (*_list[index]).getUInteger();
  }

  float getFloat(size_t index)
  {
    return (*_list[index]).getFloat();
  }

  const char* getString(size_t index)
  {
    return (*_list[index]).getString();
  }
};

class XMLNode
{
public:
  XMLNode *next;

  XMLNode(void)
  {
    next = NULL;
  }

  virtual ~XMLNode()
  {
    if (next != NULL) delete next;
  }

  virtual bool isText(void)
  {
    return false;
  }
};

class XMLElement : public XMLNode
{
public:
  TagType *type;
  XMLNode *child;
  Parameter param;

  XMLElement(TagType *type) : XMLNode()
  {
    this->type = type;
    child = NULL;
  }

  ~XMLElement()
  {
    if (child != NULL) delete child;
  }

  bool isText(void)
  {
    return false;
  }
};

class XMLTextNode : public XMLNode
{
public:
  const char *text;

  XMLTextNode(const char *text) : XMLNode()
  {
    char *buf = (char*)malloc(strlen(text) + 1);
    strcpy(buf, text);

    this->text = buf;
  }

  ~XMLTextNode()
  {
    free((void*)text);
  }

  bool isText(void)
  {
    return true;
  }

  const char* getString(void) { return text; }
};

class UNameItem
{
protected:
  string _name;
public:
  void setName(string name)
  {
    _name = name;
  }
  string& getName()
  {
    return _name;
  }
};

template <class T>
class NameContainer
{
protected:
  vector<T> _list;
public:
  static const size_t NOTFOUND = 0xFFFFFFFF;

  virtual ~NameContainer()
  {
  }

  T* operator[](size_t id)
  {
    return get(id);
  }

  T* operator[](const char* name)
  {
    return get(getId(name));
  }

  virtual T* add(void)
  {
    _list.resize(_list.size() + 1);
    T &i = _list.back();

    return &i;
  }

  T* get(size_t id)
  {
    if (id >= _list.size()) return NULL;

    return &_list[id];
  }

  virtual size_t getId(const char *name)
  {
    for (tag_type i = 0; i < _list.size(); i++)
    {
      if (_list[i].getName() == name) return i;
    }

    return NOTFOUND;
  }

  bool has(const char *name)
  {
    return getId(name) != NOTFOUND;
  }

  bool has(size_t id)
  {
    return id != NOTFOUND;
  }

  size_t size(void)
  {
    return _list.size();
  }

  void resize(size_t size)
  {
    _list.resize(size);
  }

  void clear(void)
  {
    _list.clear();
  }
};

template <class T>
class UNameContainer : public NameContainer<T>
{
public:
  size_t add(const char *name)
  {
    size_t id = NameContainer<T>::getId(name);

    if (id != NameContainer<T>::NOTFOUND)
    {
      return id;
    }
    id = NameContainer<T>::size();
    T &i = *NameContainer<T>::add();
    i.setName(name);

    return id;
  }
};

class TagType : public UNameItem
{
protected:
  XML_TAGRTN (*_func)(Parameter&, XMLParser&);
  XML_TAGRTN (*_funcClose)(Parameter&, XMLParser&);
  vector<ParameterOption> _params;

public:
  tag_type id;
  const char *parent;

  TagType(void)
  {
    _func = NULL;
    _funcClose = NULL;
  }

  ParameterOption* addParameter(void)
  {
    _params.resize(_params.size() + 1);
    return &_params.back();
  }

  ParameterOption* addParameter(size_t index, param_type type, const char *name = NULL)
  {
    ParameterOption &po = *addParameter();
    po.set(index, type, name);

    return &po;
  }

  ParameterOption* getParameter(size_t id)
  {
    if (id >= _params.size()) return NULL;

    return &_params[id];
  }

  size_t countParameter(void)
  {
    return _params.size();
  }
 
  void setFunction(XML_TAGRTN (*f)(Parameter&, XMLParser&), XML_TAGRTN (*fc)(Parameter&, XMLParser&) = NULL)
  {
    _func = f;
    _funcClose = fc;
  }

  bool hasFunction(void)
  {
    return _func != NULL;
  }

  bool hasCloseFunction(void)
  {
    return _funcClose != NULL;
  }

  XML_TAGRTN (*getFunction(void))(Parameter&, XMLParser&)
  {
    return _func;
  }

  XML_TAGRTN (*getCloseFunction(void))(Parameter&, XMLParser&)
  {
    return _funcClose;
  }
};

class TagTypeContainer : public UNameContainer<TagType>
{
public:
  TagType* add(void)
  {
    TagType &i = *NameContainer<TagType>::add();
    i.id = _list.size() - 1;

    return &i;
  }

  TagType* add(const char *name,  const char *parent = NULL)
  {
    size_t id = UNameContainer<TagType>::add(name);
    TagType &i = *UNameContainer<TagType>::get(id);
    i.id = id;
    i.parent = parent;

    return &i;
  }
};

struct ParameterType
{
  XMLParameter* (*func)(const char*, XMLParser&);
};

class XMLParser : public TagTypeContainer
{
protected:
  TagTypeContainer _tagTypeContainer;
  vector<ParameterType> _parameterTypes;
  XMLNode* _currentTag;
  XMLNode* _parentTag;

  static PARAMFUNC(paramString);

public:
  const param_type PARAM_STRING;

  XMLNode *rootNode;
  list<XMLElement*> tagLevelList;

  XMLParser(void);
  virtual ~XMLParser();
  void clearNode(void);

  param_type registerParameterType(XMLParameter* (*func)(const char*, XMLParser&));

  virtual bool loadText(const char *text) = 0;
  virtual bool loadFile(const char *file) = 0;

  TagType* addTagType(const char *name,  const char *parent = NULL)
  {
    return _tagTypeContainer.add(name, parent);
  }
  TagType* getTagType(const char *name) { return _tagTypeContainer[name]; }
  tag_type getTagTypeId(const char *name) { return _tagTypeContainer.getId(name); }
  size_t countTagType(void) { return _tagTypeContainer.size(); };
  bool checkTagLevel(TagType*);

  void setCurrentTag(XMLNode *t) { _currentTag = t; }
  XMLNode* getCurrentTag(void) { return _currentTag; }
  void setParentTag(XMLNode *t) { _parentTag = t; }
  XMLNode* getParentTag(void) { return _parentTag; }
};

#endif // __XMLPARSER_H__
