/**
 * ======================================================================
 * GT-720F device driver (Version 0.0.1)
 *
 * Reference documents:
 * Application Note AN0003
 * Binary Messages Of SkyTraq Venus 5 GPS Receiver
 * Ver 0.3 Feb. 1, 2007
 * SkyTraq Technology, Inc.
 * ======================================================================
 * Copyright (c) 2011 Shinichiro Nakamura (CuBeatSystems)
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ======================================================================
 */

#ifndef GT_720F_H
#define GT_720F_H

/**
 * @brief GT-720Fハンドラの実装。
 */
typedef struct {
    /**
     * @brief リード関数。
     *
     * @param buf 読み込み対象データ。
     * @param siz 読み込みデータバイト数。
     *
     * @retval 0 成功。
     * @retval !0 失敗。
     */
    int (*func_read)(unsigned char *buf, int siz);

    /**
     * @brief ライト関数。
     *
     * @param buf 書き込み対象データ。
     * @param siz 書き込みデータバイト数。
     *
     * @retval 0 成功。
     * @retval !0 失敗。
     */
    int (*func_write)(unsigned char *buf, int siz);
} gt_720f_t;

/**
 * @brief セットアップを実行する。
 *
 * @param p ハンドラ。
 * @param WRITE ライト関数。
 * @param READ リード関数。
 */
#define GT_720F_SETUP(P, WRITE, READ) \
    do { \
        (P)->func_write = (WRITE); \
        (P)->func_read = (READ); \
    } while (0)

/**
 * @brief Start mode.
 */
typedef enum {
    Gt720fStartModeNochange = 0,
    Gt720fStartModeHotStart = 1,
    Gt720fStartModeWarmStart = 2,
    Gt720fStartModeColdStart = 3,
    Gt720fStartModeTest = 4,
} Gt720fStartMode;

/**
 * @brief SYSTEM RESTART - Force system to restart (0x1)
 *
 * @details
 * This is a request message which will reset and restart the GPS receiver.
 * This command is issued from the host to GPS receiver and GPS receiver
 * should respond with an ACK or NACK.
 *
 * The payload length is 15 bytes.
 */
int gt_720f_system_restart(
        gt_720f_t *p,
        Gt720fStartMode start_mode,
        unsigned short year,
        unsigned char month,
        unsigned char day,
        unsigned char hour,
        unsigned char minute,
        unsigned char second,
        short latitude,
        short longitude,
        short altitude);

/**
 * @brief QUERY SOFTWARE VERSION
 *
 * @details
 * Query revision information of loaded software (0x2)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to retrieve loaded software version. The GPS receiver should respond
 * with an ACK along with information on software version when succeeded
 * and should respond with an NACK when failed.
 *
 * The payload length is 2 bytes.
 */
int gt_720f_query_software_version(gt_720f_t *p);

/**
 * @brief QUERY SOFTWARE CRC
 *
 * @details
 * Query CRC information of loaded software (0x3)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to retrieve loaded software CRC. The GPS receiver should respond with an
 * ACK along with information on software version when succeeded and should
 * respond with an NACK when failed.
 *
 * The payload length is 2 bytes.
 */
int gt_720f_query_software_crc(gt_720f_t *p);

/**
 * SET FACTORY DEFAULTS - Set the system to factory default values (0x4)
 *
 * This is a request message which is issued from the host to GPS receiver. It will reset the GPS receiver’s internal
 * parameters to factory default values. The GPS receiver should respond with an ACK when succeeded and should
 * respond with an NACK when failed. The user data will be erased and filled with factory default values. The payload
 * length is 2 bytes.
 */
int gt_720f_set_factory_defaults(gt_720f_t *p);

typedef enum {
    Gt720fBaudRate4800 = 0,
    Gt720fBaudRate9600 = 1,
    Gt720fBaudRate19200 = 2,
    Gt720fBaudRate38400 = 3,
    Gt720fBaudRate57600 = 4,
    Gt720fBaudRate11520 = 5,
} Gt720fBaudRate;

/**
 * @brief CONFIGURE SERIAL PORT
 *
 * @details
 * Set up serial port property (0x5)
 *
 * This is a request message which will configure the serial COM port,
 * baud rate. This command is issued from the host to GPS receiver and GPS
 * receiver should respond with an ACK or NACK.
 *
 * The payload length is 4 bytes.
 */
int gt_720f_configure_serial_port(
        gt_720f_t *p,
        Gt720fBaudRate baud_rate);

/**
 * @brief CONFIGURE NMEA MESSAGE
 *
 * @details
 * Configure NMEA message interval (0x8)
 *
 * This is a request message which will set NMEA message configuration.
 * This command is issued from the host to GPS receiver and GPS receiver
 * should respond with an ACK or NACK.
 *
 * The payload length is 9 bytes.
 */
int gt_720f_configure_nmea_message_interval(
        gt_720f_t *p,
        unsigned char gga,
        unsigned char gsa,
        unsigned char gsv,
        unsigned char gll,
        unsigned char rmc,
        unsigned char vtg,
        unsigned char zda);

/**
 * @brief メッセージタイプ。
 */
typedef enum {
    MessageTypeNoOutput = 0,    /**< 出力しない。 */
    MessageTypeNMEA = 1,        /**< NMEA。 */
    MessageTypeBinary = 2,      /**< バイナリ。 */
} MessageType;

/**
 * @brief CONFIGURE MESSAGE TYPE
 *
 * @details
 * Configure and select output message type (0x9)
 *
 * This is a request message which will change the GPS receiver output
 * message type. This command is issued from the host to GPS receiver and
 * GPS receiver should respond with an ACK or NACK.
 *
 * The payload length is 3 bytes.
 */
int gt_720f_configure_message_type(
        gt_720f_t *p,
        MessageType type);

/**
 * @brief CONFIGURE SYSTEM POWER MODE
 *
 * @details
 * Set the power mode of GPS system (0xC)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to configure the system power mode. The GSP receiver should respond with
 * an ACK when succeeded and should respond with an NACK when failed.
 *
 * The payload length is 3 bytes.
 */
int gt_720f_configure_system_power_mode(
        gt_720f_t *p,
        );

/**
 * @brief CONFIGURE SYSTEM POSITION RATE
 *
 * @details
 * Configure the position update rate of GPS system (0xE)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to configure the system position update rate.
 * Receivers with position rate 4 or higher needs to configure baud rate to
 * 38400 or higher value.The GSP receiver should respond with an ACK when
 * succeeded and should respond with an NACK when failed.
 *
 * The payload length is 3 bytes.
 */

/**
 * @brief QUERY POSITION UPDATE RATE
 *
 * @details
 * Query the position update rate of GPS system (0x10)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to query position update rate. The GSP receiver should respond with an
 * ACK along with information on software version when succeeded and should
 * respond with an NACK when failed.
 *
 * The payload length is 1 byte.
 */

/**
 * @brief Get Ephemeris
 *
 * @details
 * Get ephemeris used of firmware (0x30)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to retrieve ephemeris data. The GSP receiver should respond with an ACK
 * along with information on ephemeris when succeeded and should respond
 * with an NACK when failed.
 *
 * The payload length is 2 bytes.
 */

/**
 * @brief Set Ephemeris
 *
 * @details
 * Set ephemeris to GPS firmware (0x31)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to set ephemeris data (open an ephemeris file) to GPS receiver.
 * The GSP receiver should respond with an ACK when succeeded and should
 * respond with an NACK when failed.
 *
 * The payload length is 87 bytes.
 */

/**
 * @brief CONFIGURE WAAS
 *
 * @details
 * Configure the enable or disable of WAAS (0x37)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to enable or disable WAAS. The GSP receiver should respond with an ACK
 * when succeeded and should respond with an NACK when failed.
 *
 * The payload length is 3 bytes.
 */

/**
 * @brief QUERY WAAS STATUS
 *
 * @details
 * Query WAAS status of GPS receiver (0x38)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to query WAAS status. The GSP receiver should respond with an ACK along
 * with AGPS aiding status when succeeded and should respond with an NACK
 * when failed.
 *
 * The payload length is 1 bytes.
 */

/**
 * @brief CONFIGURE NAVIGATION MODE
 *
 * @details
 * Configure the navigation mode of GPS system (0x3c)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to configure the system navigation mode.
 * The GSP receiver should respond with an ACK when succeeded and should
 * respond with an NACK when failed.
 *
 * The payload length is 3 bytes.
 */

/**
 * @brief QUERY NAVIGATION MODE
 *
 * @details
 * Query the navigation mode of GPS receiver (0x3d)
 *
 * This is a request message which is issued from the host to GPS receiver
 * to query navigation mode.
 * The GSP receiver should respond with an ACK along with navigation mode
 * when succeeded and should respond with an NACK when failed.
 *
 * The payload length is 1 bytes.
 */

#endif

