/**
 * @file task_display.c
 * @author Shinichiro Nakamura
 * @brief ディスプレイタスクの実装。
 */

/*
 * ===============================================================
 *  LPCXpresso Clock
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <t_syslog.h>
#include <t_stdlib.h>
#include <itron.h>
#include "kernel_cfg.h"
#include "lpcxclk.h"
#include "lpcxhw.h"
#include "task_display.h"

#define DISPLAY_CMD_UPDATE  (0) /**< 表示更新コマンド。 */

/**
 * @brief ディスプレイタスクの更新コマンドに対するパラメータ。
 */
typedef struct {
    uint8_t h10;
    uint8_t h01;
    uint8_t m10;
    uint8_t m01;
    uint8_t s10;
    uint8_t s01;
    uint8_t col;
} display_param_update_t;

/**
 * @brief ワーク用変数。
 */
typedef struct {
    lpcxclk_t lpcxclk;
} display_work_t;

static display_work_t work; /**< ワーク用。 */

/**
 * @brief 更新コマンドの実行。
 * @param param パラメータ。
 */
void cmd_update(display_param_update_t *param)
{
    lpcxclk_7seg_write(
            &(work.lpcxclk),
            param->h10,
            param->h01,
            param->m10,
            param->m01,
            param->s10,
            param->s01,
            param->col);
}

/**
 * @brief Display task.
 */
void task_display(intptr_t exinf)
{
    syslog(LOG_NOTICE, "Display task initialized.");

    /*
     * ハードウェアをオープンする。
     */
    lpcxhw_open();
    lpcxclk_open(&(work.lpcxclk));

    {
        /*
         * 輝度設定を初期化する。
         */
        int i;
        for (i = 0; i < 8; i++) {
            lpcxclk_7seg_brightness(&(work.lpcxclk), i, 7);
        }
        lpcxclk_7seg_write(
                &(work.lpcxclk),
                SEGSEL_G,
                SEGSEL_G,
                SEGSEL_G,
                SEGSEL_G,
                SEGSEL_G,
                SEGSEL_G,
                0);
    }

    display_msg_t *p;
    while(1) {
        /*
         * メールボックスの状態を観察し、
         * メッセージがあれば処理を実行する。
         */
        if (prcv_mbx(MBX_DISPLAY, (T_MSG **)&p) == E_OK) {
            uint8_t cmd = ((display_msg_t *)p)->cmd;
            void *param = ((display_msg_t *)p)->param;
            switch (cmd) {
                case DISPLAY_CMD_UPDATE:
                    cmd_update((display_param_update_t *)param);
                    break;
                default:
                    syslog(LOG_NOTICE, "Unknown command 0x%x.", cmd);
                    break;
            }
            rel_mpf(MPF_DISPLAY, (VP)p);
        }
        lpcxclk_7seg_clock(&(work.lpcxclk));
        tslp_tsk(2);
    }

    lpcxhw_close();
    lpcxclk_close(&(work.lpcxclk));

    syslog(LOG_NOTICE, "Display task terminated.");
    ext_ker();
    assert(0);
}

/**
 * @brief ディスプレイの更新を要求する。
 * @details この関数はディスプレイタスクにリクエストを要求したいタスクが呼び出す為のものである。
 */
void tskapi_display_update(
        uint8_t h10,
        uint8_t h01,
        uint8_t m10,
        uint8_t m01,
        uint8_t s10,
        uint8_t s01,
        uint8_t col)
{
    VP vp;

    /*
     * メモリプールからバッファを取得する。
     */
    get_mpf(MPF_DISPLAY, &vp);

    /*
     * コマンドとパラメータを設定する。
     */
    ((display_msg_t *)vp)->cmd = DISPLAY_CMD_UPDATE;
    display_param_update_t *param =
        (display_param_update_t *)&(((display_msg_t *)vp)->param);
    param->h10 = h10;
    param->h01 = h01;
    param->m10 = m10;
    param->m01 = m01;
    param->s10 = s10;
    param->s01 = s01;
    param->col = col;

    /*
     * メールボックスに送信する。
     */
    snd_mbx(MBX_DISPLAY, vp);
}

