/*
COController.m

Author: Makoto Kinoshita

Copyright 2005 HMDT. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY HMDT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL HMDT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "COController.h"
#import "COPrintPanelController.h"

extern NSString*    COURLString;
extern float        COCompressionFactor;
extern NSString*    COFormatName;
extern NSString*    COOutputFileName;
extern int          COScreenWidth;
extern int          COScreenHeight;
extern float        COOutputSizeRatio;
extern int          COOutputSizeWidth;
extern int          COOutputSizeHeight;
extern BOOL         COIsVerboseMode;
extern BOOL         COIsDialogMode;
extern float        COTimerTime;

extern NSArray*     COFormats;

@interface WebPreferences (backgrounds)
- (void)setShouldPrintBackgrounds:(BOOL)flag;
@end

@implementation COController

- (void)_quit
{
    // Quit application
    [NSApp sendAction:@selector(terminate:) to:nil from:self];
}

- (void)awakeFromNib
{
    // Configure GUI
    if (COScreenWidth > 0 && COScreenHeight > 0) {
        [_window setContentSize:NSMakeSize(COScreenWidth, COScreenHeight)];
    }
    [[[_webView mainFrame] frameView] setAllowsScrolling:NO];
    [[_webView preferences] setShouldPrintBackgrounds:YES];
    
    // Register notification
    [[NSNotificationCenter defaultCenter] 
            addObserver:self 
            selector:@selector(webViewProgressEstimateChanged:) 
            name:WebViewProgressEstimateChangedNotification 
            object:nil];
    [[NSNotificationCenter defaultCenter] 
            addObserver:self 
            selector:@selector(webViewProgressFinished:) 
            name:WebViewProgressFinishedNotification 
            object:nil];
    
    // Check URL string
    if (![COURLString hasPrefix:@"http://"] && 
        ![COURLString hasPrefix:@"https://"] && 
        ![COURLString hasPrefix:@"file://"])
    {
        // For absolute file path
        if ([COURLString hasPrefix:@"/"]) {
            COURLString = [@"file://" stringByAppendingString:COURLString];
        }
        else {
            COURLString = [@"http://" stringByAppendingString:COURLString];
        }
    }
       
    // Load URL
    NSURL*  URL;
    URL = [NSURL URLWithString:COURLString];
    if (!URL) {
        // Error
        fprintf(stderr, "coral: Wrong URL string\n");
        [self _quit];
        return;
    }
    if (COIsVerboseMode) {
        fprintf(stderr, "coral: Start loading %s\n", [COURLString cString]);
    }
    [[_webView mainFrame] loadRequest:[NSURLRequest requestWithURL:URL]];
}

- (void)webViewProgressEstimateChanged:(NSNotification*)notification
{
    // For verbose mode
    if (COIsVerboseMode) {
        // Get estimated progress
        double  estimate;
        estimate = [_webView estimatedProgress];
        
        // Print estimated progress
        if (estimate < 1.0) {
            fprintf(stderr, "coral: %d%% done...\n", (int)(estimate * 100));
        }
        else {
            fprintf(stderr, "coral: 100%% done\n");
        }
    }
}

- (void)webViewProgressFinished:(NSNotification*)notification
{
    // For dialog mode
    if (COIsDialogMode) {
        [self performSelector:@selector(showPrintDialog:) withObject:self afterDelay:0.0];
        return;
    }
    
    // For self timer
    if (COTimerTime > 0.0) {
        // For verbose mode
        if (COIsVerboseMode) {
            fprintf(stderr, "coral: Take a shot after %.1f seconds...\n", COTimerTime);
        }
        
        [self performSelector:@selector(writeFile:) withObject:self afterDelay:COTimerTime];
        return;
    }
    
    // Write file
    [self performSelector:@selector(writeFile:) withObject:self afterDelay:0.0];
}

- (NSData*)_createPDFData
{
    // Get document view
    NSView*     documentView;
    documentView = [[[_webView mainFrame] frameView] documentView];
    if (!documentView) {
        // Error
        return nil;
    }
    
    // Get PDF data
    NSData* data;
    //data = [documentView dataWithPDFInsideRect:[documentView frame]];
    NSRect  frame;
    frame = [documentView frame];
    frame.size.height *= 2;
    data = [documentView dataWithPDFInsideRect:frame];
    if (!data || [data length] == 0) {
        // Error
        return nil;
    }
    
    return data;
}

- (NSData*)_createBitmapData
{
    // Draw image
    NSBitmapImageRep*   imageRep;
    [_webView lockFocus];
    imageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect:[_webView frame]];
    [_webView unlockFocus];
    
    // Decide output size
    NSSize  size;
    size = [imageRep size];
    if (COOutputSizeRatio > 0.0) {
        size.width *= COOutputSizeRatio;
        size.height *= COOutputSizeRatio;
    }
    else if (COOutputSizeWidth > 0 && COOutputSizeHeight > 0) {
        size.width = COOutputSizeWidth;
        size.height = COOutputSizeHeight;
    }
    
    // Create image
    NSImage*    image;
    image = [[NSImage alloc] initWithSize:size];
    [image autorelease];
    [image lockFocus];
    [imageRep drawInRect:NSMakeRect(0, 0, size.width, size.height)];
    [image unlockFocus];
    
    // Decide format file type
    NSBitmapImageFileType   imageFileType = NSJPEGFileType;
    if ([COFormatName isEqualToString:@"jpg"] || 
        [COFormatName isEqualToString:@"JPG"] || 
        [COFormatName isEqualToString:@"jpeg"] || 
        [COFormatName isEqualToString:@"JPEG"])
    {
        imageFileType = NSJPEGFileType;
    }
    else if ([COFormatName isEqualToString:@"png"] || 
             [COFormatName isEqualToString:@"PNG"])
    {
        imageFileType = NSPNGFileType;
    }
    else if ([COFormatName isEqualToString:@"gif"] || 
             [COFormatName isEqualToString:@"GIF"])
    {
        imageFileType = NSGIFFileType;
    }
    else if ([COFormatName isEqualToString:@"tif"] || 
             [COFormatName isEqualToString:@"TIF"] || 
             [COFormatName isEqualToString:@"tiff"] || 
             [COFormatName isEqualToString:@"TIFF"])
    {
        imageFileType = NSTIFFFileType;
    }
    
    // Create bitmap data
    NSDictionary*   properties;
    NSData*         data;
    properties = [NSDictionary dictionaryWithObject:[NSNumber numberWithFloat:COCompressionFactor] 
            forKey:NSImageCompressionFactor];
    data = [[NSBitmapImageRep imageRepWithData:[image TIFFRepresentation]] 
            representationUsingType:imageFileType properties:properties];
    
    return data;
}

- (void)writeFile:(id)sender
{
    // Decide format
    if (!COFormatName) {
        // Use extension of file name
        COFormatName = [COOutputFileName pathExtension];
        if (![COFormats containsObject:COFormatName]) {
            // Default is JPG
            COFormatName = @"jpg";
        }
    }
    
    // Decide output file name
    if (!COOutputFileName) {
        // Get page title
        NSString*   title;
        title = [[[_webView mainFrame] dataSource] pageTitle];
        if (title) {
            COOutputFileName = [title stringByAppendingPathExtension:COFormatName];
        }
        
        // Default name
        if (!COOutputFileName) {
            COOutputFileName = [@"untitled" stringByAppendingPathExtension:COFormatName];
        }
    }
    
    // Create data
    NSData* data;
    if ([COFormatName isEqualToString:@"pdf"] || 
        [COFormatName isEqualToString:@"PDF"])
    {
        data = [self _createPDFData];
    }
    else {
        data = [self _createBitmapData];
    }
    if (!data) {
        // Error
        fprintf(stderr, "coral: Failed to get image data\n");
        [self _quit];
        return;
    }
    
    // Write data to file
    NSString*   path;
    if ([COOutputFileName isAbsolutePath]) {
        path = COOutputFileName;
    }
    else {
        path = [[NSFileManager defaultManager] currentDirectoryPath];
        path = [path stringByAppendingPathComponent:COOutputFileName];
    }
    if (![data writeToFile:path atomically:YES]) {
        // Error
        fprintf(stderr, "coral: Failed to save data\n");
        [self _quit];
        return;
    }
    
    // Quit application
    [self _quit];
}

- (void)showPrintDialog:(id)sender
{
    // Create print info
    NSPrintInfo*    printInfo;
    printInfo = [[NSPrintInfo alloc] init];
    [printInfo autorelease];
    [printInfo setHorizontalPagination:NSFitPagination];
    [printInfo setVerticallyCentered:NO];
    
    // Load print panel
    COPrintPanelController* printPanelController;
    printPanelController = [[COPrintPanelController alloc] init];
    [printPanelController setWebView:_webView];
    
    // Active itself
    [NSApp activateIgnoringOtherApps:YES];
    
    // Run print operation
    NSPrintOperation*   operation;
    operation = [NSPrintOperation 
            printOperationWithView:[[[_webView mainFrame] frameView] documentView] printInfo:printInfo];
    [operation setAccessoryView:[printPanelController view]];
    [operation runOperation];
    
    [printPanelController release];
    
    [self _quit];
}

@end
