/*
 * hal_uart.h
 *
 *  Created on: 28/04/2022
 *      Author: <alexraynepe196@gmail.com>
 * ----------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 * ----------------------------------------------------------------------
 * Renesas implementation of HAL UART for proto-threads
 */

#ifndef BSP_CPU_DEV_HAL_UART_H_
#define BSP_CPU_DEV_HAL_UART_H_

//#include <halos_types.h>
#include <mcu_usart.h>
#include <OsProcess.h>

#include "r_sci_uartx.h"



//============================================================================================

// TODO: migrate -> DevResult
typedef fsp_err_t   UARTError;
enum {
    UARTERR_OK  = 0,
};



//============================================================================================

enum {
    /* uart_event_t :
    UART_EVENT_RX_COMPLETE   ,  //  = (1UL << 0), ///< Receive complete event
    UART_EVENT_TX_COMPLETE   ,  //  = (1UL << 1), ///< Transmit complete event
    UART_EVENT_RX_CHAR       ,  //  = (1UL << 2), ///< Character received
    UART_EVENT_ERR_PARITY    ,  //  = (1UL << 3), ///< Parity error event
    UART_EVENT_ERR_FRAMING   ,  //  = (1UL << 4), ///< Mode fault error event
    UART_EVENT_ERR_OVERFLOW  ,  //  = (1UL << 5), ///< FIFO Overflow error event
    UART_EVENT_BREAK_DETECT  ,  //  = (1UL << 6), ///< Break detect error event
    UART_EVENT_TX_DATA_EMPTY ,  //  = (1UL << 7), ///< Last byte is transmitting, ready for more data
*/
    UART_EVENT_ERR_ANY  =   UART_EVENT_ERR_PARITY
                        | UART_EVENT_ERR_FRAMING
                        | UART_EVENT_ERR_OVERFLOW
                        | UART_EVENT_BREAK_DETECT
                        ,
    UART_EVENT_TX_ANY   = UART_EVENT_TX_DATA_EMPTY | UART_EVENT_TX_COMPLETE,
    UART_EVENT_RX_ANY   = UART_EVENT_RX_CHAR       | UART_EVENT_RX_COMPLETE,
};

// @sa @sa uart_event_t
typedef int UARTIOEvent;

typedef void (* UARTEventHandle)(UARTIOEvent event, void const* p_context);

typedef struct UART_Device{
    uart_ctrl_t*    port;

    //process context for io ISR wake
    void*           tx_ctx;
    void*           rx_ctx;

    void*           rx_dma; /* transfer_info_t* */


    UARTEventHandle iohandle;
} UART_Device;


//============================================================================================
UARTError UART_init(UART_Device* uart, uart_ctrl_t*    port);

static inline
UARTError UART_close(UART_Device* uart){
    return R_SCI_UART_Close(uart->port);
}

UARTError UART_read(UART_Device* uart, void* p_dest, unsigned bytes);
UARTError UART_write(UART_Device* uart, const void* p_src, unsigned bytes);

/// @brief receive data pointer
/// @return != NULL - pointer to receiver buffer head
///         == NULL - read not active
uint8_t*  UART_rx_tail(UART_Device* uart);


static inline
UARTError UART_baud_set(UART_Device* uart, unsigned baud){
    sci_uart_instance_ctrl_t* self = (sci_uart_instance_ctrl_t*)(uart->port);
    return USART_set_baud(self->p_reg, baud);
}

static inline
UARTError UART_abort(UART_Device* uart, uart_dir_t communication_to_abort){
    return R_SCI_UART_Abort(uart->port, communication_to_abort);
}


/// @brief check that R_SCI_UART_Read active
static inline
bool UART_rx_busy(UART_Device* uart){
    return R_SCI_UART_rx_busy(uart->port);
}

/// @brief check that R_SCI_UART_Write active
static inline
bool UART_tx_busy(UART_Device* uart){
    return R_SCI_UART_tx_busy(uart->port);
}

/// @brief check that R_SCI_UART_Write buffers empty and io sending inactive
bool UART_flush(UART_Device* uart);

#if SCI_UART_CFG_DTC_SUPPORTED
/// @brief receive data pointer
/// @return != NULL - pointer to receiver buffer head
///         == NULL - read not active or not available
static inline
uint8_t*  UART_rx_head(UART_Device* uart){
    return R_SCI_UART_rx_head(uart->port);
}
#endif




#endif /* BSP_CPU_DEV_HAL_UART_H_ */
