/*
 * mcu_usart.h
 *
 *  Created on: 2/02/2021 г.
 *      Author: alexraynepe196@gmail.com
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 */

#ifndef HAL_RENESAS_RA_MCU_USART_H_
#define HAL_RENESAS_RA_MCU_USART_H_

#include <stdint.h>
#include <mcu-chip.h>

#if !defined(RA_H)
#warning "uncknown HAL, expects renesas RA"
#endif

#include <c_compat.h>
#include "mcu_rcc.h"



#define __UART_CAT2(a, b) a##b
#define _UART_CAT3(a,b,c)  a##b##c

#define UARTn(n)  __UART_CAT2(R_SCI, n)
#define USARTn(n) __UART_CAT2(R_SCI, n)



#define UART_PORT_IDXSTEP   (R_SCI1_BASE - R_SCI0_BASE)
#define UART_PORT_NO(port)  ( ( ((uint8_t*)(port)) - R_SCI0_BASE )/UART_PORT_IDXSTEP)
#define UART_PORTn(n)       ( (R_SCI0_Type *)( R_SCI0_BASE + ((n)*UART_PORT_IDXSTEP) ) )

#define USARTn_TX_IRQ(n)   _UART_CAT3(SCI, n, _TXI_IRQn)
#define USARTn_TE_IRQ(n)   _UART_CAT3(SCI, n, _TEI_IRQn)
#define USARTn_RX_IRQ(n)   _UART_CAT3(SCI, n, _RXI_IRQn)
#define USARTn_ERR_IRQ(n)  _UART_CAT3(SCI, n, _RRI_IRQn)


#define UART_GetFlagStatus( sci )   (sci->SSR)

// transmit complete
#define UART_FLAG_TXC              R_SCI0_SSR_TEND_Msk
#define UART_FLAG_TXE              R_SCI0_SSR_TDRE_Msk



#define USART_PORT_NO(port)         UART_PORT_NO(port)
#define USART_PORTn(n)              UART_PORTn(n)
#define USART_GetFlagStatus( sci )  UART_GetFlagStatus( sci )
#define USART_FLAG_TXC              UART_FLAG_TXC
#define USART_FLAG_TXE              UART_FLAG_TXE



enum USARTCtrl{
    USART_IT_TX         = R_SCI0_SCR_TIE_Msk,
    USART_IT_RX         = R_SCI0_SCR_RIE_Msk,
    USART_TE            = R_SCI0_SCR_TE_Msk,
    USART_RE            = R_SCI0_SCR_RE_Msk,
    USART_IT_MP         = R_SCI0_SCR_MPIE_Msk,
    USART_IT_TXE        = R_SCI0_SCR_TEIE_Msk,      USART_IT_TC         = R_SCI0_SCR_TEIE_Msk,
    USART_CKE           = R_SCI0_SCR_CKE_Msk,
};

#ifdef __cplusplus
extern "C" {
#endif



typedef R_SCI0_Type USART_TypeDef;

/// @brief evaluates and assign <uart> configuration from baud
///         takes to account port Clock rate, and uart->SEMR_b.ABCS state
int USART_set_baud(USART_TypeDef* uart, unsigned baud);


#define RCC_EnableClock_UART( port )     RCC_EnableClock_UARTn( UART_PORT_NO(port) )
#define RCC_DisableClock_UART( port )    RCC_DisableClock_UARTn( UART_PORT_NO(port) )




enum {
    USART_CPHA        = R_SCI0_SPMR_CKPH_Msk,
    USART_PHASE_2EDGE = 0,
    USART_PHASE_1EDGE = USART_CPHA,

    USART_POLARITY        = R_SCI0_SPMR_CKPOL_Msk,
    USART_CPOL            = USART_POLARITY,
    USART_POLARITY_LOW    = 0,
    USART_POLARITY_HIGH   = USART_POLARITY,

    USART_MODE_Msk        = USART_CPHA | USART_CPOL,
};

#define USART_DATA_MODE0           (USART_PHASE_1EDGE |USART_POLARITY_LOW)
#define USART_DATA_MODE1           (USART_PHASE_2EDGE |USART_POLARITY_LOW)
#define USART_DATA_MODE2           (USART_PHASE_1EDGE |USART_POLARITY_HIGH)
#define USART_DATA_MODE3           (USART_PHASE_2EDGE |USART_POLARITY_HIGH)



__STATIC_INLINE
void USART_SetDataMode(USART_TypeDef *SPIx, uint8_t mode){
    SPIx->SPMR = (uint8_t) ((SPIx->SPMR & (uint8_t)~USART_MODE_Msk) | (mode /* & SPI_MODE_Msk*/))  ;
}


enum {
    USART_LSB_FIRST  = 0 ,
    USART_MSB_FIRST  = R_SCI0_SCMR_SDIR_Msk ,
};

__STATIC_INLINE
void USART_SetTransferBitOrder(USART_TypeDef* port, uint8_t mode){
    if (mode == USART_LSB_FIRST)
        port->SCMR_b.SDIR = 0;
    else
        port->SCMR_b.SDIR = 1;
}

__STATIC_INLINE
unsigned USART_TransferBitOrderLSB(USART_TypeDef* port){
    return (port->SCMR_b.SDIR == 0)? USART_LSB_FIRST :USART_MSB_FIRST ;
}



//********************************************************************************************
//              HAL API
#define __HAL_USART_GET_IT( uart, itf)      ((uart)->SCR & (itf))
#define __HAL_USART_ENABLE_IT( uart, itf)   do{ (uart)->SCR |= (itf)); } while(0)
#define __HAL_USART_DISABLE_IT( uart, itf)  do{ (uart)->SCR &= (uint8_t)~(itf); } while(0)


//********************************************************************************************
//              LL API

__STATIC_INLINE
bool USART_IsEnabledIT_RXNE(USART_TypeDef* port){
    return __HAL_USART_GET_IT( port, USART_IT_RX) != 0;
}

__STATIC_INLINE
bool USART_IsEnabledIT_TXE(USART_TypeDef* port){
    return __HAL_USART_GET_IT( port, USART_IT_TX) != 0;
}

__STATIC_INLINE
bool USART_IsEnabledIT_TC(USART_TypeDef* port){
    return __HAL_USART_GET_IT( port, USART_IT_TC) != 0;
}

__STATIC_INLINE
void USART_DisableIT_RXNE(USART_TypeDef* port){
    __HAL_USART_DISABLE_IT( port, USART_IT_RX);
}



#ifdef __cplusplus
}
#endif

#endif /* HAL_MDR32F9QX_MCU_USART_H_ */
