/*
 * eeprom_m95m04.c
 *
 *  Created on: 3/06/2021
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
      STM M95M04 SPI eeprom driver
 */

#include <c_compat.h>
#include "eeprom_m95m04.h"
#include <OsTime.h>


/////////////////////////////////////////////////////////////////////////////
#include <trace_probes.h>

#ifndef trace_flash_rd
trace_need(flash_wr)
trace_need(flash_wrwe)
trace_need(flash_rd)
#endif



/////////////////////////////////////////////////////////////////////////////

const struct FlashChipDescribe   info_m95m04 = {
    struct_field(nb_sectors)          M95M04_SECTORS ,
    struct_field(nb_pages_in_sector)  M95M04_SEC_PAGES,
    struct_field(page_size)           M95M04_PAGE_SIZE ,
    struct_field(burn_page_ticks)     (FLASH_toYIELD*2) +CLOCKMS_MORE(5) ,
    struct_field(erase_ticks)         CLOCKMS_MORE(5) ,
    struct_field(erase_all_ticks)     CLOCKMS_MORE(5) ,
};

DevResult m95m04_bind(FLASH_Device* this){
    flash_assign_info(this, &info_m95m04);
    return DEV_OK;
}


DevResult m95m04_init(SPIFLASH_M95M04* this){
    wre_eepromspi_init(this);
    this->flash.bind    = m95m04_bind;

    this->flash.id      = NULL;
    m95m04_bind(&this->flash);

    sspio_set_mode(&this->ssp.io, sspio_mode_default_8bit);
    sspdev_style(&this->ssp, SSPDEV_addr3BYTE | SSPDEV_addrLE);

    return DEV_OK;
}

void m95m04_connect(SPIFLASH_M95M04* this, SSP_IOPort*    _port){
    sspio_connect(&this->ssp.io, _port, sspio_mode_default_8bit, SSP_speedKEEP);
}


DevResult m95m04_id_read(SPIFLASH_M95M04* self, unsigned addr, void* dst, unsigned len){
    SPIFlash_GenDevice* this = (SPIFlash_GenDevice*)self;

    trace_flash_rd_on();
    int readen = sspdev_readCmdData(&this->ssp, M95_CMD_RDID, addr, dst, len);
    trace_flash_rd_off();

    if (readen >= 0)
        return (readen == (int)len)? DEV_OK:DEV_NOK;
    else
        return readen;
}

PTResult  m95m04_id_write(SPIFLASH_M95M04* self, unsigned addr, const void* src, unsigned len){
    SPIFlash_GenDevice* this = (SPIFlash_GenDevice*)self;

    flashspi_force_busy(this);

    // flash burn starts after CS disabled
    sspio_cs_hold(&this->ssp.io, false);

    trace_flash_wr_on();
    flash_busy_start(&self->flash);
    int writen = sspdev_writeCmdData(&this->ssp, M95_CMD_WRID, addr, (const void*)src, len);
    trace_flash_wr_off();

    if (writen >= 0)
        return (writen == (int)len)? DEV_OK:DEV_NOK;
    else
        return writen;
}

PTResult  m95m04_id_lock(SPIFLASH_M95M04* this){
    flashspi_force_busy(this);

    trace_flash_wr_on();
    flash_busy_start(&this->flash);
    SSPResult ok = sspdev_writeCmdByte(&this->ssp, M95_CMD_LID, M95M04_PAGE_SIZE*2, 1);
    trace_flash_wr_off();

    return ok;
}

bool      m95m04_id_locked(SPIFLASH_M95M04* self){
    unsigned st;

    trace_flash_rd_on();
    SSPResult ok = sspdev_readCmdByte(&self->ssp, M95_CMD_RDID, M95M04_PAGE_SIZE*2, &st);
    trace_flash_rd_off();

    if (ok < 0)
        return false;
    return (st & 1) != 0;
}


