/*
  *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
  реализация протокола SLIP.

 * */
#ifndef HAL_SLIP_H
#define HAL_SLIP_H

#include "hal_device.h"
#include "lib/hal_ringbuffer.h"
#include "uart_hal.h"
#include "net/slip-filter.hpp"

#include <OsSync.h>
#include <OsTime.h>

// SLIP_Device - выделяет фреймы SLIP из входного потока байт от io, и декодирует
//      его фильтром rx<SLIPBufFilter>.
//      см wait_frame, leave_frame
class SLIP_Device
    : public Power_Manager
    //, public HAL_OUTBlock_Device
    , public StdOUT_Device
{
public:
    typedef UART_BUFDevice          io_t;
    typedef SLIPBufFilter           slip_t;
    typedef SLIPBufFilter::buffer_t buffer_t;
    SLIP_Device(io_t* _io);

    // shoud invoke on io internals change
    void init();

    io_t*   io() {return _io;};
    virtual int setPower(POWER_STATE state) {return DEV_NOT_IMPLEMENTED;};

public:
    //------------  SLIP receive proxy      ------------------
    //  входной поток копит фрейм и декодирует прямо на буфере io->rx
    //  декодированный фрейм доступен через виртуальный кольцевой буфер см frame()
    //      этот буфер доступен только по чтению
    PTResult  wait_frame(unsigned to = toInfinite);
    // \return frame size
    int         have_frame() const {return rx.have_frame();};
    // checkout buffers, and reports like have_fame
    // \return frame size
    int         refresh_frame();
    // отбрасывает текущий фрейм, и готовит буфер приему нового
    void        leave_frame() {rx.leave_frame();};
    const buffer_t&   frame() const {return rx.frame();};

    // очистка буффера приемника
    void rx_drop();

    //  SLIP send
public:
    // post SLIP END to serial
    bool put_frame();
    // post frame to serial: <END> str <END>
    bool put_frame( const void* str, unsigned len );

public:
    //*******             stdout_device       ***************
    // эта печать скремблирует данные для SLIP,
    //      !!! оформление фреймов автоматически не делается
    //* блокирующая печать
    virtual int putChar(int ch);
    virtual int puts( const char* str);
    //*  \return - длинна отправленного участка
    virtual int putData ( const void* str, unsigned len);
    //*  неблокирующая печать
    virtual int postData ( const void* str, unsigned len);

    // WaitResult использую в функция ожидания для протониток.
    //  \return 0 - на время ожидания
    //  \return < 0 - по таймауту или ошибке
    //  \return > 0 - количество оступных данных
    typedef UART_BUFDevice::WaitResult WaitResult;

    //*  ожидание доступности печати
    static const unsigned TO_Poll = io_t::TO_Poll;
    //*  \return - количество байт возможных для неблокирующей печати
    virtual WaitResult put_wait(unsigned to = TO_Poll);
    //*  почти тоже put_wait, ждет полного опустошения
    virtual WaitResult put_flush(unsigned to = toInfinite){
        return io()->put_flush(to);
    };
    //*  очищает буфер, прерывая текущую отправку
    virtual int put_drop(){
        return io()->put_drop();
    };

    //* монополизация вывода
    //* \arg onoff - захват/освобождение
    //* \return    - состояние захвачн ли вывод
    virtual bool put_access(bool onoff, unsigned to = toInfinite){
        return io()->put_access(onoff, to);
    };

protected:
    io_t*           _io;
    SLIPBufFilter   rx;
    os_timeout_t    wait_to;

    void            on_rx_over(unsigned ch);

    static void rx_over_event(void* arg, UART_BUFDevice* uart, unsigned c);
};




#endif
