using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;
using hardware;

namespace imagefile{
	class Fds : GameImage{
		//public static readonly long MAX_SIZE = (long) (HEADER.Length + Disksystem.Filesystem.SIDE_SIZE * 8);
		static readonly byte [] HEADER = {
			(byte)'F', (byte)'D', (byte)'S', 0x1a, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0
		};
		//---- member and property ----
		Disksystem.Filesystem [] m_side;
		int Sidenum{
			get {return m_side.Length;}
		}
		int m_mdcoffset = 0;
		//---- method ----
		public Fds() : base(mdc5.Script.imagetype.disk)
		{
		}
		bool compare(byte [] a0, byte[] a1, int offset, int length)
		{
			for(int i = 0; i < length; i++){
				int j = offset + i;
				if(a0[j] != a1[j]){
					return false;
				}
			}
			return true;
		}
		int header_check(byte [] data, out int offset)
		{
			//header check
			offset = 0;
			if(compare(HEADER, data, offset, 4) == false){
				return 0;
			}
			offset += 4;
			int side_num = (int) data[offset];
			if((side_num == 0) || (side_num >= 5)){
				return 0;
			}
			offset += 1;
			if(compare(HEADER, data, offset, HEADER.Length - offset) == false){
				return 0;
			}
			//image file size check
			int SIDE_SIZE = Disksystem.Filesystem.SIDE_SIZE;
			if(data.Length != (SIDE_SIZE * side_num + HEADER.Length)){
				return 0;
			}
			offset = HEADER.Length;
			return side_num;
		}
		
		bool load(string imagefilepath, mdc5.DiskScript script)
		{
			byte [] image;
			if(Static.Utility.binary_load(imagefilepath, out image) == false){
				return false;
			}
			int offset;
			int side_num = header_check(image, out offset);
			if(side_num == 0){
				return false;
			}
			m_hash = new Interface.ImageHash();
			m_side = new Disksystem.Filesystem[side_num];
			for(int i = 0; i < side_num; i++){
				if(script == null){
					m_side[i] = new Disksystem.Filesystem(i);
				}else{
					m_side[i] = new Disksystem.Filesystem(i, script);
				}
				if(m_side[i].Load(image, ref offset, m_hash) == false){
					return false;
				}
			}
			m_hash.Final();
			return true;
		}
		
		override public bool Load(string diskimage_name)
		{
			m_name = diskimage_name;
			return this.load(m_name, null);
		}

		override public bool Load(mdc5.Script k)
		{
			mdc5.DiskScript s = (mdc5.DiskScript) k;
			m_name = s.BiosName;
			m_code = s.GameCode;
			return this.load(s.ImageFilename, s);
		}
		override public void HashListAdd(List<string> list)
		{
			foreach(Disksystem.Filesystem t in m_side){
				t.HashListAdd(list);
			}
		}
		override public patchresult PatchManual(RomRecoard.MotorolaSRecoard r, out string log)
		{
			Debug.Assert(r.Valid == true);
			int found_count = 0;
			int side_index = 0, file_index = 0;
			for(int i = 0; i < m_side.Length; i++){
				int t = m_side[i].PatchSearch(r.Address, r.TargetFilename, ref file_index);
				found_count += t;
				if(t != 0){
					side_index = i;
				}
			}
			switch(found_count){
			case 0:
				log = "error: patchable area not found!";
				return patchresult.AREA_NOTFOUND;
			case 1: //正常
				break;
			default:
				log = "error: patchable area too much!";
				return patchresult.AREA_TOOMUCH;
			}
			m_side[side_index].PatchManual(file_index, r, out log);
			return patchresult.OK;
		}
		
		override public patchresult PatchAuto(out string [] log)
		{
			List<string> list = new List<string>();
			foreach(Disksystem.Filesystem d in m_side){
				d.PatchAuto(list);
			}
			log = list.ToArray();
			return patchresult.OK;
		}
		
		public string [] FilesystemShow()
		{
			List<string> list = new List<string>();
			string prefix = String.Format(
				"{0,-8} {1,-11} {2,-13} {3}",
				"name", "region", "address-range", "length"
			);
			list.Add(prefix);
			foreach(Disksystem.Filesystem side in m_side){
				side.FilesystemAdd(list);
			}
			return list.ToArray();
		}
		override public int RawByte()
		{
			int size = 0;
			foreach(Disksystem.Filesystem side in m_side){
				size += side.EmptyOffset;
			}
			return size;
		}
		override public bool SavedataSearch(imagestream.W_Ram workram)
		{
			if(workram.DiskBufferRequest() == false){
				return false;
			}
			foreach(Disksystem.Filesystem side in m_side){
				side.SavedataSearch(workram);
			}
			return true;
		}
		override public bool StreamMake(imagestream.W_Ram workram, out int blocksize, out int streamsize)
		{
			blocksize = 0;
			streamsize = 0;
			foreach(Disksystem.Filesystem side in m_side){
				side.MdcOffset = streamsize;
				side.Encode();
				streamsize += side.EncodedLength();
			}
			return true;
		}
		override public void RomStreamAllocate(ref int offset)
		{
			m_mdcoffset = offset;
			foreach(Disksystem.Filesystem side in m_side){
				side.MdcOffset += offset;
			}
			foreach(Disksystem.Filesystem side in m_side){
				offset += side.EncodedLength();
			}
		}
		override protected void manage_make(imagestream.W_Ram workram, List<byte>manage)
		{
			//0: side_count
			manage.Add((byte) m_side.Length);
			//1: mdc5 offset struct[8]
			int sideoffset = 0;
			foreach(Disksystem.Filesystem side in m_side){
				hardware.mmc5.offset_add(side.MdcOffset, manage);
				sideoffset += 1;
			}
			for(int i = sideoffset; i < 8; i++){
				hardware.mmc5.offset_add(0, manage);
			}
			//0x1a: save_count
			int savefilecount = 0;
			foreach(Disksystem.Filesystem side in m_side){
				side.SavefileCountIncrement(ref savefilecount);
			}
			manage.Add((byte) savefilecount);
			//0x1b: stream struct
			workram.DisksaveManegeWrite(savefilecount, manage);
		}
		override public bool RomStreamWrite(ref int offset, byte [] rom)
		{
			if(offset != m_mdcoffset){
				return false;
			}
			foreach(Disksystem.Filesystem side in m_side){
				side.RomStreamWrite(ref offset, rom);
			}
			return true;
		}
	}
}

namespace imagefile.Disksystem{
	class Filesystem{
		const string SAVEFILE_GUESS = "<guess>";
		public const int SIDE_SIZE = 0xffdc;
		File[] m_file;
		int m_empty_offset = 0;
		int m_mdc_offset;
		Interface.ImageHash m_hash;
		readonly string m_specitied_savefilename, m_sidename;
		string m_guessed_savefilename = "";
		EncodeMap m_encoder;
		
		public int EmptyOffset{
			get{return m_empty_offset;}
		}
		public int MdcOffset{
			get{return m_mdc_offset;}
			set{
				m_mdc_offset = value;
				file_offset_update(value);
			}
		}
		public Filesystem(int sidenum)
		{
			m_specitied_savefilename = SAVEFILE_GUESS;
			m_sidename = sidename_get(sidenum);
		}
		public Filesystem(int sidenum, mdc5.DiskScript script)
		{
			m_specitied_savefilename = script.SavefileNameGet(sidenum);
			m_sidename = sidename_get(sidenum);
			m_encoder = script.EncodeMapGet(sidenum);
		}
		string sidename_get(int side)
		{
			string ret = "disk" + ((side / 2) + 1).ToString() + ".";
			ret += "side" + ((side & 1) == 0 ? "A" : "B") + ".";
			return ret;
		}
		void savefile_guess(File[] file, ref string log)
		{
			File header = null;
			foreach(File t in file){
				if(t.IsEmpty == true){
					break;
				}
				header = t;
			}
			header.SaveFileGuess();
			header.TailFileInfomationGet(ref log);
		}
		Interface.ImageHash hash_calc(File[] file)
		{
			Interface.ImageHash s = new Interface.ImageHash();
			foreach(File t in file){
				t.HashTransform(s);
			}
			return s;
		}
		public bool Load(byte [] data, ref int offset, Interface.ImageHash hash)
		{
			byte [] sidedata = new byte[SIDE_SIZE];
			Array.Copy(data, offset, sidedata, 0, SIDE_SIZE);
			offset += SIDE_SIZE;
			if(set_and_check(sidedata) == false){
				return false;
			}
			if(m_encoder != null){ //scan 時は skip する
				if(m_encoder.HaveStream() == false){
					int filecount = m_file.Length - 2 - 1; //diskheader, fileamount, empty
					m_encoder = new EncodeCbrMulti(filecount);
					int i = 0;
					foreach(File t in m_file){
						t.EncodeSizeAutoset(ref i, m_encoder);
					}
				}
			}
			hash_transform(hash);
			return true;
		}
		
		void hash_transform(Interface.ImageHash s)
		{
			s.Transform(m_hash.Hash);
		}
		bool set_and_check(byte [] data)
		{
			int offset = 0;
			List<File> file_list = new List<File>();
			while(offset < data.Length){
				if(File.file_add(ref offset, data, m_specitied_savefilename, ref m_empty_offset, file_list) == false){
					return false;
				}
			}
			m_file = file_list.ToArray();
			//ファイル2以上は空のディスクイメージでセーブファイルを特定すると落ちるので対策
			if((m_specitied_savefilename == SAVEFILE_GUESS) && (m_file.Length >= 2)){
				savefile_guess(m_file, ref m_guessed_savefilename);
			}
			m_hash = hash_calc(m_file);
			return true;
		}
		public void HashListAdd(List<string> list)
		{
			list.Add(m_sidename + "hash = " + m_hash.ToString());
			if(m_guessed_savefilename != ""){
				list.Add(m_sidename + "savefilename guessed" + m_guessed_savefilename);
			}
			EvaluateDisk e = new EvaluateDisk(m_sidename);
			foreach(File t in m_file){
				t.CompressEvaluate(e);
			}
			e.ListAdd(list);
		}
		public int EncodedLength()
		{
			int length = 0;
			foreach(File t in m_file){
				if(t.IsEmpty == false){
					length += t.Length;
				}
			}
			return length;
		}
		public int PatchSearch(uint address, string filename, ref int data_index)
		{
			int found = 0;
			int i = 0;
			foreach(File t in m_file){
				if(t.IsArea(address ,filename) == true){
					found += 1;
					data_index = i;
				}
				i++;
			}
			return found;
		}
		public void PatchManual(int file_index, RomRecoard.MotorolaSRecoard r, out string log)
		{
			log = m_sidename;
			m_file[file_index].PatchManual(r, ref log);
		}
		public void PatchAuto(List<string> log)
		{
			foreach(File f in m_file){
				f.PatchAuto(m_sidename, log);
			}
		}
		public void RomStreamWrite(ref int offset, byte [] rom)
		{
			foreach(File f in m_file){
				f.Write(ref offset, rom);
			}
		}
		public void SavedataSearch(imagestream.W_Ram workram)
		{
			foreach(File f in m_file){
				f.SavedataSearch(m_sidename, workram);
			}
		}
		void file_offset_update(int mdcoffset)
		{
			int sideoffset = 0; //encode する値も入れる
			foreach(File f in m_file){
				f.OffsetUpdate(mdcoffset, ref sideoffset);
			}
		}
		public void SavefileCountIncrement(ref int count)
		{
			if(m_specitied_savefilename != ""){
				count += 1;
			}
		}
		public void FilesystemAdd(List<string> list)
		{
			list.Add("-- " + m_sidename + " -------------------------");
			foreach(File f in m_file){
				f.FileInfomationAdd(list);
			}
		}
		public void Encode()
		{
			foreach(File f in m_file){
				f.Encode(m_encoder);
			}
		}
	}

	abstract class File{
		protected enum blockid{
			EMPTY = 0, DISK_HEADER,
			FILE_AMOUNT, FILE_HEADER, FILE_DATA
		}
		protected byte [] m_data; //先頭の blockid を抜いた data
		readonly protected blockid m_id;
		readonly protected int m_offset_plain;
		protected int m_offset_encoded;
		public bool IsEmpty{
			get{return m_id == blockid.EMPTY;}
		}
		virtual public int Length{
			get{return 1 + m_data.Length;}
		}
		protected byte [] cut(byte [] data, ref int offset, int length)
		{
			byte[] filedata = new byte[length];
			Array.Copy(data, offset, filedata, 0, length);
			offset += length;
			return filedata;
		}
		protected File(blockid id, byte [] data, ref int offset, int length)
		{
			m_offset_plain = offset;
			m_data = cut(data, ref offset, length);
			m_id = id;
		}
		virtual public void TailFileInfomationGet(ref string log)
		{
			Debug.Assert(false);
		}
		virtual public void FileInfomationAdd(List<string> list)
		{
		}
		virtual public bool IsArea(uint address, string filename)
		{
			return false;
		}
		virtual public void SaveFileGuess()
		{
			Debug.Assert(false);
		}
		virtual public void PatchAuto(string prefix, List<string> log)
		{
		}
		virtual public void PatchManual(RomRecoard.MotorolaSRecoard r, ref string log)
		{
			Debug.Assert(false);
		}
		abstract public void HashTransform(Interface.ImageHash s);
		virtual public void Write(ref int offset, byte [] rom)
		{
			rom[offset++] = (byte) m_id;
			m_data.CopyTo(rom, offset);
			offset += m_data.Length;
		}

		virtual public void SavedataSearch(string sidename, imagestream.W_Ram workram)
		{
			//do nop
		}
		public void OffsetUpdate(int mdcoffset, ref int sideoffset)
		{
			m_offset_encoded = sideoffset + mdcoffset + 1; //1 is id length
			sideoffset += this.Length;
			savedata_offset_update();
		}
		virtual protected void savedata_offset_update()
		{
		}
		virtual public void Encode(EncodeMap encoder)
		{
		}
		virtual public void CompressEvaluate(EvaluateDisk e)
		{
		}
		virtual public void EncodeSizeAutoset(ref int index, EncodeMap encoder)
		{
		}
		static public bool file_add(ref int offset, byte [] data, string specitied_savefilename, ref int empty_offset, List<File> file_list)
		{
			blockid id = (blockid) data[offset];
			offset += 1;
			Debug.Assert(id != blockid.FILE_DATA);
			switch(id){
			case blockid.DISK_HEADER:
				{
					DiskHeader h = new DiskHeader(data, ref offset);
					file_list.Add(h);
				}
				break;
			case blockid.FILE_AMOUNT:
				{
					FileAmount a = new FileAmount(data, ref offset);
					file_list.Add(a);
				}
				break;
			case blockid.FILE_HEADER:
				{
					//filedata.header and data
					FileHeader h = new FileHeader(data, ref offset, specitied_savefilename);
					if(h.DataBlockError == true){
						return false;
					}
					file_list.Add(h);
				}
				break;
			case blockid.EMPTY:
				{
					empty_offset = offset - 1;
					Empty e = new Empty(data, ref offset, Filesystem.SIDE_SIZE - offset);
					if(e.Error == true){
						return false;
					}
					file_list.Add(e);
				}
				break;
			default: //image error
				return false;
			}
			return true;
		}
	}
/*
fds loader からの dumped image は 末尾から0x02-0x80byte程度に不規則な
ゴミデータが入っている場合がある。ゴミを許容するので末尾 0xc0 byte を
check せずに 0 で fill する
*/
	class Empty : File{
		bool m_error;
		public bool Error{
			get{return m_error;}
		}
		public Empty(byte [] data, ref int offset, int length) 
		  :base(blockid.EMPTY, data, ref offset, length)
		{
			int i;
			for(i = 0; i < m_data.Length - 0x100; i++){
				if(m_data[i] != 0){
					m_error = true;
					return;
				}
			}
			for(; i < m_data.Length; i++){
				m_data[i] = 0;
			}
			m_error = false;
		}
		override public void HashTransform(Interface.ImageHash s)
		{
			s.Final(m_data);
		}
		override public void Write(ref int offset, byte [] rom)
		{
			//do nop!!
		}
	}
/*
SIZE CONTENTS
14   FC Disk String  "**NINTENDO-HVC**"
1   Manufacture Code  
4   Game Name Code
1   Game Version Number
1   Side Number  0: Side-A  1: Side-B
1   Disk Number
1   Err.9  (Ext Disk Indicate No)
1   Err.10 (Ext Disk Indicate No)
1   Boot Read File Code  
   Specify ??? code read on boot
5   Unknown
3   Manufacture Permit Date(???)  
   Recorded in BCD, in the year of "showa"(+1925)
10   Unknown
3   Created Date  
   Recorded in BCD, in the year of "showa"(+1925)
9   Unknown*/
	/*
	disk個別で内容が異なるデータが含まれているのでcheckは行わない。
	imagefile 生成ツールにおいてはここの情報を欠落させる。
	*/
	class DiskHeader : File{
		public DiskHeader(byte [] data, ref int offset) 
			:base(blockid.DISK_HEADER, data, ref offset, 0x38 - 1)
		{
		}
		override public void HashTransform(Interface.ImageHash s)
		{
			//個別情報なので算出範囲外
		}
	}
/* 
SIZE CONTENTS
1   File Amount*/
	/*
	コピーツール対策にファイル数が一致しないものがあるので無効な情報とみなす。よって check はおこなわない。
	*/
	class FileAmount :File{
		public FileAmount(byte [] data, ref int offset) 
			:base(blockid.FILE_AMOUNT, data, ref offset, 2 - 1)
		{
		}
		override public void HashTransform(Interface.ImageHash s)
		{
			//あてにならないので算出範囲外
		}
	}
/*
SIZE CONTENTS
1   File Number
1   File Indicate Code (file identification code)
 ID specified at disk-read function call
8   File Name
2   File Address
 the destination address when loading
2   File Size
1   Kind of File  
   0:Program (CPU $0000-$07ff or $6000-$dfff)
   1:Character(PPU $0000-$1fff)
   2:Name table(PPU $2000-$2fff)
 The destination is shown.*/
	class FileHeader :File{
		enum region{
			CPU_DATA, PPU_PATTERN, PPU_NAME
		};
		public const string PATCHFILE_AUTO = "<auto>";
		readonly byte [] m_name_byte = new byte[8];
		readonly string m_name_string;
		readonly uint m_address;
		readonly int m_size;
		readonly region m_region;
		bool m_savedata = false;
		readonly bool m_content_error = false;
		readonly FileData m_content;
		SaveArgument m_saveargment;
		public bool DataBlockError{
			get {return m_content_error;}
		}
		override public int Length{
			get{return 1 + m_data.Length + m_content.EncodedLength();}
		}
		public FileHeader(byte [] data, ref int dataoffset, string savefilename) 
		  :base(blockid.FILE_HEADER, data, ref dataoffset, 0x10 - 1)
		{
			int offset = 1+1;
			m_name_byte = cut(m_data, ref offset, 8);
			m_name_string = name_set(m_name_byte);
			m_address = (uint) m6502.word_get(m_data, ref offset);
			m_size = m6502.word_get(m_data, ref offset);
			m_region = (region) m_data[offset];
			if(m_name_string == savefilename){
				m_savedata = true;
			}
			
			blockid dataid = (blockid) data[dataoffset++];
			if(dataid == blockid.FILE_DATA){
				m_content = new FileData(data, ref dataoffset, m_size);
			}else{
				m_content_error = true;
			}
		}
		override public void Write(ref int offset, byte [] rom)
		{
			rom[offset++] = (byte) m_id;
			rom[offset++] = m_data[0];
			rom[offset++] = m_data[1];
			m_name_byte.CopyTo(rom, offset);
			offset += m_name_byte.Length;
			m6502.word_set(m_address, rom, ref offset);
			m6502.word_set((uint) (m_content.EncodedLength() - 1), rom, ref offset);
			rom[offset++] = (byte) m_region;
			m_content.ContainerWrite(ref offset, rom);
		}
		override public void HashTransform(Interface.ImageHash s)
		{
			s.Transform(m_data);
			if(m_savedata == false){
				m_content.HashTransform(s);
			}
		}
		string name_set(byte [] bytename)
		{
			string s = "";
			bool nullfound = false;
			foreach(byte t in bytename){
				if(t == 0){
					nullfound = true;
				}else if(
					(t >= 0x20 && t < 0x7f) &&
					(nullfound == false)
				){
					//1文字だけ変換する method がわからない...
					byte [] au = new byte[2];
					au[0] = t;
					au[1] = 0;
					s += BitConverter.ToChar(au, 0);
				}else {
					s += String.Format("${0:x2}", t);
				}
			}
			if(s == ""){
				s = "$00";
			}
			return s.TrimEnd();
		}
		override public bool IsArea(uint address, string filename)
		{
			if(m_region != region.CPU_DATA){
				return false;
			}
			if((filename == PATCHFILE_AUTO) || (m_name_string == filename)){
				uint start = m_address;
				uint end = m_address + (uint)(m_size) - 1;
				if(address >= start && address <= end){
					return true;
				}
			}
			return false;
		}

		override public void SaveFileGuess()
		{
			if((m_region == region.CPU_DATA) && (m_size < 0x3000)){
				m_savedata = true;
			}
		}

		override public void SavedataSearch(string sidename, imagestream.W_Ram workram)
		{
			if(m_savedata == false){
				return;
			}
			m_saveargment = new SaveArgument();
			m_saveargment.Name = m_name_string;
			m_saveargment.Length = m_content.Length - 1; //id をぬく
			m_saveargment.Side = sidename;
			//m_saveargment.ImageOffset = m_offset_plain;
			workram.DiskSaveRequest(m_saveargment);
		}
		override protected void savedata_offset_update()
		{
			if(m_savedata == true){
				//workam にある Queue の ImageOffset を更新する
				m_saveargment.ImageOffset = m_data.Length + 1 + m_offset_encoded;
			}
		}
		override public void PatchManual(RomRecoard.MotorolaSRecoard r, ref string log)
		{
			log += m_name_string + " ";
			r.TargetOffset = m_address;
			m_content.PatchManual(r, ref log);
		}
		override public void PatchAuto(string prefix, List<string> log)
		{
			if(m_region != region.CPU_DATA){
				return;
			}
			if(m_address < 0x6000){
				return;
			}
			m_content.PatchAuto(m_address, prefix + m_name_string + " ", log);
		}
		override public void TailFileInfomationGet(ref string log)
		{
			log += String.Format(" {0} #0x{1:x4} byte", m_name_string, m_size);
		}
		override public void FileInfomationAdd(List<string> list)
		{
			string t = String.Format(
				"{0,-8} {1,-11} 0x{2:x4}-0x{3:x4} 0x{4:x4}", 
				m_name_string, m_region,
				m_address, m_address + m_size - 1, m_size
			);
			list.Add(t);
		}
		override public void Encode(EncodeMap encoder)
		{
			m_content.Encode(encoder, m_savedata);
		}
		override public void CompressEvaluate(EvaluateDisk e)
		{
			//save file 推測したものは圧縮する可能性があるので全て評価する
			m_content.CompressEvaluate(e);
		}
		override public void EncodeSizeAutoset(ref int index, EncodeMap encoder)
		{
			bool t;
			if(m_size < 0x20){
				t = encoder.BuffersizeSet(index++, EncodeCbrMulti.RAW);
			}else{
				t = encoder.BuffersizeSet(index++, 0x800);
			}
			Debug.Assert(t);
		}
	}
/* 
SIZE CONTENTS
1   File Amount*/
	class FileData :File{
		const int BLOCK_SIZE = 0x0800;
		EncodeMap m_encoder;
		int m_encodedlength;
		public FileData(byte [] data, ref int offset, int length) 
		  :base(blockid.FILE_DATA, data, ref offset, length)
		{
		}
		override public void HashTransform(Interface.ImageHash s)
		{
			byte[] t = new byte [1 + m_data.Length];
			t[0] = (byte) m_id;
			m_data.CopyTo(t, 1);
			s.Transform(t);
		}
		override public void PatchManual(RomRecoard.MotorolaSRecoard r, ref string log)
		{
			GameImage.PatchMemory(r, ref m_data, "0x{0:x4}", ref log);
		}
		void patching(uint jsr_address, string jsr_name, uint mapped_address, ref int offset, string prefix, List<string> log)
		{
			if(rp2c33.control(m_data, offset + 1) == true){
				string au = prefix + String.Format("0x{0:x4}/${1:x4}: jsr {2} ", offset, mapped_address + offset, jsr_name);
				log.Add(au);
				m6502.jsr_set(jsr_address, ref m_data, ref offset);
			}else{
				offset += 1;
			}
		}
		override public void PatchAuto(string prefix, List<string> log)
		{
			Debug.Assert(false, "fileheader から呼び出されるべき");
		}
		public void PatchAuto(uint mapped_address, string prefix, List<string> log)
		{
			int offset = 0;
			// - 2 は 3byte 連続参照で m_data の範囲を超えないようにするため
			while(offset < m_data.Length - m6502.POINTER_SIZE){
				uint jsr;
				string name;
				if(m6502.patch_store(m_data[offset], out jsr, out name) == true){
					patching(jsr, name, mapped_address, ref offset, prefix, log);
				}else{
					offset += 1;
				}
			}
		}
		public void Encode(EncodeMap encoder, bool forceraw)
		{
			m_encodedlength = encoder.Encode(m_data, forceraw, 0);
			m_encoder = encoder;
		}
		public int EncodedLength()
		{
			return 1 + m_encodedlength; //id
		}
		public void ContainerWrite(ref int offset, byte [] rom)
		{
			rom[offset++] = (byte) m_id;
			m_encoder.Write(ref offset, rom);
		}
		override public void CompressEvaluate(EvaluateDisk e)
		{
			e.CompressEvaluate(m_data);
		}
	}
}

