#
# Cookbook Name:: docker-grid
# Recipe:: engine
#
# Copyright 2016, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

# https://dcos.io/docs/1.8/administration/installing/custom/system-requirements/

bash 'systemctl_daemon-reload' do
  code <<-EOH
    systemctl daemon-reload
  EOH
  action :nothing
end

if node['docker-grid']['engine']['storage-driver'] == 'overlay'
  bash 'load_overlayfs_kernel_module' do
    code <<-EOH
      modprobe overlay
    EOH
    not_if 'lsmod | grep overlay'
  end

  template '/etc/modules-load.d/overlay.conf' do
    source  'etc/modules-load.d/overlay.conf'
    owner 'root'
    group 'root'
    mode '0644'
  end
end

directory '/etc/systemd/system/docker.service.d' do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
end

template '/etc/systemd/system/docker.service.d/override.conf' do
  source  'etc/systemd/system/docker.service.d/override.conf'
  owner 'root'
  group 'root'
  mode '0644'
  notifies :run, 'bash[systemctl_daemon-reload]'
  notifies :restart, 'service[docker]'
end

case node['platform_family']
when 'rhel'
  # https://dcos.io/docs/1.8/administration/installing/custom/system-requirements/install-docker-centos/
  template '/etc/yum.repos.d/docker.repo' do
    source  'etc/yum.repos.d/docker.repo'
    owner 'root'
    group 'root'
    mode '0644'
  end

  [
    'docker-engine-selinux',
    'docker-engine',
  ].each {|pkg|
    package pkg do
      action :install
      version node['docker-grid']['engine']['version']
      # dockerrepo is disabled by default to prevent automatic update.
      options '--enablerepo=dockerrepo'
    end
  }
when 'debian'
  # https://docs.docker.com/engine/installation/linux/ubuntulinux/
  pkgs = [
    'apt-transport-https',
    'ca-certificates',
  ]

  if node['docker-grid']['engine']['storage-driver'] == 'aufs'
    pkgs += [
      "linux-image-extra-#{node['os_version']}",
      'linux-image-extra-virtual',
    ]
  end

  pkgs.each {|pkg|
    package pkg do
      action :install
    end
  }

  template '/etc/apt/sources.list.d/docker.list' do
    source  'etc/apt/sources.list.d/docker.list'
    owner 'root'
    group 'root'
    mode '0644'
  end

  apt_repo_config = node['docker-grid']['apt_repo']
  bash 'apt-key_adv_docker_tools_key' do
    code <<-"EOH"
      apt-key adv --keyserver #{apt_repo_config['keyserver']} --recv-keys #{apt_repo_config['recv-keys']}
      apt-get update
    EOH
    not_if 'apt-key list | grep -i docker'
  end

  # Pinning Docker version
  template '/etc/apt/preferences.d/docker.pref' do
    source  'etc/apt/preferences.d/docker.pref'
    owner 'root'
    group 'root'
    mode '0644'
  end

  [
    'docker-engine',
  ].each {|pkg|
    package pkg do
      action :install
      version node['docker-grid']['engine']['version']
    end
  }
end

service 'docker' do
  action [:start, :enable]
end
