#
# Cookbook Name:: hadoop
# Recipe:: default
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

users = {
  :hadoop => {:name => 'hadoop', :uid => 10001},
  :hdfs   => {:name => 'hdfs',   :uid => 10002},
  :mapred => {:name => 'mapred', :uid => 10003}
}

users.each {|key, user|
if key != :hadoop
  group user[:name] do
    gid user[:uid]
    members []
    action :create
  end

  user user[:name] do
    uid user[:uid]
    gid user[:uid]
    home "/home/#{user[:name]}"
    shell '/bin/sh'
    password nil
    supports :manage_home => false
  end
end
}

group users[:hadoop][:name] do
  gid users[:hadoop][:uid]
  members ['hdfs', 'mapred']
  append true
  action :create
end

user users[:hadoop][:name] do
  uid users[:hadoop][:uid]
  gid users[:hadoop][:uid]
  home "/home/#{users[:hadoop][:name]}"
  shell '/bin/sh'
  password nil
  supports :manage_home => false
end

directory node['grid']['app_root'] do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

active_vol_nums = 0
node['grid']['max_vol_nums'].to_i.times {|vol_num|
  target_vol_dir = "#{node['grid']['vol_root']}/#{vol_num}"

  if vol_num == 0 || FileTest::directory?(target_vol_dir) then
    directory "#{target_vol_dir}/var" do
      owner 'root'
      group 'root'
      mode '0755'
      action :create
      recursive true
    end

    %w{lib log}.each {|dir|
      directory "#{target_vol_dir}/var/#{dir}" do
        owner 'root'
        group 'hadoop'
        mode '0775'
        action :create
        recursive true
      end
    }

    directory "#{target_vol_dir}/tmp" do
      owner 'root'
      group 'root'
      mode '1777'
      action :create
      recursive true
    end

    if vol_num == 0 then
      directory "#{target_vol_dir}/var/run" do
        owner 'root'
        group 'hadoop'
        mode '0755'
        action :create
        recursive true
      end

      directory "#{target_vol_dir}/var/log/hdfs" do
        owner 'hdfs'
        group 'hdfs'
        mode '0755'
        action :create
        recursive true
      end
    end
  else
    break
  end
  active_vol_nums = vol_num + 1
}
log "This node active volumes: #{active_vol_nums}"

hadoop_tarball = "hadoop-#{node['hadoop']['version']}.tar.gz"
remote_file "#{Chef::Config[:file_cache_path]}/#{hadoop_tarball}" do
  source "#{node['hadoop']['archive_url']}/hadoop-#{node['hadoop']['version']}/#{hadoop_tarball}"
  action :create_if_missing
end

bash 'install_hadoop' do
  code <<-EOC
    tar xvzf "#{Chef::Config[:file_cache_path]}/#{hadoop_tarball}" -C "#{node['grid']['app_root']}"
  EOC
  creates "#{node['grid']['app_root']}/hadoop-#{node['hadoop']['version']}"
end
=begin
link '/grid/usr/hadoop' do
  action :delete
  only_if 'test -L /grid/usr/hadoop'
end
=end
link "#{node['grid']['app_root']}/hadoop" do
  to "#{node['grid']['app_root']}/hadoop-#{node['hadoop']['version']}"
end

conf_files = [
  'capacity-scheduler.xml',
  'configuration.xsl',
  'core-site.xml',
  'fair-scheduler.xml',
  'hadoop-env.sh',
  'hadoop-metrics2.properties',
  'hadoop-policy.xml',
  'hdfs-site.xml',
  'hosts.include',
  'hosts.exclude',
  'log4j.properties',
  'mapred-queue-acls.xml',
  'mapred-site.xml',
  'masters',
  'slaves',
  'ssl-client.xml.example',
  'ssl-server.xml.example'
]

conf_files.each {|conf_file|
  template "#{node['grid']['app_root']}/hadoop-#{node['hadoop']['version']}/conf/#{conf_file}" do
    source "conf/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0644'
    variables({
      :active_vol_nums => active_vol_nums
    })
  end
}

# with security
if node['hadoop']['with_security'] then

directory node['hadoop']['this.keytab.dir'] do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

template "#{node['grid']['app_root']}/hadoop-#{node['hadoop']['version']}/conf/taskcontroller.cfg" do
  source "conf/taskcontroller.cfg"
  owner 'root'
  group 'root'
  mode '0400'
end

package 'jsvc' do
  action :install
end

end

log <<-EOM
Note:
You must initialize HDFS in the first installation:
  $ cd #{node['grid']['app_root']}/hadoop
  $ sudo -u hdfs ./bin/hadoop namenode -format
  $ sudo -u hdfs ./bin/hadoop-daemon.sh start namenode
  $ sudo -u hdfs ./bin/hadoop-daemon.sh start datanode
  $ sudo -u hdfs ./bin/hadoop fs -chown hdfs:hdfs /
  $ sudo -u hdfs ./bin/hadoop fs -chmod 755 /
  $ sudo -u hdfs ./bin/hadoop fs -mkdir /user
  $ sudo -u hdfs ./bin/hadoop fs -mkdir #{node['grid']['vol_root']}/0/var/lib/mapred
  $ sudo -u hdfs ./bin/hadoop fs -chown mapred:mapred #{node['grid']['vol_root']}/0/var/lib/mapred
EOM
log <<-EOM
Note:
Example MapReduce job execution:
  $ sudo adduser alice
  $ sudo -u hdfs ./bin/hadoop fs -mkdir /user/alice
  $ sudo -u hdfs ./bin/hadoop fs -chown alice:alice /user/alice
  $ sudo -u alice ./bin/hadoop jar hadoop-examples-#{node['hadoop']['version']}.jar pi 5 10
EOM

