#
# Cookbook Name:: nagios
# Recipe:: default
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

conf_dir = nil
nagios_service = nil
httpd_service = nil
cluster_name = node['nagios']['cluster_name']
cluster_conf_dir = nil

case node[:platform_family]
  when 'debian'
    conf_dir = '/etc/nagios3'
    cluster_conf_dir = "#{conf_dir}/#{cluster_name}"

    %w{nagios3 nagios-plugins nagios-images}.each {|pkg|
      package pkg do
        action :install
      end
    }

    nagios_service = 'nagios3'
    service nagios_service do
      action [:enable]
      supports :status => true, :restart => true, :reload => true
    end

    %w{nagios.cfg cgi.cfg}.each {|conf_file|
      template "/etc/nagios3/#{conf_file}" do
        source "etc/nagios3/#{conf_file}"
        owner 'root'
        group 'root'
        mode '0644'
        variables({
          'cfg_dir' => cluster_conf_dir
        })
        notifies :restart, "service[#{nagios_service}]"
      end
    }

    httpd_service = 'apache2'
    resources(:service => httpd_service) rescue service httpd_service do
      action [:enable]
      supports :status => true, :restart => true, :reload => true
    end

    if node['nagios']['web']['AuthType'] == 'Kerberos' then
      pkg = 'libapache2-mod-auth-kerb'
      resources(:package => pkg) rescue package pkg do
        action :install
      end
    end

    template '/etc/nagios3/apache2.conf' do
      source 'etc/nagios3/apache2.conf'
      owner 'root'
      group 'root'
      mode '0644'
      notifies :restart, "service[#{httpd_service}]"
    end

    if node['nagios']['check_external_commands'] == '1' then
      bash 'modify_permissions_for_check_external_commands' do
        code <<-EOC
          service #{nagios_service} stop
          override='nagios www-data 2710 /var/lib/nagios3/rw'
          dpkg-statoverride --list | grep "$override"
          if [ $? -ne 0 ]; then
            dpkg-statoverride --update --add $override
          fi
          override='nagios nagios 751 /var/lib/nagios3'
          dpkg-statoverride --list | grep "$override"
          if [ $? -ne 0 ]; then
            dpkg-statoverride --update --add $override
          fi
          #service #{nagios_service} start
        EOC
        notifies :restart, "service[#{nagios_service}]"
      end
    end
  when 'rhel'
    conf_dir = '/etc/nagios'
    cluster_conf_dir = "#{conf_dir}/#{cluster_name}"

    %w{nagios nagios-plugins}.each {|pkg|
      package pkg do
        action :install
      end
    }
    %w{load users ping ssh ntp http disk swap procs file_age}.each {|plugin|
      package "nagios-plugins-#{plugin}" do
        action :install
      end
    }

    nagios_service = 'nagios'
    service nagios_service do
      action [:enable]
      supports :status => true, :restart => true, :reload => true
    end

    %w{nagios.cfg cgi.cfg}.each {|conf_file|
      template "/etc/nagios/#{conf_file}" do
        source "etc/nagios/#{conf_file}"
        owner 'root'
        group 'root'
        mode '0644'
        variables({
          'cfg_dir' => cluster_conf_dir
        })
        notifies :restart, "service[#{nagios_service}]"
      end
    }

    httpd_service = 'httpd'
    resources(:service => httpd_service) rescue service httpd_service do
      action [:enable, :start]
      supports :status => true, :restart => true, :reload => true
    end

    if node['nagios']['web']['AuthType'] == 'Kerberos' then
      resources(:package => 'mod_auth_kerb') rescue package 'mod_auth_kerb' do
        action :install
      end
    end

    template '/etc/httpd/conf.d/nagios.conf' do
      source 'etc/httpd/conf.d/nagios.conf'
      owner 'root'
      group 'root'
      mode '0644'
      notifies :restart, "service[#{httpd_service}]"
    end

    if node['nagios']['check_external_commands'] == '1' then
      bash 'modify_permissions_for_check_external_commands' do
        code <<-EOC
          service #{nagios_service} stop
          chown nagios:apache /var/spool/nagios/cmd
          chmod 2710 /var/spool/nagios/cmd
          chmod  751 /var/spool/nagios
          #service #{nagios_service} start
        EOC
        notifies :restart, "service[#{nagios_service}]"
      end
    end
end

if node['nagios']['check_ganglia_metric']['enabled'] then
  pkg = 'python-setuptools'
  resources(:package => pkg) rescue package pkg do
    action :install
  end

  if node.platform_family?('rhel') then
    bash 'easy_install_upgrade_distribute' do
      code <<-EOC
        easy_install -U distribute
      EOC
    end
  end
  
  easy_install_package 'check_ganglia_metric' do
    action :install
  end

  easy_install_package 'NagAconda' do
    version node['nagios']['NagAconda']['version']
    action :install
  end
end

directory cluster_conf_dir do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

cluster_conf_files = [
  'hosts.cfg',
  'services.cfg',
  'contacts.cfg',
  'timeperiods.cfg',
  'checkcommands.cfg',
]

cluster_conf_files.each {|conf_file|
  template "#{cluster_conf_dir}/#{conf_file}" do
    source "etc/nagios/CLUSTER_NAME/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0644'
    notifies :restart, "service[#{nagios_service}]"
  end
}

