#
# Cookbook Name:: bubbleupnp-server
# Recipe:: docker-compose
#
# Copyright 2019-2022, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

doc_url = 'https://github.com/wdstar/docker-bubbleupnpserver'

include_recipe 'platform_utils::kernel_user_namespace'
include_recipe 'docker-grid::compose'

app_dir = node['bubbleupnp-server']['docker-compose']['app_dir']
data_dir = node['bubbleupnp-server']['docker-compose']['data_dir']
dot_env = node['bubbleupnp-server']['docker-compose']['dot_env']

file_owner = 'root'
file_group = 'root'
workdir = '/usr/share/bubbleupnpserver'
if dot_env['VER'] > '0.9-5' || dot_env['VER'] == 'latest'
  file_owner = 'bubbleupnpserver'
  file_group = 'users'
  workdir = '/opt/bubbleupnpserver'
end

[
  app_dir,
  "#{data_dir}/cache",
  "#{data_dir}/Playlists",
].each {|dir|
  resources(directory: dir) rescue directory dir do
    owner 'root'
    group 'root'
    mode '0755'
    recursive true
  end
}

config_srvs = node['bubbleupnp-server']['docker-compose']['config']['services']
override_config_srvs = node.override['bubbleupnp-server']['docker-compose']['config']['services']
force_override_config_srvs = node.force_override['bubbleupnp-server']['docker-compose']['config']['services']
envs = {}
vols = config_srvs['bubbleupnp-server']['volumes'].to_a

# Note: the official image already contains the following volume definitions.
if dot_env['REGISTRY'] != 'bubblesoftapps'
  # bind mounting
  vols.push("#{data_dir}/configuration.xml:#{workdir}/configuration.xml:rw")
  vols.push("#{data_dir}/cache:#{workdir}/cache:rw")
  vols.push("#{data_dir}/Playlists:#{workdir}/Playlists:rw")
end

# merge environment hash
force_override_config_srvs['bubbleupnp-server']['environment'] = envs unless envs.empty?
# reset volumes array.
override_config_srvs['bubbleupnp-server']['volumes'] = vols unless vols.empty?

file "#{data_dir}/configuration.xml" do
  owner 'root'
  group 'root'
  mode '0644'
  action :touch
end

[
  '.env',
  'docker-compose.yml',
].each {|conf_file|
  template "#{app_dir}/#{conf_file}" do
    source "opt/docker-compose/app/bubbleupnp-server/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0644'
  end
}

[
  'backup.sh',
  'restore.sh',
].each {|conf_file|
  template "#{app_dir}/#{conf_file}" do
    source "opt/docker-compose/app/bubbleupnp-server/#{conf_file}"
    owner 'root'
    group 'root'
    mode '0755'
    variables(
      file_owner: file_owner,
      file_group: file_group,
      workdir: workdir
    )
  end
}

dockerfile_tpl = 'Dockerfile'
dockerfile_tpl = 'Dockerfile.armhf' if node['kernel']['machine'] == 'armv7l'
template "#{app_dir}/Dockerfile" do
  source "opt/docker-compose/app/bubbleupnp-server/#{dockerfile_tpl}"
  owner 'root'
  group 'root'
  mode '0644'
  variables(
    workdir: workdir
  )
end

# autopilot
srv = 'bubbleupnp-server'
template "/etc/cron.d/#{srv}-local" do
  source  "etc/cron.d/#{srv}-local"
  owner 'root'
  group 'root'
  mode '0644'
  action :delete unless node['bubbleupnp-server']['docker-compose']['autopilot']['enabled']
end

log 'bubbleupnp-server docker-compose post install message' do
  message <<-"EOM"
Note: You must execute the following command manually.
  See #{doc_url}
  * Start:
    $ cd #{app_dir}
    $ docker-compose up -d
  * Stop
    $ docker-compose down
EOM
end
