#
# Cookbook Name:: docker-grid
# Attributes:: default
#
# Copyright 2016-2018, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

::Chef::Node::Attribute.send(:include, DockerGrid::Helper)

platform = node['platform']

default['docker-grid']['install_flavor'] = 'dockerproject'  # or 'os-repository'
default['docker-grid']['dockerproject']['enable_new_repo'] = true
# read only
force_override['docker-grid']['dockerproject']['apt_new_repo_url'] = "https://download.docker.com/linux/#{platform}"
force_override['docker-grid']['dockerproject']['apt_old_repo_url'] = 'https://apt.dockerproject.org/repo'
# e.g. 'stable edge', 'edge test',...
default['docker-grid']['dockerproject']['apt_new_repo_sections'] = 'stable'
default['docker-grid']['dockerproject']['package_name'] \
  = node['docker-grid']['dockerproject']['enable_new_repo'] ? 'docker-ce' : 'docker-engine'
apt_repo_url = \
  if node['docker-grid']['dockerproject']['enable_new_repo']
    node['docker-grid']['dockerproject']['apt_new_repo_url']
  else
    node['docker-grid']['dockerproject']['apt_old_repo_url']
  end
default['docker-grid']['apt_repo'] = {
  'url' => apt_repo_url,
  'override_apt_line' => '',  # e.g. 'deb https://apt.dockerproject.org/repo ubuntu-xenial main'
  # for old repository
  'keyserver' => 'hkp://p80.pool.sks-keyservers.net:80',
  'recv-keys' => '58118E89F3A912897C070ADBF76221572C52609D',
}
# e.g. 'docker-ce-edge,docker-ce-test'
default['docker-grid']['dockerproject']['yum_new_repo_extra_enablerepo'] = ''
# Old yum repository
default['docker-grid']['yum_repo'] = {
  'baseurl' => 'https://yum.dockerproject.org/repo/main/centos/$releasever/',
  'gpgcheck' => '1',
  'gpgkey' => 'https://yum.dockerproject.org/gpg',
}

default['docker-grid']['compose']['install_flavor'] = 'dockerproject'  # or 'os-repository'
default['docker-grid']['compose']['skip_setup'] = false
# dockerproject: direct download.
# Note: non-support by this cookbook.
#   os-repository (Ubuntu): http://packages.ubuntu.com/search?keywords=docker-compose&searchon=names
#   os-repository (CentOS): none.
default['docker-grid']['compose']['auto_upgrade'] = false
# latest: 'https://github.com/docker/compose/releases/download/1.22.0'
default['docker-grid']['compose']['version'] = '1.21.2'
default['docker-grid']['compose']['release_base_url'] = "https://github.com/docker/compose/releases/download/#{node['docker-grid']['compose']['version']}"
default['docker-grid']['compose']['release_url'] = "#{node['docker-grid']['compose']['release_base_url']}/docker-compose-#{node['kernel']['name']}-#{node['kernel']['machine']}"
default['docker-grid']['compose']['home_dir'] = '/opt/docker-compose'
default['docker-grid']['compose']['app_dir'] = "#{node['docker-grid']['compose']['home_dir']}/app"

default['docker-grid']['dind-compose']['app_dir'] = "#{node['docker-grid']['compose']['app_dir']}/docker-in-docker"
default['docker-grid']['dind-compose']['data_dir'] = "#{node['docker-grid']['dind-compose']['app_dir']}/data"
default['docker-grid']['dind-compose']['config'] = {
  # Version 2 docker-compose format
  'version' => '2',
  'services' => {
    'dind' => {
      'image' => 'docker:stable-dind',
      'privileged' => true,
      'command' => [
        #'--storage-driver=overlay2',  # same as host Docker's storage driver
      ],
      'volumes' => [
        # These volumes will be set by the docker-grid::dind-compose recipe automatically.
        #"#{node['docker-grid']['dind-compose']['data_dir']}:/var/lib/docker",
      ],
      'environment' => {
      },
    },
  },
}

default['docker-grid']['engine']['skip_setup'] = false
default['docker-grid']['engine']['autopilot'] = {
  'enabled' => false,
  'log_dir' => '/var/log',
  'containers_prune_period' => '5  */4 * * *',
  'images_prune_period'     => '10 */4 * * *',
  'volumes_prune_period'    => '15 */4 * * *',
}

# dockerproject: 18.06.0.ce-3, 18.03.1.ce-1, 17.12.1.ce-1, 1.13.1-1
# os-repository: yum list docker
#   http://mirror.centos.org/centos/7.3.1611/extras/x86_64/Packages/
default['docker-grid']['engine']['version_on_centos'] = '17.12.1.ce-1'
# dockerproject: 5:18.09.0~3-0, 18.06.1~ce~3-0, 18.03.1~ce-0, 17.12.1~ce-0, 1.13.1-0
default['docker-grid']['engine']['version_on_debian'] = '17.12.1~ce-0'
# dockerproject: 5:18.09.0~3-0, 18.06.1~ce~3-0, 18.03.1~ce-0, 17.12.1~ce-0
# os-repository: http://packages.ubuntu.com/search?keywords=docker.io&searchon=names
default['docker-grid']['engine']['version_on_ubuntu'] = '17.12.1~ce-0'

# '' (empty) or 'latest' version -> latest version
case platform
when 'centos', 'redhat'
  version_on_centos = node['docker-grid']['engine']['version_on_centos']
  if !version_on_centos.nil? && !version_on_centos.empty? && version_on_centos != 'latest'
    version_on_centos = docker_full_ver(node, version_on_centos)
  end
when 'debian'
  version_on_debian = node['docker-grid']['engine']['version_on_debian']
  if !version_on_debian.nil? && !version_on_debian.empty? && version_on_debian != 'latest' \
    && node['docker-grid']['install_flavor'] == 'dockerproject'
    version_on_debian = docker_full_ver(node, version_on_debian)
  end
when 'ubuntu'
  version_on_ubuntu = node['docker-grid']['engine']['version_on_ubuntu']
  if !version_on_ubuntu.nil? && !version_on_ubuntu.empty? && version_on_ubuntu != 'latest' \
    && node['docker-grid']['install_flavor'] == 'dockerproject'
    version_on_ubuntu = docker_full_ver(node, version_on_ubuntu)
  end
end

# '' (empty) or 'latest' version -> latest version
# Note: if you would OVERRIDE the ['docker-grid']['engine']['version'] attribute ONCE,
#       the ['docker-grid']['engine']['version_on_*'] attributes are IGNORED.
default['docker-grid']['engine']['version'] = node.value_for_platform(
  ['centos', 'redhat'] => {
    'default' => version_on_centos,
  },
  'debian' => {
    'default' => version_on_debian,
  },
  'ubuntu' => {
    'default' => version_on_ubuntu,
  }
)
# overlay2: Docker >= 1.12, Kernel >= 4.0
default['docker-grid']['engine']['storage-driver_on_centos'] = 'overlay'
default['docker-grid']['engine']['storage-driver_on_debian'] = 'overlay2'
default['docker-grid']['engine']['storage-driver_on_ubuntu'] = 'aufs'
default['docker-grid']['engine']['storage-driver'] = node.value_for_platform(
  ['centos', 'redhat'] => {
    'default' => node['docker-grid']['engine']['storage-driver_on_centos'],
  },
  'debian' => {
    'default' => node['docker-grid']['engine']['storage-driver_on_debian'],
  },
  'ubuntu' => {
    'default' => node['docker-grid']['engine']['storage-driver_on_ubuntu'],
  }
)
default['docker-grid']['engine']['userns-remap'] = nil  # default: inactive
# CentOS default: '--selinux-enabled --log-driver=journald --signature-verification=false'
default['docker-grid']['engine']['daemon_extra_options'] = '-H fd://'
default['docker-grid']['engine']['users_allow'] = []

# dockerproject: container image from https://hub.docker.com/
# Note: non-support by this cookbook.
#   os-repository (Ubuntu): http://packages.ubuntu.com/search?keywords=docker-registry&searchon=names
#   os-repository (CentOS): http://mirror.centos.org/centos/7.3.1611/extras/x86_64/Packages/
default['docker-grid']['registry']['with_ssl_cert_cookbook'] = false
# If node['docker-grid']['registry']['with_ssl_cert_cookbook'] is true,
# node['docker-grid']['registry']['docker-compose']['config']
# are overridden by the following 'ca_name' and 'common_name' attributes.
#default['docker-grid']['registry']['ssl_cert']['ca_name'] = nil
default['docker-grid']['registry']['ssl_cert']['common_name'] = node['fqdn']
# See https://docs.docker.com/registry/configuration/
rootdirectory = node.value_for_platform(
  ['centos', 'redhat'] => {
    'default' => '/var/lib/registry',
  },
  ['debian', 'ubuntu'] => {
    'default' => '/var/lib/docker-registry',
  }
)
default['docker-grid']['registry']['server']['autopilot'] = {
  'enabled' => false,
  'log_dir' => '/var/log',
  'wait_time' => '120s',
}
default['docker-grid']['registry']['server']['config'] = {
  'version' => '0.1',
  'log' => {
    'fields' => {
      'service' => 'registry',
    },
  },
  'storage' => {
    'cache' => {
      # NOTE: Formerly, blobdescriptor was known as layerinfo.
      # While these are equivalent, layerinfo has been deprecated.
      'blobdescriptor' => 'inmemory',
    },
    'filesystem' => {
      'rootdirectory' => rootdirectory,
    },
  },
  'http' => {
    'addr' => ':5000',
    'headers' => {
      'X-Content-Type-Options' => [
        'nosniff',
      ],
    },
  },
  #'proxy' => {
  #  'remoteurl' => 'https://registry-1.docker.io',
  #},
  'health' => {
    'storagedriver' => {
      'enabled' => true,
      'interval' => '10s',
      'threshold' => 3,
    },
  },
}

default['docker-grid']['registry']['docker-compose']['autopilot'] = {
  'enabled' => false,
  'log_dir' => '/var/log',
  'wait_time' => '180s',
  'docker-compose_opts' => '--no-ansi',
}
default['docker-grid']['registry']['docker-compose']['app_dir'] = "#{node['docker-grid']['compose']['app_dir']}/registry"
# ./docker-compose.yml
default['docker-grid']['registry']['docker-compose']['config_format_version'] = '1'
default['docker-grid']['registry']['docker-compose']['service_name'] = 'registry'
default['docker-grid']['registry']['docker-compose']['host_data_volume'] = '/var/lib/docker-registry'
service_name = node['docker-grid']['registry']['docker-compose']['service_name']
version_1_config = {
  # Version 1 docker-compose format
  service_name => {
    'restart' => 'always',
    'image' => 'registry:2',
    #'ports' => [
    #  '5000:5000',
    #],
    #'environment' => {
    #  'REGISTRY_HTTP_TLS_CERTIFICATE' => '/certs/domain.crt',
    #  'REGISTRY_HTTP_TLS_KEY' =>         '/certs/domain.key',
    #  'REGISTRY_AUTH' =>                'htpasswd',
    #  'REGISTRY_AUTH_HTPASSWD_PATH' =>  '/auth/htpasswd',
    #  'REGISTRY_AUTH_HTPASSWD_REALM' => 'Registry Realm',
    #},
    #'volumes' => [
    #  #'./etc/config.yml:/etc/docker/registry/config.yml',  # Overriding the entire configuration file
    #  "#{node['docker-grid']['registry']['docker-compose']['host_data_volume']}:/var/lib/registry",
    #  '/path/certs:/certs',
    #  '/path/auth:/auth',
    #],
  },
}
version_2_config = {
  # Version 2 docker-compose format
  'version' => '2',
  'services' => version_1_config,
}
default['docker-grid']['registry']['docker-compose']['config'] = \
  node['docker-grid']['registry']['docker-compose']['config_format_version'] == '2' ? version_2_config : version_1_config
# ./etc/config.yml
# See: https://docs.docker.com/registry/configuration/#/overriding-the-entire-configuration-file
default['docker-grid']['registry']['docker-compose']['registry-config'] = nil
