/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * MICSのエンジンおよびMICSで規定するインターフェースの定義などのパッケージ
 */
package net.wasamon.mics;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;

import net.wasamon.mics.gui.awt.MicsAWTWindow;
import net.wasamon.mics.gui.swing.MicsSwingWindow;
import net.wasamon.mjlib.util.GetOpt;
import net.wasamon.mjlib.util.GetOptException;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * MICSシミュレータのメインクラス
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public class Mics {

	public static final int majorVersion = 0;

	public static final int minorVersion = 5;

	public static final int revision = 4;

	private static Mics instance = new Mics();

	/**
	 * MICSシミュレータのインスタンスを返す。このインスタンスはシングルトン。
	 * 
	 * @return
	 */
	public static Mics getInstance() {
		return instance;
	}

	private Mics() {
	}

	private MicsShell shell;

	private void setMicsShell(MicsShell shell) {
		this.shell = shell;
	}

	/**
	 * MICSシミュレータを操作するインターフェースを返す
	 * 
	 * @return
	 */
	public MicsShell shell() {
		return this.shell;
	}

	private MicsCompositeElement engine = new MicsCompositeElement();

	/**
	 * MICSシミュレータの保持しているトップレベルのシミュレーション集合要素を返す
	 * 
	 * @return
	 */
	public MicsCompositeElement engine() {
		return engine;
	}

	private void init() {
		engine.clear();
	}

	/**
	 * シミュレータ要素として登録されている全エレメントの配列を返す
	 * 
	 * @return
	 */
	public MicsElement[] getMicsElementArray() {
		return engine.getMicsElementArray();
	}

	/**
	 * 指定した識別子のDataBufferに入力ストリームからのデータを書き込む
	 * 
	 * @param id
	 * @param offset
	 * @param reader
	 * @throws MicsException
	 */
	public void setMemoryData(String id, int offset, BufferedInputStream reader)
			throws MicsException {
		DataBuffer r = engine.getDataBuffer(id);
		r.write(offset, reader);
	}

	/**
	 * リセット
	 */
	public void reset() throws MicsException {
		engine.reset();
	}

	/**
	 * シミュレーションスレッドを開始する
	 */
	public void execEngine() {
		new Thread(engine).start();
	}

	/**
	 * シミュレーションの開始。シミュレーションが終了するまでブロックする
	 */
	public void execEngineBlock() {
		engine.run();
	}

	/**
	 * シミュレーションを一時停止する
	 */
	public void stopEngine() {
		engine.stop();
	}

	/**
	 * 指定したidのメモリの内容を表示する
	 */
	public void printMemory(String id, int addr, int length) throws MicsException {
		System.out.println(engine.getDataBuffer(id).toString(addr, length));
	}

	/**
	 * 1ステップだけシミュレーションを実行する
	 */
	public void execStepByStep() throws ExecutableElementException,
			MicsException, DataBufferException {
		execStepByStep(1);
	}

	/**
	 * 指定したステップ数分だけシミュレーションを実行
	 * 
	 * @param count
	 * @throws MicsException
	 * @throws MicsException
	 * @throws DataBufferException
	 */
	public void execStepByStep(int count) throws MicsException, MicsException,
			DataBufferException {
		StringBuffer str = new StringBuffer();
		for (int i = 0; i < count - 1; i++) {
			engine.step();
		}
		str.append(engine.getProcStatusString());
		engine.step();
		System.out.println(str.toString());
	}

	private final String defaultDir = ".";

	/**
	 * MICSシミュレーションの起動ディレクトリを返す。
	 * 
	 * @return
	 */
	public String getDefaultDir() {
		return defaultDir;
	}

	/**
	 * シミュレーション対象のアーキテクチャを設定ファイルから読み込む
	 * 
	 * @param file
	 * @throws MicsException
	 */
	public void configTargetArchitecture(File file) throws MicsException {
		init();
		try {
			Node top = XMLParser.getTopNode(file);
			String base = file.getParent();
			engine.initialize_base(null, base, "engine", top);
			engine.initialize();
		} catch (XMLParserException e) {
			throw new MicsException("configuration error: " + e);
		}
		System.out.println("architecture update as " + file.getName());
	}

	/**
	 * MICSシミュレータを終了させる
	 * 
	 * @param stat
	 */
	public void shutdown(int stat) {
		engine.shutdown();
		System.exit(stat);
	}

	public static String printVersion() {
		return String.valueOf(majorVersion) + "." + String.valueOf(minorVersion)
				+ "." + String.valueOf(revision);
	}

	public static void main(String[] args) {

		GetOpt opt = new GetOpt("f:ge:", "swt", args);
		boolean script = false;
		File scriptSource = null;

		if (opt.flag("swt")) {
			System.out.println("SWT shell has not been suported yet.");
			System.exit(0);
			// Mics.getInstance().setMicsShell(SWTMicsWindow.getInstance());
		} else if (opt.flag("g")) {
			Mics.getInstance().setMicsShell(new MicsPrompt());
		} else if (opt.flag("awt")) {
			Mics.getInstance().setMicsShell(MicsAWTWindow.getInstance());
		} else {
			Mics.getInstance().setMicsShell(MicsSwingWindow.getInstance());
		}

		try {
			if (opt.flag("f")) {
				try {
					Mics.getInstance().configTargetArchitecture(
							new File(opt.getValue("f")));
				} catch (MicsException e) {
					System.out.println("configuration error: " + e);
					System.exit(0);
				}
			}
			if (opt.flag("e")) {
				script = true;
				scriptSource = new File(opt.getValue("e"));
				if (!scriptSource.exists()) {
					System.out.println("usage: net.wasamon.mics.Mics -e script");
					System.exit(0);
				}
			}
		} catch (GetOptException e) {
			System.out.println("usage: net.wasamon.mics.Mics [-f config.xml]");
			System.exit(0);
		}

		try {
			System.out.println("MICS is Cycle-based Simulator");
			System.out.println("Version: " + printVersion());
			System.out
					.println("Copyright (c) 2005-2007 Takefumi MIYOSHI All Rights Reserved.");
			if (!script) {
				Mics.getInstance().shell.exec();
			} else {
				Mics.getInstance().shell.exec(scriptSource);
			}
		} catch (MicsException e) {
			System.err.println(e);
		}
	}

	public void readMemory(String[] cmd) throws MicsException {
		if (cmd.length == 4) {
			Mics.getInstance().printMemory(cmd[1], Integer.parseInt(cmd[2]),
					Integer.parseInt(cmd[3]));
		} else {
			System.out.println("memory id addr length");
		}
	}

	public void loadFile(String[] cmd) throws MicsException {
		try {
			if (cmd.length == 4) {
				Mics.getInstance().setMemoryData(cmd[1], Integer.parseInt(cmd[2]),
						new BufferedInputStream(new FileInputStream(new File(cmd[3]))));
			} else {
				throw new MicsException("load id offset \"file\"");
			}
		} catch (FileNotFoundException e) {
			System.out.println("no such file " + cmd[3]);
		}
	}
}
