/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * Swingを用いてGUIを実装したパッケージ
 */
package net.wasamon.mics.gui.swing;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.Mics;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsViewable;
import net.wasamon.mics.gui.awt.MemoryLoader;

public class ArchitectureList implements TreeSelectionListener, ActionListener {

	private static ArchitectureList instance = new ArchitectureList();

	public static ArchitectureList getInstance() {
		return instance;
	}

	private JFrame frame;

	private JTree tree;

	private ArchitectureList() {
		frame = new JFrame("Architecture Viewer");
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				frame.setVisible(false);
			}
		});
		frame.setBounds(10, 10, 640, 240);
		frame.getContentPane().add(getToolBar(), BorderLayout.NORTH);
		JScrollPane scrPane = new JScrollPane();
		tree = new JTree(makeArchitectureTree());
		tree.addTreeSelectionListener(this);
		scrPane.getViewport().setView(tree);
		frame.getContentPane().add(scrPane, BorderLayout.CENTER);
	}

	private TreeModel makeArchitectureTree() {
		DefaultMutableTreeNode root = new DefaultMutableTreeNode("Mics");
		DefaultMutableTreeNode proc = new DefaultMutableTreeNode(
				"ExecutableElement");
		DefaultMutableTreeNode memory = new DefaultMutableTreeNode("DataBuffer");
		DefaultMutableTreeNode bus = new DefaultMutableTreeNode("Channel");
		DefaultMutableTreeNode cc = new DefaultMutableTreeNode("ChannelConnectable");

		MicsElement[] elements = Mics.getInstance().getMicsElementArray();
		for (int i = 0; i < elements.length; i++) {
			MicsElement element = elements[i];
			String name = element.getClass().getName();
			if (element instanceof DataBuffer) {
				DefaultMutableTreeNode node = new DefaultMutableTreeNode(element.id()
						+ ":" + name);
				memory.add(node);
			}
			if (element instanceof Channel) {
				DefaultMutableTreeNode node = new DefaultMutableTreeNode(element.id()
						+ ":" + name);
				bus.add(node);
			}
			if (element instanceof ExecutableElement) {
				DefaultMutableTreeNode node = new DefaultMutableTreeNode(element.id()
						+ ":" + name);
				proc.add(node);
			}
			if (element instanceof ChannelConnectable) {
				DefaultMutableTreeNode node = new DefaultMutableTreeNode(element.id()
						+ ":" + name);
				cc.add(node);
			}
		}

		root.add(proc);
		root.add(memory);
		root.add(bus);
		root.add(cc);

		return new DefaultTreeModel(root);

	}

	private MicsElement getSelectedElement() {
		TreePath path = tree.getSelectionPath();
		MicsElement element = null;
		if (path == null) {
			return null;
		}
		if (path.getPath().length < 3) {
			return null;
		}
		String value = path.getLastPathComponent().toString();
		String label[] = value.split(":");
		if (label.length < 2) {
			return null;
		}
		try {
			element = Mics.getInstance().engine().getMicsElement(label[0]);
		} catch (MicsException e) {
			return null;
		}
		return element;
	}

	public void valueChanged(TreeSelectionEvent ev) {
		MicsElement element = getSelectedElement();
		if (element != null) {
			System.out.println(element.getInfo());
			if (element instanceof DataBuffer) {
				btnLoad.setEnabled(true);
			} else {
				btnLoad.setEnabled(false);
			}
			if (element instanceof MicsViewable) {
				btnView.setEnabled(true);
			} else {
				btnView.setEnabled(false);
			}
		} else {
			btnView.setEnabled(false);
			btnLoad.setEnabled(false);
		}
	}

	private JButton btnLoad;

	private JButton btnView;

	private JToolBar getToolBar() {
		JToolBar bar = new JToolBar();
		btnLoad = new JButton("Load");
		btnLoad.addActionListener(this);
		bar.add(btnLoad);
		btnView = new JButton("View");
		btnView.addActionListener(this);
		bar.add(btnView);
		return bar;
	}

	public void update() {
		tree.setModel(makeArchitectureTree());
	}

	public void show() {
		update();
		frame.setVisible(true);
	}

	public void actionPerformed(ActionEvent ev) {
		Object src = ev.getSource();
		if (src == null) {
		} else if (src == btnView) {
			MicsElement element = getSelectedElement();
			if (element != null && element instanceof MicsViewable) {
				((MicsViewable) element).show();
			}
		} else if (src == btnLoad) {
			MicsElement element = getSelectedElement();
			if (element != null && element instanceof DataBuffer) {
				MemoryLoader.getInstance().show((DataBuffer) element);
			}
		} else {
		}
	}
}
