/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * Swingを用いてGUIを実装したパッケージ
 */
package net.wasamon.mics.gui.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintStream;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import javax.swing.filechooser.FileFilter;

import net.wasamon.mics.Mics;
import net.wasamon.mics.MicsCompositeElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsPrompt;
import net.wasamon.mics.MicsShell;
import net.wasamon.mics.gui.MicsArchitectureViewer;
import net.wasamon.mics.tools.AsciiToBinary;
import net.wasamon.mjlib.ui.JTextAreaOutputStream;

public class MicsSwingWindow implements MicsShell, ActionListener {

	private static MicsSwingWindow instance = new MicsSwingWindow();

	private JFrame frame;

	private JTextArea log;

	private JTextField input;

	public static MicsSwingWindow getInstance() {
		return instance;
	}

	private void exit(int stat) {
		Mics.getInstance().shutdown(stat);
	}

	private MicsSwingWindow() {
		frame = new JFrame("Mics " + Mics.printVersion());
		frame.setSize(640, 540);
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				exit(0);
			}
		});
		frame.setJMenuBar(getMenuBar());
		frame.getContentPane().add(getToolBar(), BorderLayout.NORTH);
		log = new JTextArea();
		log.setFont(new Font("MonoSpaced", Font.PLAIN, 12));
		log.setEditable(false);
		log.setBackground(Color.WHITE);
		frame.getContentPane().add(new JScrollPane(log), BorderLayout.CENTER);

		PrintStream stream = new PrintStream(new JTextAreaOutputStream(log), true);
		System.setOut(stream);
		System.setErr(stream);

		JTextField prompt = new JTextField("[command]");
		prompt.setEditable(false);
		input = new JTextField();
		input.addActionListener(this);
		JPanel p0 = new JPanel(new BorderLayout());
		p0.add(prompt, BorderLayout.WEST);
		p0.add(input);
		frame.getContentPane().add(p0, BorderLayout.SOUTH);
		frame.setVisible(false);
	}

	private JButton btnStep;

	private JButton btnRun;

	private JButton btnStop;

	private JButton btnReset;

	private JButton btnLoad;

	private JButton btnView;

	private JTextField stepNumber;

	private JButton newJButton(String label, ActionListener listener, boolean flag){
		JButton btn = newJButton(label, listener);
		btn.setEnabled(flag);
		return btn;
	}
	private JButton newJButton(String label, ActionListener listener){
		JButton btn;
		btn = new JButton(label);
		btn.addActionListener(this);
		return btn;
	}
	
	private JPanel getToolBar() {
		JPanel bar = new JPanel();
		{
			JPanel p = new JPanel();
			p.setLayout(new GridLayout(1, 2));
			p.add(btnLoad = newJButton("Load", this));
			p.add(btnView = newJButton("View", this));
			p.setBorder(new TitledBorder(new LineBorder(Color.gray, 1), "Architecture",
					TitledBorder.LEFT, TitledBorder.TOP));
			bar.add(p);
		}
		{
			JPanel p = new JPanel();
			p.setLayout(new GridLayout(1, 3));
			p.add(btnReset = newJButton("Reset", this, false));
			p.add(btnRun = newJButton("Run", this, false));
			p.add(btnStop = newJButton("Stop", this, false));
			p.setBorder(new TitledBorder(new LineBorder(Color.gray, 1), "processor",
					TitledBorder.LEFT, TitledBorder.TOP));
			bar.add(p);
		}
		{
			JPanel p = new JPanel(); 
			p.setLayout(new GridLayout(1, 2));
			stepNumber = new JTextField("1", 3);
			stepNumber.addActionListener(this);
			p.add(stepNumber);
			p.add(btnStep = newJButton("Do", this, false));
			p.setBorder(new TitledBorder(new LineBorder(Color.gray, 1), "step",
					TitledBorder.LEFT, TitledBorder.TOP));
			bar.add(p);
		}
		
		return bar;
	}

	private JMenuItem openMenuItem;

	private JMenuItem viewMenuItem;

	private JMenuItem exitMenuItem;

	private JMenuItem a2bMenuItem;

	private JMenuBar getMenuBar() {
		JMenuBar bar = new JMenuBar();
		{
			JMenu m = new JMenu("File");
			openMenuItem = new JMenuItem("Load");
			openMenuItem.addActionListener(this);
			m.add(openMenuItem);
			viewMenuItem = new JMenuItem("View");
			viewMenuItem.addActionListener(this);
			m.add(viewMenuItem);
			m.addSeparator();
			exitMenuItem = new JMenuItem("Exit");
			exitMenuItem.addActionListener(this);
			m.add(exitMenuItem);
			bar.add(m);
		}
		{
			JMenu m = new JMenu("Tools");
			a2bMenuItem = new JMenuItem("Ascii to Binary");
			a2bMenuItem.addActionListener(this);
			m.add(a2bMenuItem);
			bar.add(m);
		}
		return bar;
	}

	public void exec() {
		frame.setVisible(true);
	}

	/**
	 * バッチ処理といっても、特に何もしない
	 */
	public void exec(File batch) {
		exec();
	}

	public void setMicsEngineEnableState(boolean flag) {
		if (flag) {
			btnStep.setEnabled(true);
			btnReset.setEnabled(true);
			btnRun.setEnabled(true);
			btnStop.setEnabled(false);
		} else {
			btnStep.setEnabled(false);
			btnReset.setEnabled(false);
			btnRun.setEnabled(false);
			btnStop.setEnabled(true);
		}
	}

	public File openFile() {
		File f = null;
		JFileChooser fd = new JFileChooser();
		fd.setFileFilter(new FileFilter() {
			public boolean accept(File arg0) {
				return arg0.isDirectory() || arg0.getName().endsWith(".xml");
			}

			public String getDescription() {
				return "MICS configuration File (.xml)";
			}
		});
		int returnVal = fd.showOpenDialog(frame);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			f = fd.getSelectedFile();
		}
		return f;
	}

	class SimpleProcessorAssemblerFilter implements FilenameFilter {
		public boolean accept(File dir, String name) {
			if (name.endsWith(".s")) {
				return true;
			}
			return false;
		}
	}

	class CoinsInputFilter implements FilenameFilter {
		public boolean accept(File dir, String name) {
			if (name.endsWith(".c")) {
				return true;
			}
			if (name.endsWith(".i")) {
				return true;
			}
			if (name.endsWith(".lir")) {
				return true;
			}
			return false;
		}
	}

	class ConfigDataSourceFilter implements FilenameFilter {
		public boolean accept(File dir, String name) {
			if (name.endsWith(".context")) {
				return true;
			}
			return false;
		}
	}

	public void actionPerformed(ActionEvent ev) {
		Object obj = ev.getSource();
		if (obj == input) {
			try {
				MicsPrompt.execLine(((JTextField) obj).getText());
			} catch (IOException e) {
			}
			((JTextField) obj).setText("");
		} else if (obj == openMenuItem || obj == btnLoad) {
			File f = openFile();
			if (f != null) {
				try {
					Mics.getInstance().configTargetArchitecture(f);
				} catch (Exception e) {
					JOptionPane.showMessageDialog(frame, e + ":" + e.getMessage(),
							"Configuration Error", JOptionPane.ERROR_MESSAGE);
					e.printStackTrace();
				}
				Mics.getInstance().engine().show();
				setMicsEngineEnableState(true);		
			}
		} else if (obj == exitMenuItem) {
			exit(0);
		} else if (obj == a2bMenuItem) {
			File f = openFile();
			if (f != null) {
				try {
					AsciiToBinary asm = new AsciiToBinary();
					File dest = asm.parser(f);
					System.out.println("compile succeeded: " + dest.getAbsolutePath());
				} catch (NumberFormatException e) {
					System.out.println(e.getMessage());
				} catch (FileNotFoundException e) {
					System.out.println("no such file: " + f.getAbsolutePath());
				} catch (IOException e) {
					System.out.println("File I/O error");
				}
			}
		} else if (obj == btnRun) {
			setMicsEngineEnableState(false);
			Mics.getInstance().execEngine();
		} else if (obj == btnStop) {
			Mics.getInstance().stopEngine();
			setMicsEngineEnableState(true);
		} else if (obj == btnStep) {
			try {
				int count = 1;
				try {
					count = Integer.parseInt(stepNumber.getText());
				} catch (NumberFormatException e) {
					count = 1;
					stepNumber.setText("1");
				}
				if (count < 1) {
					count = 1;
					stepNumber.setText("1");
				}
				Mics.getInstance().execStepByStep(count);
			} catch (MicsException e) {
				System.out.println(e);
			}
		} else if (obj == btnReset) {
			try {
				Mics.getInstance().reset();
			} catch (MicsException e) {
				System.out.println(e);
			}
		} else if (obj == viewMenuItem || obj == btnView) {
			if (Mics.getInstance().engine() != null) {
				Mics.getInstance().engine().show();
			}
		} else {
			System.out.println("Uninitialized as architecture config.");
		}
	}

	public MicsArchitectureViewer getMicsArchitectureViewerData(
			MicsCompositeElement composite) {
		return new SwingArchitectureViewer(composite);
	}

}
