/**
 * LPCXPresso1769用の高レベルMiMicAPI
 MiMicCoreの関数を必要とします。
 */
var LPCXpresso1769;

(function(){
	var isUndef=MiMicLib.isUndef;
	
	var DEV=
	{
		/**
		 * エラーIDテーブルの拡張
		 */
		_EE:function(i_base){
		return{
			NG:[i_base|0x00,"Unknown exception in LPCXpresso1769"],
			INVALID_ARG :[i_base|0x01,"Invalid argument"],
			INVALID_PIN :[i_base|0x02,"Invalid pin"],
			INVALID_CFG :[i_base|0x03,"Invalid configulation"],
			VM_RUNTIME  :[i_base|0x04,"MVM runtime error"],
		};}(MiMicError.NG[0]|MiMicError.MID_MiMic|MiMicError.CAID_LPCXPresso1769),
		/**
		 * BCフラグメント・ライブラリ
		 */
		_BCF:{
			/**
			 * ([i_addr]&(~(i_mask<<i_shl))|((i_mask & i_val)<<i_shl)を実行するMimicBCを生成する。
			 [S0]のアドレスのメモリ値に[S1]の値でマスクして,[S2]の値とORして[S0]に書き戻す。
			 SGET #0;
			 SGET #1;
			 SGET #2;
			 MGET #7,#0;
			 AND #7,#1;
			 OR #7,#2;
			 MPUT #7,#0;
			 @param i_mask
			 ビットマスク
			 @param i_val
			 ビット値
			 */
			setBit:function setBit(i_addr,i_mask,i_val,i_shl,i_db)
			{
				try{
					i_db.push(i_addr);
					i_db.push(~(i_mask<<i_shl));
					i_db.push((i_mask & i_val) << i_shl);
					return "EA00EA01EA02DB0700AA0701AE0702DF0700";
				}catch(e){
					throw new MiMicException(e);
				}			
			},
			/**
			 * 値をメモリに設定する。
			 	SGET #0
			 	SGET #1
			 	MPUT #1,#0
			 */
			setMem:function setMem(i_addr,i_val,i_db)
			{
				try{
					i_db.push(i_addr);
					i_db.push(i_val);
					return "EA00EA01DF0100";
				}catch(e){
					throw new MiMicException(e);
				}			
			},
			/*	メモリを32bit読む
			 	SGET #0
			 	MGET #0,#0
			 	SPUT #0
			 */			
			getMem:function getMem(i_addr,i_db)
			{
				try{
					i_db.push(i_addr);
					return "EA00DB0000EE00";
				}catch(e){
					throw new MiMicException(e);
				}			
			},
			/*	メモリを32bit読む
			 	SGET #0
			 	MGET #0,#0
			 	SPUT #0
			 */
			READMEM:"EA00DB0000EE00",
			
			/**
			 * BCフラグメントを終了する。
				 EXIT;
				 .END
			 */
			END:"ZZ.E"	
		},
		/**
		 * @see API manual
		 */
		FUNC_NAME:[
			"GPIO","AD","DA"
		],
		/**
		 * @see API manual
		 */
		PHY_NAME:[
			"GPIO","ADC","DAC"
		],
		/**
		 * @see API manual
		 PHY:{PHY_NAME:[PCLKSELreg#PCLKSELbit,PCONPbit]}
		 */
		PHY:{
			WDT:	[0,0,null],
			TIMER0:	[0,2,1],
			TIMER1:	[0,4,2],
			UART0:	[0,6,3],
			UART1:	[0,8,4],
			PWM1:	[0,12,6],
			I2C0:	[0,14,7],
			SPI:	[0,16,8],
			RTC:	[null,null,9],
			SSPI1:	[0,20,10],
			DAC:	[0,22,null],
			ADC:	[0,24,12],
			CAN1:	[0,26,13],
			CAN2:	[0,28,14],
			ACF:	[0,30,null],
			QEI:	[1,0,18],
			GPIOINT:[1,2,15],
			PCB:	[1,4,null],
			I2C1:	[1,6,19],
			SSP0:	[1,10,21],
			TIMER2:	[1,12,22],
			TIMER3:	[1,14,23],
			UART2:	[1,16,24],
			UART3:	[1,18,25],
			I2C2:	[1,20,26],
			I2S:	[1,22,27],
			RIT:	[1,26,16],
			SYSCON:	[1,28,null],
			PWM_MC:[1,30,17],//MC?
			GPDMA:[null,null,29],
			ENET:[null,null,30],
			USB:[null,null,31]
		},
		/**
		 * @see API Manual
		 */
		P0:[
			[0,0,0,0,["GPIO0.0","RD1","TXD3","SDA1"]],//0
			[0,0,0,1,["GPIO0.1","TD1","RXD3","SCL1"]],//1
			[0,0,0,2,["GPIO0.2","TXD0","AD0.7",null]],//2
			[0,0,0,3,["GPIO0.3","RXD0","AD0.6",null]],//3
			[0,0,0,4,["GPIO0.4","I2SRX_CLK","RD2","CAP2.0"]],//4
			[0,0,0,5,["GPIO0.5","I2SRX_WS","TD2","CAP2.1"]],//5
			[0,0,0,6,["GPIO0.6","I2SRX_SDA","SSEL1","MAT2.0"]],//6
			[0,0,0,7,["GPIO0.7","I2STX_CLK","SCK1",	"MAT2.1"]],//7
			[0,0,0,8,["GPIO0.8","I2STX_WS","MISO1","MAT2.2"]],//8
			[0,0,0,9,["GPIO0.9","I2STX_SDA","MOSI1","MAT2.3"]],//9
			[0,0,0,10,["GPIO0.10","TXD2","SDA2","MAT3.0"]],//10
			[0,0,0,11,["GPIO0.11","RXD2","SCL2","MAT3.1"]],//11
			null,null,null,//12,13,14
			[0,0,0,15,["GPIO0.15","TXD1","SCK0","SCK"]],//15
			[1,1,0,16,["GPIO0.16","RXD1","SSEL0","SSEL"]],//16
			[1,1,0,17,["GPIO0.17","CTS1","MISO0","MISO"]],//17
			[1,1,0,18,["GPIO0.18","DCD1","MOSI0","MOSI"]],//18
			[1,1,0,19,["GPIO0.19","DSR1",null,"SDA1"]],//19
			[1,1,0,20,["GPIO0.20","DTR1",null,"SCL1"]],//20
			[1,1,0,21,["GPIO0.21","RI1",null,"RD1"]],//21
			[1,1,0,22,["GPIO0.22","RTS1",null,"TD1"]],//22
			[1,1,0,23,["GPIO0.23","AD0.0","I2SRX_CLK","CAP3.0"]],//23
			[1,1,0,24,["GPIO0.24","AD0.1","I2SRX_WS","CAP3.1"]],//24
			[1,1,0,25,["GPIO0.25","AD0.2","I2SRX_SDA","TXD3"]],//25
			[1,1,0,26,["GPIO0.26","AD0.3","AOUT","RXD3"]],//26
			[1,null,null,22,["GPIO0.27","SDA0",	"USB_SDA",null]],//27
			[1,null,null,24,["GPIO0.28","SCL0",	"USB_SCL",null]],//28
			[1,null,0,29,["GPIO0.29","USB_D+",null,null]],//29
			[1,null,0,30,["GPIO0.30","USB_D-",null,null]],//30
			],
		P1:[	//P1[x]
			[2,2,1,0,["GPIO1.0","ENET_TXD0",null,null]],//0
			[2,2,1,1,["GPIO1.1","ENET_TXD1",null,null]],//1
			null,null,//2,3
			[2,2,1,4,["GPIO1.4","ENET_TX_EN",null,null]],//4
			null,null,null,//5,6,7
			[2,2,1,8,["GPIO1.8","ENET_CRS",null,null]],//8
			[2,2,1,9,["GPIO1.9","ENET_RXD0",null,null]],//9
			[2,2,1,10,["GPIO1.10","ENET_RXD1",null,null]],//10
			null,null,null,//11,12,13
			[2,2,1,14,["GPIO1.14","ENET_RX_ER",null,null]],//14
			[2,2,1,15,["GPIO1.15","ENET_REF_CLK",null,null]],//15
			[3,3,1,16,["GPIO1.16","ENET_MDC",null,null]],//16
			[3,3,1,17,["GPIO1.17","ENET_MDIO",null,null]],//17
			[3,3,1,18,["GPIO1.18","USB_UP_LED","PWM1.1","CAP1.0"]],//18
			[3,3,1,19,["GPIO1.19","MCOA0","USB_PPWR","CAP1.1"]],//19
			[3,3,1,20,["GPIO1.20","MCI0","PWM1.2","SCK0"]],//20
			[3,3,1,21,["GPIO1.21","MCABORT","PWM1.3","SSEL0"]],//21
			[3,3,1,22,["GPIO1.22","MCOB0","USB_PWRD","MAT1.0"]],//22
			[3,3,1,23,["GPIO1.23","MCI1","PWM1.4","MISO0"]],//23
			[3,3,1,24,["GPIO1.24","MCI2","PWM1.5","MOSI0"]],//24
			[3,3,1,25,["GPIO1.25","MCOA1",null,"MAT1.1"]],//25
			[3,3,1,26,["GPIO1.26","MCOB1","PWM1.6","CAP0.0"]],//26
			[3,3,1,27,["GPIO1.27","CLKOUT","USB_OVRCR","CAP0.1"]],//27
			[3,3,1,28,["GPIO1.28","MCOA2","PCAP1.0","MAT0.0"]],//28
			[3,3,1,29,["GPIO1.29","MCOB2","PCAP1.1","MAT0.1"]],//29
			[3,3,1,30,["GPIO1.30",null,"VBUS","AD0.4"]],//30
			[3,3,1,31,["GPIO1.31",null,"SCK1","AD0.5"]],//31
			],
		P2:[	//P2[x]
			[4,4,2,0,["GPIO2.0","PWM1.1","TXD1",null]],//0
			[4,4,2,1,["GPIO2.1","PWM1.2","RXD1",null]],//1
			[4,4,2,2,["GPIO2.2","PWM1.3","CTS1",null]],//2
			[4,4,2,3,["GPIO2.3","PWM1.4","DCD1",null]],//3
			[4,4,2,4,["GPIO2.4","PWM1.5","DSR1",null]],//4
			[4,4,2,5,["GPIO2.5","PWM1.6","DTR1",null]],//5
			[4,4,2,6,["GPIO2.6","PCAP1.0","RI1",null]],//6
			[4,4,2,7,["GPIO2.7","RD2","RTS1",null]],//7
			[4,4,2,8,["GPIO2.8","TD2","TXD2","ENET_MDC"]],//8
			[4,4,2,9,["GPIO2.9","USB_CONNECT","RXD2","ENET_MDIO"]],//9
			[4,4,2,10,["GPIO2.10","EINT0","NMI",null]],//10
			[4,4,2,11,["GPIO2.11","EINT1",null,"I2STX_CLK"]],//11
			[4,4,2,12,["GPIO2.12","EINT2",null,"I2STX_WS"]],//12
			[4,4,2,13,["GPIO2.13","EINT3",null,"I2STX_SDA"]],//13
			],
		P3:[	//P3[x]
			null/* 0*/,null/* 1*/,null/* 2*/,null/* 3*/,null/* 4*/,null/* 5*/,null/* 6*/,null/* 7*/,null/* 8*/,null/* 9*/,
			null/*10*/,null/*11*/,null/*12*/,null/*13*/,null/*14*/,null/*15*/,null/*16*/,null/*17*/,null/*18*/,null/*19*/,
			null/*20*/,null/*21*/,null/*22*/,null/*23*/,null/*24*/,
			[7,7,3,25,["GPIO3.25",null,"MAT0.0","PWM1.2"]],//25
			[7,7,3,26,["GPIO3.26","STCLK","MAT0.1","PWM1.3"]],//26
		],
		P4:[	//P4[x]
			null/* 0*/,null/* 1*/,null/* 2*/,null/* 3*/,null/* 4*/,null/* 5*/,null/* 6*/,null/* 7*/,null/* 8*/,null/* 9*/,
			null/*10*/,null/*11*/,null/*12*/,null/*13*/,null/*14*/,null/*15*/,null/*16*/,null/*17*/,null/*18*/,null/*19*/,
			null/*20*/,null/*21*/,null/*22*/,null/*23*/,null/*24*/,null/*25*/,null/*26*/,null/*27*/,
			[9,9,4,28,["GPIO4.28","RX_MCLK","MAT2.0","TXD3"]],//28
			[9,9,4,29,["GPIO4.29","TX_MCLK","MAT2.1","RXD3"]]//29
		],
		/**
		 * [forSystem]ピン識別子のアクセス関数である。
		ピン機能シンボルのプレフィクス（機能シンボル）を、 ピン識別子を元に、完全な名称に変換する。
		 ピンにFunctionが含まれているか調べることが出来る。
		 */
		completePinFunctionName:function completePinFunctionName(i_pin,i_prefix)
		{
			try{
				//配列探索
				for(var i=0;i<i_pin[4].length;i++){
					if(i_pin[4][i].indexOf(i_prefix)==0){
						//ピン情報の構成
						return i_pin[4][i];
					}
				}
				throw new MiMicException("The function name '"+i_prefix+"' is not unknown.");
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * [forSystem]ピン識別子のアクセス関数。
		 ピン識別値から、Pinレジスタの番号と、ビット位置を取り出す。
		 @return
		 {s:pinselのレジスタ番号、m:modeのレジスタ番号、,o:odのレジスタ番号,sbm:mode,selのビット位置,ob:odのビット位置}
		 */
		getPinRegInfo:function getPinRegInfo(i_pin)
		{
			try{
				return {s:i_pin[0],m:i_pin[1],o:i_pin[2],smb:(i_pin[3]%16)*2,ob:i_pin[3]};
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * [forSystem]ピン識別子のアクセス関数。
		 PIN機能名に一致するPinsel値を取り出す。
		 */
		getPinSelByFunctionName:function getPinSelByFunctionName(i_pin,i_name)
		{
			try{
				for(var i=0;i<i_pin[4].length;i++){
					if(i_pin[4][i]==i_name){
						//ピン情報の構成
						return i;
					}
				}
				throw new MiMicException("The function name '"+i_name+"' has not function.");
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * [forSystem]ピンが、ピン機能シンボルを持つか返す。
		 */
		hasPinFunctionName:function hasPinFunctionName(i_pin,i_name)
		{
			return i_pin[4].indexOf(i_name)>=0;
		}
	}


DEV.I2c=function I2c()
{
	throw new MiMicException("Not imprement.");
}
DEV.Pwm=function Pwm()
{
	throw new MiMicException("Not imprement.");
}
DEV.Dma=function Dma()
{
	throw new MiMicException("Not imprement.");
}
DEV.Uart=function Uart()
{
	throw new MiMicException("Not imprement.");
}
DEV.Usb=function Usb()
{
	throw new MiMicException("Not imprement.");
}


LPCXpresso1769=DEV;
}());

(function(){
	var DEV=LPCXpresso1769;
	var BCF=DEV._BCF;
	var EE=DEV._EE;
	var isUndef=MiMicLib.isUndef;

	/**
	 * 配列をMiMicDBへ変換する。
	 @throws
	 変換失敗時は例外
	 */
	function array2MimicDb(i_array)
	{
		try{
			var dt=["0000000","000000","00000","0000","000","00","0",""];
			var p="";
			for(var i=0;i<i_array.length;i++){
				var v=(i_array[i]>>>0);
				var s=v.toString(16).toLowerCase();
				if(s.length<1 || s.length>8){
					throw new MiMicException(EE.NG);
				}
				p+=dt[s.length-1]+s;
			}
			return p;
		}catch(e){
			throw MiMicException(e);
		}
	}
	
	/**
	 * PINの機能名を含むPin識別子を得る。
	 */
	function getPinByFuncName(i_name)
	{
		try{
			function getPinByFName(i_pintbl,i_name)
			{
				for(var i=0;i<i_pintbl.length;i++){
					if(i_pintbl[i]==null){
						continue;
					}
					if(DEV.hasPinFunctionName(i_pintbl[i],i_name)){
						return i_pintbl[i];
					}
				}
				return null;
			}
			var tbl=[DEV.P0,DEV.P1,DEV.P2,DEV.P3,DEV.P4];
			for(var i=0;i<tbl.length;i++){
				var pin=getPinByFName(tbl[i],i_name);
				if(pin!=null){
					return pin;
				}
			}
			throw new MiMicException("pin function '"+i_name+"' not found");
		}catch(e){
			throw new MiMicException(e);
		}
	}
	/**
	 * PIN機能名から使用するペリフェラルシンボルを推定する。
	 */
	function pinFuncName2PhyName(i_pinfuncname)
	{
		try{
			//Pin function nameプレフィクスとペリフェラル名の対比表
			var TBL={
				"GPIO":"GPIO",
				"AD":"ADC",
				"DA":"DAC"
			};
			for(var key in TBL){
				if(i_pinfuncname.indexOf(key)==0){
					return TBL[key];
				}
			}
			//見つからぬ。
			throw new MiMicException("Peripheral for "+i_pinfunction+" not found.");
		}catch(e){
			throw new MiMicException(e);
		}
	}
	
	/**
	 * @see API Manual
	 */
	DEV.Mcu=function Mcu(i_mimic_addr,i_is_activate)
	{
		try{
			//リモートインタフェイスの取得
			this._mif=new MiMicRemoteMcuInterface(i_mimic_addr);
			var ac=isUndef(i_is_activate)?true:i_is_activate;
			if(ac){
				this.activate();
			};
		}catch(e){
			throw new MiMicException(e);
		}
	};
	DEV.Mcu.prototype=
	{
		/**
		 * MiMicInterface
		 */
		_mif:null,
		/**
		 * @see API manual
		 */
		isActive:function isActive()
		{
			try{
				return this._mif.isConnected();
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * @see API manual
		 */
		events:{
			/**
			 * @see API manual
			 */
			onActivateChanged:null
		},
		/**
		 * ペリフェラルオブジェクトのホルダ。ペリフェラル名をキーにした、ペリフェラルオブジェクトの連想配列。
		 MCUが、自身のペリフェラルオブジェクトを管理する為に使う。
		 */
		_phy_holder:[],
		/**
		 * @see API manual
		 */
		activate:function activate()
		{
			try{
				var _t=this;
				//既にアクティブならエラー。
				if(this.isActive()){
					throw new MiMicException("Already activated!");
				}
				//接続
				this._mif.connect(
					function(b){
						if(!b){
							//非同期切断を検出。
							if(_t.events.onActivateChanged!=null){
								_t.events.onActivateChanged(false);
							}
						}
					}
				);
				if(this.events.onActivateChanged!=null){
					this.events.onActivateChanged(true);
				}
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * @see API manual
		 */
		deactivate:function deactivate()
		{
			try{
				if(!this.isActive()){
					throw new MiMicException("Already activated!");
				}
	
				this._mif.disconnect();
				if(this.events.onActivateChanged!=null){
					this.events.onActivateChanged(false);
				}
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * @see API manual
		 */
		callMiMic:function callMiMic(/**/)
		{
			try{
				var bc;
				switch(arguments.length){
				case 1:
					//mimicBC
					bc=arguments[0];
					break;
				case 2:
					//mimicBC:string,db:array[int]
					bc=arguments[0]+array2MimicDb(arguments[1]);
					break;
				default:
					throw new MiMicException(e);
				}
				return this._mif.execBc(bc);
			}catch(e){
				throw new MiMicException(e);
			}
		},
		/**
		 * @see API manual
		 */
		callMiMicWithCheck:function execBc2WithException(/**/)
		{
			try{
				var ret=this.callMiMic.apply(this,arguments);
				if(ret.result!=0x0){
					throw new MiMicException(EE.VM_RUNTIME,"Result="+ret.toString());
				}
				return ret;
			}catch(e){
				throw MiMicException(e);
			}
		},
		/**
		 * @see API manual
		 */
		getPin:function getPin(/*...*/)
		{
			switch(arguments.length){
			case 1://getPin:function getPin(i_pin_function)
				//pin_functionを持ってるPINを探す。
				var pin=getPinByFuncName(arguments[0]);
				//function名からペリフェラル名を得る。
				var phy_name=pinFuncName2PhyName(arguments[0]);
				//ピンコンストラクタとペリフェラルを使ってピンを生成。
				return this.getPeripheral(phy_name).getPin(pin);
			case 2://getPin:function getPin(i_pin,i_function_name)
				//function名からペリフェラル名を得る。(PINシンボル変換を流用)
				var phy_name=pinFuncName2PhyName(arguments[1]);
				return this.getPeripheral(phy_name).getPin(arguments[0]);
			default:
				throw new MiMicException();
			}
		},
		/**
		 * @see API manual
		 */
		getPort:function getPort(i_port_name)
		{
			throw new MiMicException("Not implemented.");
		},
		/**
		 * @see API manual
		 */
		getPeripheral:function getPeripheral(i_phy_symbol)
		{
			//ペリフェラルアイテムが生成済か確認
			if(isUndef(this._phy_holder[i_phy_symbol])){
				//存在しなければPhyを生成。
				var phy;
				switch(i_phy_symbol){
				case "GPIO":new DEV.Gpio(this);break;
				case "ADC":new DEV.Adc(this);break;
				default:
					throw new MiMicException("Unknown peripheral symbol "+i_phy_symbol);
				}
			}
			//phyからregisterのコールバックがかかるはず。
			return this._phy_holder[i_phy_symbol];
		},
		/**
		 * [forSystem]システム関数。MCU依存のペリフェラルオブジェクトを登録する。
		 */
		registerPhy:function registerPhy(i_phy,i_symbol)
		{
			try{
				//登録済みのシンボルは登録できない。
				if(!isUndef(this._phy_holder[i_symbol])){
					throw new MiMicException("Peripheral symbol "+i_symbol+" is already exist on mcu.");
				}
				this._phy_holder[i_symbol]=i_phy;
			}catch(e){
				throw new MiMicException(e);
			}
		}
	}
}());/**
 * LPCXPressoのメモリアクセサを定義します。
 */
(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;
/**
 * @see API manual
 */
DEV.Memory=function Memory(i_mcu,i_base)
{
	if(!isUndef(i_base)){
		this._base=i_base;
	}
	this._mcu=i_mcu;
}
DEV.Memory.prototype=
{
	_base:0x00000000,
	_mcu:null,
	/**
	 *@see API manual
	 */
	read32:function read32(/*arguments*/)
	{
		try{
			function checkAlign(v){
				if(v%4!=0){
					throw new MiMicException(EE.INVALID_ARG,"An alignment is not 32bit unit.");
				}
			}
			var bc="";
			var ar=new Array();
			var offset=arguments[0];
			switch(arguments.length){
			case 1:
				if(!isNaN(offset)){
					//read32(i_offset:int)
					checkAlign(offset);
					bc=BCF.READMEM;
					ar.push(this._base+offset);
				}else{
					//read32(i_offsets:array)
					for(var i=0;i<offset.length;i++){
						checkAlign(offset[i]);
						bc+=BCF.getMem(this._base+offset[i],ar);
					}
				}
				break;
			case 2:
				//read32(i_offset:int,i_size:int)
				checkAlign(offset);
				checkAlign(arguments[1]);
				var l=arguments[1]/4;
				for(var i=0;i<l;i++){
					bc+=BCF.READMEM;
					ar.push(this._base+offset+i*4);
				}
				break;
			default:
				break;
			}
			var ret=this._mcu.callMiMicWithCheck(bc+BCF.END,ar).stream;
			return ret.length==1?ret[0]:ret;
		}catch(e){
			throw new MiMicException(e);
		}
	}
}

}());(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;
/**
 * @see API manual
 */
DEV.Peripheral=function(i_mcu,i_phy,i_opt)
{
	try{
		if(isUndef(i_phy)){
			throw new MiMicException(EE.INVALID_PHY);
		}
		this._mcu=i_mcu;
		this._phy=i_phy;
		//オプション設定するならする。
		if(!isUndef(i_opt)){
			this.setOpt(i_opt);
		}
	}catch(e){
		throw new MiMicException(e);
	}	
}
DEV.Peripheral.prototype=
{
	_PCLKSEL:[0x400FC1A8,0x400FC1AC],
	_PCONP:0x400FC0C4,
	_phy:null,
	_mif:null,
	BCF_setOpt:function BCF_setOpt(i_opt,i_db)
	{
		try{
			var bc="";
			//pconp
			if(!isUndef(i_opt.power)){
				if(this._phy[2]==null){
					throw new MiMicException(EE.INVALID_CFG,"The pin does not support PCONP.");
				}
				//pinselAddrを得る
				bc+=BCF.setBit(this._PCONP,0x00000001,i_opt.power,this._phy[2],i_db);
			}
			//clock
			if(!isUndef(i_opt.clock)){
				if(this._phy[0]==null){
					throw new MiMicException(EE.INVALID_CFG,"The pin does not support PCLKSEL.");
				}
				bc+=BCF.setBit(this._PCLKSEL[this._phy[0]],0x00000003,i_opt.clock,this._phy[1],i_db);
			}
			return bc;
		}catch(e){
			throw new MiMicException(e);
		}		
	},
	/**
	 * @see API manual
	 */
	setOpt:function setOpt(i_opt)
	{
		try{
			var db=new Array();
			var bc=this.BCF_setOpt(i_opt,db);
			if(bc.length==0){
				throw new MiMicException("i_opt is empty or invalid.");
			}
			//MiMicBCを生成して実行
			this._mcu.callMiMicWithCheck(bc+BCF.END,db);

		}catch(e){
			throw new MiMicException(e);
		}
		return;		
	}
}

}());(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;

/**
 * @see API manual.
 */
DEV.Pin=function Pin(i_mcu,i_pin,i_opt)
{
	try{
		if(isUndef(i_pin)){
			throw new MiMicException(EE.INVALID_PIN);
		}
		this._mcu=i_mcu;
		this._pininfo=DEV.getPinRegInfo(i_pin);
		//オプション設定するならする。
		if(!isUndef(i_opt)){
			this.setOpt(i_opt);
		}
	}catch(e){
		throw new MiMicException(e);
	}
}
DEV.Pin.prototype=
{	
	_PINSEL:[0x4002C000,0x4002C004,0x4002C008,0x4002C00C,0x4002C010,null,null,0x4002C0C0,null,0x4002C024,0x4002C028],
	_PINMODE:[0x4002C040,0x4002C044,0x4002C048,0x4002C04C,0x4002C050,0x4002C054,0x4002C058,0x4002C05C,null,0x4002C064],
	_PINMODE_OD:[0x4002C068,0x4002C06C,0x4002C070,0x4002C074,0x4002C078],
	_pininfo:null,
	_mcu:null,


	/**
	 * setPinのBCを生成します。
	 @param i_opt
	 setOptBcを参照してください。
	 @return
	 BCフラグメントを返します。
	 */
	BCF_setOpt:function BCF_setOpt(i_opt,i_db)
	{
		try{
			var bc="";
			//PINFUNC,PINMODEのbit位置
			//pinsel
			if(!isUndef(i_opt.sel)){
				
				//pinselAddrを得る
				bc+=BCF.setBit(this._PINSEL[this._pininfo.s],0x00000003,i_opt.sel,this._pininfo.smb,i_db);
			}
			//pinmode
			if(!isUndef(i_opt.mode)){
				if(this._pininfo.m==null){
					throw new MiMicException(EE.INVALID_CFG,"The pin does not support PINMODE. pininfo.");
				}
				bc+=BCF.setBit(this._PINMODE[this._pininfo.m],0x00000003,i_opt.mode,this._pininfo.smb,i_db);
			}
			//pinmode_od
			if(!isUndef(i_opt.od)){
				if(this._pininfo.o==null){
					throw new MiMicException(EE.INVALID_CFG,"The pin does not support PINMODE_OD. pininfo.");
				}
				bc+=BCF.setBit(this._PINMODE_OD[this._pininfo.o],0x00000001,i_opt.od,this._pininfo.ob,i_db);
			}
			return bc;
		}catch(e){
			throw new MiMicException(e);
		}
	},
	/**
	 * @see API manual
	 */
	setOpt:function setOpt(i_opt)
	{
		try{
			var db=new Array();
			var bc=this.BCF_setOpt(i_opt,db);
			if(bc.length==0){
				throw new MiMicException("i_opt is empty or invalid.");
			}
			//MiMicBCを生成して実行
			this._mcu.callMiMicWithCheck(bc+BCF.END,db);
		}catch(e){
			throw new MiMicException(e);
		}
		return;
	}
}

}());/**
 * LPCXPresso1769のADCクラスを定義します。
 class Adc
 ADCペリフェラルを操作するクラス
 class AdcPin
 ADCピンを操作するクラス
 */
(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;



/**
 * このクラスは、ADCペリフェラルを管理します。
 @param i_opt
 オプション値です。省略時は、{phy:{power:1}}を指定します。
 {phy:{ペリフェラルオプション}}
 */
DEV.Adc=function Adc(i_mcu,i_opt)
{
	try{
		this._mcu=i_mcu;
		var phyopt=isUndef(i_opt)?{power:1}:i_opt.phy;
		//PHY生成。
		this._phy=new DEV.Peripheral(i_mcu,DEV.PHY.ADC);
		//初期化。
		var bc="";
		var db=new Array();
		if(isUndef(i_opt)){
			//パワーONのみ
			bc+=this._phy.BCF_setOpt({power:1},db);
		}else{
			bc+=this._phy.BCF_setOpt(i_opt.phy,db);
		}
		//0x00210000(PDN,BURSTを1)
		bc+=BCF.setMem(this._AD0CR,0x00200500,db);
		bc+=BCF.setMem(this._AD0CR,0x00210500,db);
		//
		this._mcu.callMiMicWithCheck(bc+BCF.END,db);
		//ペリフェラルをMCUに登録
		this._mcu.registerPhy(this,"ADC");
	}catch(e){
		throw new MiMicException(e);
	}

}
DEV.Adc.prototype=
{
	_AD0CR:0x40034000,
	_AD0DR:[0x40034010,0x40034014,0x40034018,0x4003401C,0x40034020,0x40034024,0x40034028,0x4003402C],
	_phy:null,
	_mcu:null,
	/**
	 * AD0CRのSELフィールドの値を更新するBC
	 */
	BCF_setSel:function BCF_setSel(i_bit,i_val,i_db)
	{
		try{
			return BCF.setBit(this._AD0CR,0x00000001,i_val,i_bit,i_db);
		}catch(e){
			throw new MiMicException(e);
		}
	},
	/**
	 * n番のAD0DRの値を得るBC
	 */
	BCF_getAD0DR:function BCF_getAD0DR(i_ch,i_db)
	{
		try{
			return BCF.getMem(this._AD0DR[i_ch],i_db);
		}catch(e){
			throw new MiMicException(e);
		}
	},
	/**
	 * see
	 */
	getPin:function getPin(i_pin,i_opt)
	{
		try{
			return new DEV.AdcPin(this,i_pin,i_opt);
		}catch(e){
			throw new MiMicException(e);
		}
	}	
}

/**
 * このクラスは、ADCペリフェラルとPin識別子から、ADCピンを構成します。
 @param i_adc
 ADCペリフェラル
 @param i_pin
 ADCを構成するピン番号
 @param i_opt
 ピンオプション。省略してください。
 */
DEV.AdcPin=function AdcPin(i_adc,i_pin,i_opt)
{
	try{
		this._adc=i_adc;
		//PINがAD持ってるか確認
		this._pin=new DEV.Pin(i_adc._mcu,i_pin);
		//PinNameからピン情報を得る。
		this._adinfo=function(i_pin){
			//pinの完全な機能名を得る。(得られれば機能がある。)
			var func_name=DEV.completePinFunctionName(i_pin,"AD");
			//portとbitを得る(AD0だけしか管理しないよ)
			var a=func_name.substring(2).split(".");
			var r={port:0,ch:parseInt(a[1]),pin_sel:DEV.getPinSelByFunctionName(i_pin,func_name)};
			if(!isNaN(r.ch)){
				return r;
			}
			throw new MiMicException(EE.INVALID_CFG,"The pin has not AD fuction.");
		}(i_pin);
		//設定用のBCを作って実行
		{
			var bc="";
			var db=new Array();
			bc+=this._pin.BCF_setOpt({sel:this._adinfo.pin_sel},db);
			//AD0CRに値設定
			bc+=this._adc.BCF_setSel(this._adinfo.ch,1,db);
			var ret=this._adc._mcu.callMiMicWithCheck(bc+BCF.END,db);
		}
	}catch(e){
		throw new MiMicException(e);
	}
}

DEV.AdcPin.prototype=
{
	_adc:null,
	_pin:null,
	_adinfo:null,
	/**
	 * ピンからAD変換した値を得る。
	 @return
	 12bitのAD変換値
	 */
	getValue:function getValue()
	{
		try{
			//メモリから値取得
			var db=new Array();
			var bc=this._adc.BCF_getAD0DR(this._adinfo.ch,db);
			var ret=this._adc._mcu.callMiMicWithCheck(bc+BCF.END,db);
			return (ret.stream[0]>>4)&0x00000fff;
		}catch(e){
			throw new MiMicException(e);
		}

	}	
}
	


}());
(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;

/**
 * このクラスは、GPIOペリフェラルを管理します。
 */
DEV.Gpio=function Gpio(i_mcu)
{
	try{
		this._mcu=i_mcu;
		i_mcu.registerPhy(this,"GPIO");
	}catch(e){
		throw new MiMicException(e);
	}
}
DEV.Gpio.prototype={
	_FIO_DIR :[0x2009C000,0x2009C020,0x2009C040,0x2009C060,0x2009C080],
	_FIO_PIN :[0x2009C014,0x2009C034,0x2009C054,0x2009C074,0x2009C094],
	_FIO_SET :[0x2009C018,0x2009C038,0x2009C058,0x2009C078,0x2009C098],
	_FIO_CLR :[0x2009C01C,0x2009C03C,0x2009C05C,0x2009C07C,0x2009C09C],
	_FIO_MASK:[0x2009C010,0x2009C030,0x2009C050,0x2009C070,0x2009C090],
	_mcu:null,
	/**
	 * ピン用のBCF生成関数
	 */
	BCF_setDir:function BCF_setDir(i_ch,i_bit,i_dir,i_db)
	{
		try{
			return BCF.setBit(this._FIO_DIR[i_ch],0x00000001,i_dir,i_bit,i_db);
		}catch(e){
			throw new MiMicException(e);
		}
	},
	BCF_setValue:function BCF_setValue(i_ch,i_bit,i_val,i_db)
	{
		/*GPIOレジスタにセットするBC
		SGET #0;//GPIOの値レジスタアドレス
		SGET #1;//GPIOのMASKレジスタアドレス
		SGET #2;//MASKレジスタの値
		MPUT #2,#1;//MASK設定
		LD #2,0xffffffff
		MPUT #2,#0;//値セット
		*/		
		try{
			if(i_val){
				i_db.push(this._FIO_SET[i_ch],this._FIO_MASK[i_ch],~(0x00000001<<i_bit));
			}else{
				i_db.push(this._FIO_CLR[i_ch],this._FIO_MASK[i_ch],~(0x00000001<<i_bit));
			}
			return "EA00EA01EA02DF0201FB02ffffffffDF0200";
		}catch(e){
			throw new MiMicException(e);
		}
	},
	BCF_getValue:function BCF_getValue(i_ch,i_bit,i_db)
	{
		/*GPIOレジスタからとってくるBC
		SGET #0;//GPIOの値レジスタアドレス
		SGET #1;//GPIOのMASKレジスタアドレス
		SGET #2;//MASKレジスタの値
		MSET #2,#1;//MASK設定
		MGET #3,#0;//値取得
		SPUT #3;//SPUT
		 */
		try{
			i_db.push(this._FIO_PIN[i_ch],this._FIO_MASK[i_ch],~(0x00000001<<i_bit));
			return "EA00EA01EA02DF0201DB0300EE03";
		}catch(e){
			throw new MiMicException(e);
		}		
	},
	/**
	 * @see API manual
	 */
	getPin:function createPin(i_pin,i_opt)
	{
		try{
			return new DEV.GpioPin(this,i_pin,i_opt);
		}catch(e){
			throw new MiMicException(e);
		}
	}

}

/**
 * @see API manual
 */
DEV.GpioPin=function GpioPin(i_gpio,i_pin,i_opt)
{
	try{
		//PINを作る。
		this._gpio=i_gpio;
		this._pin=new DEV.Pin(i_gpio._mcu,i_pin);
		//pinからGPIOに関するピン情報を計算。
		this._gpioinfo=function(i_pin)
		{
			//pinの完全な機能名を得る。(得られれば機能がある。)
			var func_name=DEV.completePinFunctionName(i_pin,"GPIO");
			//pin名からポートとビットを得る。
			var a=func_name.substring(4).split(".");
			//pin情報を構成。
			var r={port:parseInt(a[0]),bit:parseInt(a[1]),pin_sel:DEV.getPinSelByFunctionName(i_pin,func_name)};
			if(!isNaN(r.port) && !isNaN(r.bit)){
				return r;
			}
			throw new MiMicException(EE.INVALID_CFG,"The pin has not GPIO fuction.");
		}(i_pin);
		if(isUndef(i_opt)){
			//optionが無いときは、GPIO化だけ。
			this.setOpt({sel:this._gpioinfo.pin_sel});
		}else{
			//optionがあるときはGPIO化と、pinコンフィギュレーションの継承
			this.setOpt(i_opt);
		}
	}catch(e){
		throw new MiMicException(e);
	}
}
DEV.GpioPin.prototype=
{
	_gpio:null,
	_pin:null,//LPCXpressoのpin
	_gpioinfo:null,//GPIOの情報{port,bit}
	/**
	 * @see API manual
	 */
	setOpt:function setOpt(i_opt)
	{
		try{
			var bc="";
			var db=new Array();
			//dirの設定
			if(!isUndef(i_opt.dir)){
				bc+=this._gpio.BCF_setDir(this._gpioinfo.port,this._gpioinfo.bit,i_opt.dir,db);
			}
			//pinselの値
			if(isUndef(i_opt.pin)){
				//pinコンフィギュレーションが無いとき
				bc+=this._pin.BCF_setOpt({sel:this._gpioinfo.pin_sel},db);
			}else{
				//pinコンフィギュレーションがあるとき
				var t=i_opt.pin.sel;//待避
				i_opt.pin.sel=this._gpioinfo.pin_sel;//selの上書き
				bc+=this._pin.BCF_setOpt(i_opt.pin,db);
				i_opt.pin.sel=t;
			}
			//実行なう
			//MiMicBCを生成して実行
			this._gpio._mcu.callMiMicWithCheck(bc+BCF.END,db);
		}catch(e){
			throw new MiMicException(e);
		}
	},
	/**
	 * @see API manual.
	 */
	setValue:function setValue(i_val)
	{
		try{
			var bc="";
			var db=new Array();
			bc+=this._gpio.BCF_setValue(this._gpioinfo.port,this._gpioinfo.bit,i_val,db);
			this._gpio._mcu.callMiMicWithCheck(bc+BCF.END,db);
		}catch(e){
			throw new MiMicException(e);
		}
	},	
	/**
	 * @see API manual.
	 */
	getValue:function getValue()
	{

		try{
			var bc="";
			var db=new Array();
			bc+=this._gpio.BCF_getValue(this._gpioinfo.port,this._gpioinfo.bit,db);
			var ret=this._gpio._mcu.callMiMicWithCheck(bc+BCF.END,db);
			return (ret.stream[0]>>this._gpioinfo.bit)&0x01;
		}catch(e){
			throw new MiMicException(e);
		}
	},
	/**
	 * ビットパターンを出力します。
	 @param i_val_array
	 Array[int]:ビットパターンです。[0,1,0,1]の場合、0,1,0,1の順に、パターンを出力します。
	 最大数は20です。
	 @remark
	 最適化対象
	 */
	outPatt:function outPatt(i_val_array)
	{
		try{
			var bc="";
			var db=new Array();
			for(var i=0;i<i_val_array.length;i++){
				bc+=this._gpio.BCF_setValue(this._gpioinfo.port,this._gpioinfo.bit,i_val_array[i],db);
			}
			this._gpio._mcu.callMiMicWithCheck(bc+BCF.END,db);
		}catch(e){
			throw new MiMicException(e);
		}
	},
	
}

}());
