/**
 * @fileOverview LPC1769の物理ピンを制御するクラスを定義する。
 */

(function(){
var DEV=LPC1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;

/**
 * LPCXPresso1769.Pin (Pin)クラスのコンストラクタ。
 * ピン識別子を元に、MCUに関連付けたPinインスタンスを生成する。
 * Pinクラスは、MCUの物理ピン単位に、操作インタフェイスを定義する。
 * クラスは、1ピンのポートとして実装する。
 * @name LPC1769.Pin
 * @constructor
 * @param {object as LPC1769.Mcu} i_mcu
 * インスタンスを結びつけるMcuオブジェクト。
 * @param {object as ピン識別子} i_pin
 * 生成するピンのピン識別子。
 * @param {object as associative array} i_opt
 * インスタンス生成と同時にsetOpt関数で設定する値。省略時は無視する。
 * 詳細はsetOpt関数を参照。
 * このクラスは、上位クラスへピンの基本的な操作機能を提供する為のものであり、ユーザが直接使用することは（あまり）ない。
 * この関数は、MiMicの管理しているピン（ENET_?）も操作することが出来るが、操作してしまうとMiMicRemoteMcuとのコネクションが破壊されるので、注意すること。
 * @example
 * //create pin instance at P0[0]
 * var mcu=new LPC1769.Mcu(“192.168.0.39”);
 * var pin=new LPC1769.Pin(mcu,LPCXpresso.P0[0]);
 */
DEV.Pin=function Pin(i_mcu,i_pin,i_opt)
{
	try{
		if(isUndef(i_pin)){
			throw new MiMicException(EE.INVALID_PIN);
		}
		//1ピンのポートとして実装する。
		this._port=new DEV.Port(i_mcu,[i_pin],isUndef(i_opt)?undefined:[i_opt]);
	}catch(e){
		throw new MiMicException(e);
	}
}
DEV.Pin.prototype=
{	
	_port:null,
	/**
	 * ピンにオプションパラメータをセットする。
	 * 関数は、物理ピンのレジスタに、i_optで与えられる値を設定する。
	 * 関数は、レジスタの位置に合わせてパラメータ値を自動的にシフトする。
	 * @name LPC1769.Pin#setOpt
	 * @function
	 * @param {object as associative array} i_opt
	 * ピンコンフィグレーションのパラメタである。必要な値を格納した連想配列で指定する。
	 * 全ての値を省略することは出来ない。連想配列のメンバは以下の通り。
	 * <pre>{sel,mode,od}</pre>
	 * <ul>
	 * <li>sel:int - 2bitのint値。 PINSEL? レジスタに指定する値。UM10360 Chapter8.LPC17xx Pin connect block を参照。</li>
	 * <li>mode:int - 1bitのint値。PINMODE? レジスタに指定する値。UM10360 Chapter8.LPC17xx Pin connect block を参照。</li>
	 * <li>od:int - 1bitのbit値　PINMODE_OD? レジスタに指定する値。UM10360 Chapter8.LPC17xx Pin connect block を参照。</li>
	 * </ul>
	 * @example
	 * //set GPIO,mode=1,open drain=0
	 * var mcu=new LPC1769.Mcu(“192.168.0.39”);
	 * var pin=new LPC1769.Pin(mcu,LPCXpresso.P2[3]);
	 * pin.setOpt({sel:0,mode:1,od:0});	 
	 */
	setOpt:function setOpt(i_opt)
	{
		try{
			//portのsetOptで合成
			this._port.setOpts([i_opt]);
		}catch(e){
			throw new MiMicException(e);
		}
		return;
	}
}

}());