/*********************************************************************************
 * PROJECT: MiMic
 * --------------------------------------------------------------------------------
 *
 * This file is part of MiMic
 * Copyright (C)2011 Ryo Iizuka
 *
 * MiMic is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by　the Free Software Foundation, either version 3 of the　License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * For further information please contact.
 *	http://nyatla.jp/
 *	<airmail(at)ebony.plala.or.jp> or <nyatla(at)nyatla.jp>
 *
 *********************************************************************************/
#include "NyLPC_cModMiMicSetting.h"
#include "NyLPC_stdlib.h"
#include "NyLPC_cHttpModUtils_protected.h"
#include "NyLPC_http.h"
#include "NyLPC_mimicVm.h"
#include "NyLPC_flash.h"
#include "../NyLPC_cHttpdConnection_protected.h"
#include "../../NyLPC_cNet.h"

#define MOD_VERSION "ModMiMicSetting/1.0"
#define SIZE_OF_SETUP_PARAM 7
struct TModMiMicSettingRequest
{
	struct NyLPC_THttpBasicHeader super;
	NyLPC_TUInt8 _content_id;
	//解析用
	NyLPC_TUInt8 _qery_name_id;
	NyLPC_TUInt8 _astate;
	NyLPC_TInt16 _prefix_len;
	NyLPC_TcStr_t _tstr;
	NyLPC_TChar _tstr_buf[16];
	/** 文字列のパーサ*/
	NyLPC_TcMiMicDbCompiler_t _binparser;
	union{
		struct{
			/**pパラメータ。最大長さは16。長さ6であること。
			 * [0]:MACAddrの、[0][1][2][3]
			 * [1]:MACAddrの、[4][5][X][X]
			 * [2]:IPAddr(Networkorder)
			 * [3]:Subnetmask(Networkorder)
			 * [4]:Defaultgateway
			 * [5]:[0:port0][1:port1][X][X]
			 * [6]:accessmode [0:AC1][1:AC2][X][X]
			 */
			NyLPC_TUInt32 param_buf[SIZE_OF_SETUP_PARAM];
			NyLPC_TUInt16 param_len;
			NyLPC_TUInt32 cval;//コマンド値
		}setup;
		struct{
			/**
			 * 不明な名前の場合は、ここに名前をコピー
			 */
			NyLPC_TChar path[32];
		}unknown;
	}content;
};


static NyLPC_TBool messageHandler(NyLPC_TcHttpBasicHeaderParser_t* i_inst,const NyLPC_TChar* i_name,NyLPC_TChar i_c,struct NyLPC_THttpBasicHeader* o_out)
{
	(void)i_inst;
	(void)i_name;
	(void)i_c;
	return NyLPC_TBool_TRUE;
}

#define ST_PARSE_PATH 1
#define ST_PARSE_QUERY_NAME 2
#define ST_PARSE_QUERY_VALUE 3		//Query読み出し中
#define ST_PARSE_QUERY_VALUE_P 4
#define ST_PARSE_QUERY_VALUE_C 5
/**
 * コンテンツID定義(コンテンツ名に対応)
 */
#define CONTENT_ID_UNKNOWN 1
#define CONTENT_ID_SETUP   2
#define CONTENT_ID_INDEX   3
#define CONTENT_ID_STATUS  4
#define CONTENT_ID_CSS     5
#define CONTENT_ID_LOGO    6

#define QNAME_ID_P	4
#define QNAME_ID_C	5
#define QNAME_ID_UNKNOWN 0

#define QVAL_C_GET 1
#define QVAL_C_UPDATE 2
#define QVAL_C_UNKNOWN 0


static const struct NyLPC_TTextIdTbl url_tbl[]=
{
	{"setup.api",CONTENT_ID_SETUP},
	{NULL,CONTENT_ID_UNKNOWN}
};

static const struct NyLPC_TTextIdTbl qname_id_table[]=
{
	{"p",QNAME_ID_P},
	{"c",QNAME_ID_C},
	{NULL,QNAME_ID_UNKNOWN}
};




static NyLPC_TBool urlHandler(NyLPC_TcHttpBasicHeaderParser_t* i_inst,NyLPC_TChar i_c,struct NyLPC_THttpBasicHeader* o_out)
{

	struct TModMiMicSettingRequest* out=(struct TModMiMicSettingRequest*)o_out;
	//読み飛ばし
	if(out->_prefix_len<0){
		out->_prefix_len++;
		return NyLPC_TBool_TRUE;//読み飛ばし
	}
	if(out->_astate==ST_PARSE_PATH){
		if(i_c!='\0' && i_c!='?'){
			if(!NyLPC_cStr_put(&(out->_tstr),i_c)){
				NyLPC_OnErrorGoto(ERROR);
			}
			return NyLPC_TBool_TRUE;
		}
		out->_content_id=NyLPC_TTextIdTbl_getMatchId(NyLPC_cStr_str(&(out->_tstr)),url_tbl);
		switch(out->_content_id)
		{
		case CONTENT_ID_SETUP:
			out->content.setup.param_len=0;
			out->content.setup.cval=QVAL_C_UNKNOWN;
			break;
		default:
			break;
		}
		NyLPC_cStr_clear(&(out->_tstr));
		out->_astate=ST_PARSE_QUERY_NAME;//クエリ名解析へ
		return NyLPC_TBool_TRUE;
	}
	switch(out->_content_id)
	{
	case CONTENT_ID_SETUP:
		switch(out->_astate){
		case ST_PARSE_QUERY_NAME:
			if(i_c!='\0' && i_c!='&' && i_c!='='){
				if(!NyLPC_cStr_put(&(out->_tstr),i_c)){
					NyLPC_OnErrorGoto(ERROR);
				}
				return NyLPC_TBool_TRUE;
			}
			//Query確定。
			out->_qery_name_id=NyLPC_TTextIdTbl_getMatchId(NyLPC_cStr_str(&(out->_tstr)),qname_id_table);
			NyLPC_cStr_clear(&(out->_tstr));
			//クエリ値がある場合
			switch(out->_qery_name_id){
			case QNAME_ID_P:
				out->_astate=ST_PARSE_QUERY_VALUE_P;//MIMICBCのDBパラメータパーサを借用。
				out->content.setup.param_len=0;
				break;
			case QNAME_ID_C:
				out->_astate=ST_PARSE_QUERY_VALUE_C;
				break;
			default:
				out->_astate=ST_PARSE_QUERY_VALUE;
				break;
			}
			return NyLPC_TBool_TRUE;
		case ST_PARSE_QUERY_VALUE:
			//未知のクエリは無視
			if(i_c!='\0' && i_c!='&'){
				return NyLPC_TBool_TRUE;
			}
			//クエリ値解析完了
			out->_astate=ST_PARSE_QUERY_NAME;
			return NyLPC_TBool_TRUE;
		case ST_PARSE_QUERY_VALUE_C:
			if(i_c!='\0' && i_c!='&'){
				if(!NyLPC_cStr_put(&(out->_tstr),i_c)){
					NyLPC_OnErrorGoto(ERROR);
				}
				return NyLPC_TBool_TRUE;
			}
			if(NyLPC_cStr_isEqual(&out->_tstr,"get")){
				out->content.setup.cval=QVAL_C_GET;
			}else if(NyLPC_cStr_isEqual(&out->_tstr,"update")){
				out->content.setup.cval=QVAL_C_UPDATE;
			}else{
				NyLPC_OnErrorGoto(ERROR);
			}
			out->_astate=ST_PARSE_QUERY_NAME;
			NyLPC_cStr_clear(&(out->_tstr));
			return NyLPC_TBool_TRUE;
		case ST_PARSE_QUERY_VALUE_P:
			if(i_c!='\0' && i_c!='&'){
				if(out->content.setup.param_len>=SIZE_OF_SETUP_PARAM)
				{
					NyLPC_OnErrorGoto(ERROR);
				}
				switch(NyLPC_cMiMicDbCompiler_compileFragment2(&(out->_binparser),i_c,out->content.setup.param_buf+out->content.setup.param_len))
				{
				case NyLPC_TcMiMicDbCompiler_RET_CONTINUE:
					break;
				case NyLPC_TcMiMicDbCompiler_RET_OK:
					out->content.setup.param_len++;
					break;
				case NyLPC_TcMiMicDbCompiler_RET_ERROR:
				default:
					//ERROR
					NyLPC_OnErrorGoto(ERROR);
				}
				return NyLPC_TBool_TRUE;
			}
			//区切りのいいところで終わってる？
			if(NyLPC_cMiMicDbCompiler_hasFragment(&(out->_binparser))){
				//ERROR
				NyLPC_OnErrorGoto(ERROR);
			}
			//終端しているなら、次のクエリへ
			out->_astate=ST_PARSE_QUERY_NAME;
			NyLPC_cStr_clear(&(out->_tstr));
			return NyLPC_TBool_TRUE;
		default:
			break;
		}
		NyLPC_OnErrorGoto(ERROR);
	default:
		NyLPC_OnErrorGoto(ERROR);
	}
	return NyLPC_TBool_TRUE;
ERROR:
	return NyLPC_TBool_FALSE;
}
/**
 * デフォルトハンドラ
 */
static const struct NyLPC_TcHttpBasicHeaderParser_Handler handler=
{
	messageHandler,
	urlHandler
};


/**
 * コンストラクタ。
 */
void NyLPC_cModMiMicSetting_initialize(NyLPC_TcModMiMicSetting_t* i_inst,const NyLPC_TChar* i_ref_root_path)
{
	NyLPC_cModRomFiles_initialize(&i_inst->super,i_ref_root_path,NULL,0);
}
void NyLPC_cModMiMicSetting_finalize(NyLPC_TcModMiMicSetting_t* i_inst)
{
	NyLPC_cModRomFiles_finalize(&i_inst->super);
}
/**
 * モジュールがコネクションをハンドリングできるかを返します。
 */
NyLPC_TBool NyLPC_cModMiMicSetting_canHandle(NyLPC_TcModMiMicSetting_t* i_inst,NyLPC_TcHttpdConnection_t* i_connection)
{
	return NyLPC_cModRomFiles_canHandle(&i_inst->super,i_connection);
}



static void setup_proc(NyLPC_TcHttpdConnection_t* i_connection,struct TModMiMicSettingRequest* i_req);

/**
 * モジュールを実行します。
 */
NyLPC_TBool NyLPC_cModMiMicSetting_execute(NyLPC_TcModMiMicSetting_t* i_inst,NyLPC_TcHttpdConnection_t* i_connection)
{
	struct TModMiMicSettingRequest header;
	NyLPC_TcHttpBasicHeaderParser_t parser;
	//connectonの状態を確認
	if(!NyLPC_cHttpdConnection_getReqStatus(i_connection)==NyLPC_cHttpdConnection_ReqStatus_REQPARSE)
	{
		NyLPC_OnErrorGoto(Error1);
	}
	//リクエストParse済へ遷移(この関数の後はModが責任を持ってリクエストを返却)
	NyLPC_cHttpdConnection_setReqStatusParsed(i_connection);

	//URL解析の準備
	header._prefix_len=-((NyLPC_TInt16)strlen(i_inst->super._ref_root_path)+2);
	header._astate=ST_PARSE_PATH;
	NyLPC_cStr_initialize(&header._tstr,header._tstr_buf,16);
	NyLPC_cMiMicDbCompiler_initialize(&header._binparser);

	NyLPC_cHttpBasicHeaderParser_initialize(&parser,&handler);
	//プリフェッチしたデータを流す
	NyLPC_cHttpBasicHeaderParser_parseInit(&parser,&(header.super));
	NyLPC_cHttpdConnection_pushPrefetchInfo(i_connection,&parser,&header.super);
	//後続をストリームから取り込む
	if(!NyLPC_cHttpBasicHeaderParser_parseStream(&parser,NyLPC_cHttpdConnection_refStream(i_connection),&(header.super))){
		NyLPC_OnErrorGoto(Error2);
	}
	if(!NyLPC_cHttpBasicHeaderParser_parseFinish(&parser,&(header.super))){
		NyLPC_OnErrorGoto(Error2);
	}
	//GETのみ
	if(NyLPC_cHttpdConnection_getMethod(i_connection)!=NyLPC_THttpMethodType_GET)
	{
		NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,405,"text/html",0,NULL);
		NyLPC_OnErrorGoto(Error1);
	}
	//Connection Modeの設定 1.1 && !closeの場合はCONTINUE
	if(header.super.connection!=NyLPC_THttpMessgeHeader_Connection_CLOSE && header.super.startline.req.version==NyLPC_THttpVersion_11)
	{
		NyLPC_cHttpdConnection_setConnectionMode(i_connection,NyLPC_TcHttpdConnection_CONNECTION_MODE_CONTINUE);
	}
	//CGIの実行
	switch(header._content_id)
	{
	case CONTENT_ID_SETUP:
		setup_proc(i_connection,&header);
		break;
	case CONTENT_ID_UNKNOWN:
	default:
		NyLPC_OnErrorGoto(Error2);
	}
	NyLPC_cHttpBasicHeaderParser_finalize(&parser);
	NyLPC_cMiMicDbCompiler_finalize(&header._binparser);
	NyLPC_cStr_finalize(&(header._tstr));
	return NyLPC_TBool_TRUE;
Error2:
	NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,500,"text/html",0,NULL);
	NyLPC_cHttpBasicHeaderParser_finalize(&parser);
	NyLPC_cMiMicDbCompiler_finalize(&header._binparser);
	NyLPC_cStr_finalize(&(header._tstr));
Error1:
	return NyLPC_TBool_FALSE;
}


static void setup_proc(NyLPC_TcHttpdConnection_t* i_connection,struct TModMiMicSettingRequest* i_req)
{
	NyLPC_TBool ret;
	const struct NyLPC_TMimicConfigulation* config;
	struct NyLPC_TMimicConfigulation cfg_image;

	switch(i_req->content.setup.cval){
	case QVAL_C_GET:
		if(!NyLPC_cHttpdModUtils_sendJsonHeader(i_connection)){
			NyLPC_OnErrorGoto(Error);
		}
		config=NyLPC_cMiMicConfiglation_loadFromFlash();
		//JSONを書く。
		if(!NyLPC_cHttpdConnection_sendResponseBodyF(i_connection,
			"{\"application\":\""MOD_VERSION";%s\",\"mac00010203\":%u,\"mac0405xxxx\":%u,\"ip\":%u,\"mask\":%u,\"droute\":%u,\"port\":%u,\"access\":%u}",
			NyLPC_cNet_PlatformName,
			config->mac_00_01_02_03,
			config->mac_04_05_xx_xx,
			config->ipv4_addr_net,
			config->ipv4_mask_net,
			config->ipv4_drut_net,
			config->ipv4_port,
			config->accessmode
			)){
			NyLPC_OnErrorGoto(Error);
		}
		break;
	case QVAL_C_UPDATE:
		//check parameter length
		if(i_req->content.setup.param_len!=SIZE_OF_SETUP_PARAM)
		{
			NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,500,"text/html",0,NULL);
		}else{
			//パラメータ→ROMイメージ変換
			cfg_image.fast_boot=0xffffffff;
			cfg_image.mac_00_01_02_03=(i_req->content.setup.param_buf[0]);
			cfg_image.mac_04_05_xx_xx=(i_req->content.setup.param_buf[1]&0xffff0000);
			cfg_image.ipv4_addr_net  =i_req->content.setup.param_buf[2];
			cfg_image.ipv4_mask_net  =i_req->content.setup.param_buf[3];
			cfg_image.ipv4_drut_net  =i_req->content.setup.param_buf[4];
			cfg_image.ipv4_port =(NyLPC_TUInt16)(i_req->content.setup.param_buf[5]>>16);
			cfg_image.accessmode=(i_req->content.setup.param_buf[6]);
			//一応確認。
			if((cfg_image.ipv4_port==0)|| (cfg_image.accessmode & 0xFEFE0000)!=0x00000000){
				NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,500,"text/html",0,NULL);
			}else{
				//FreeRTOSの停止
				NyLPC_cIsr_enterCritical();
				//Flashへの書き込み
				ret=NyLPC_cMiMicConfiglation_updateConfigulation(&cfg_image);
				//FreeRTOSの復帰
				NyLPC_cIsr_exitCritical();
				if(!ret){
					NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,500,"text/html",0,NULL);
				}else{
					if(!NyLPC_cHttpdModUtils_sendJsonHeader(i_connection)){
						NyLPC_OnErrorGoto(Error);
					}
					NyLPC_cHttpdConnection_sendResponseBodyF(i_connection,
						"{\"application\":\""MOD_VERSION";%s\",\"result\":%u}",
						NyLPC_cNet_PlatformName,ret?0x00000000:0x80000000);
				}
			}
		}
		//JSONを書く。
		break;
	default:
		NyLPC_cHttpdConnection_sendResponseHeader2(i_connection,403,"text/html",0,NULL);
		NyLPC_OnErrorGoto(Error);
		break;
	}
	return;
Error:
	return;
}
