# -*- coding: utf-8 -*-

require File.dirname(__FILE__) + '/../spec_helper.rb'

module Mint

  describe 'build' do
    context 'one node' do
      [
        %w[addition +],
        %w[subtraction -],
        %w[multiple *],
        %w[division div],
        %w[fraction /],
      ].each do |name, op|
        it name do
          r, l = 5, 8
          left = LiteralNode.new(l)
          right = LiteralNode.new(r)
          node_class = Mint.const_get("#{name.capitalize}Node")
          expression = node_class.new(left, right)
          expression.to_s.should == "#{l} #{op} #{r}"
        end
      end
    end

    context 'two node' do
      before(:all) do
        @left  = LiteralNode.new(2)
        @right = LiteralNode.new(5)
      end

      %w[
        addition
        subtraction
        multiple
        division
        fraction
      ].permutation(3).each do |l, c, r, op|
        op = lambda {|n|
          case n
          when 'addition'    then '+'
          when 'subtraction' then '-'
          when 'multiple'    then '*'
          when 'division'    then 'div'
          when 'fraction'    then '/'
          else; 'unknown'
          end
        }
        it "#{l} #{c} #{r}" do
          left_node_class = Mint.const_get("#{l.capitalize}Node")
          left  = left_node_class.new(@left, @right)
          right_node_class = Mint.const_get("#{r.capitalize}Node")
          right = right_node_class.new(@left, @right)
          center_node_class = Mint.const_get("#{c.capitalize}Node")
          expression = center_node_class.new(left, right)
          expression.to_s.should == "#{left} #{op[c]} #{right}"
        end
      end

      context 'parent' do
        before do
          @left = LiteralNode.new(1)
          @right = LiteralNode.new(1)
          @expr = ExpressionNode.new(@left, @right)
        end
        it { @left.parent.should == @expr }
        it { @right.parent.should == @expr }
      end
    end

    context 'with parenthesis' do
      before(:all) do
        @left  = LiteralNode.new(1)
        @right = LiteralNode.new(2)
      end
      it 'shows parenthesis' do
        left  = AdditionNode.new(@left, @right)
        right = AdditionNode.new(@left, @right)
        left.parenthesis = true
        expression = MultipleNode.new(left, right)
        expression.to_s.should == "(#{@left} + #{@right}) * #{right}"
      end
    end

    context 'with minus' do
      before(:all) do
        @left  = LiteralNode.new(1)
        @right = LiteralNode.new(2)
      end
      it 'shows minus (expression)' do
        left  = AdditionNode.new(@left, @right)
        right = AdditionNode.new(@left, @right)
        left.minus = true
        expression = MultipleNode.new(left, right)
        expression.to_s.should == "(-(#{@left} + #{@right})) * #{right}"
      end
    end
  end

  describe ExpressionNode do
    subject { ExpressionNode.new(@left, @right) }
    before(:all) do
      @left     = LiteralNode.new(2)
      @right    = LiteralNode.new(6)
      @operator = :unknown
    end
    it 'has left hand number' do
      subject.left.should == @left
    end
    it 'has right hand number' do
      subject.right.should == @right
    end
    it 'has operator' do
      subject.operator.should == @operator
    end
    it 'shows itself as string' do
      subject.to_s.should == "#{@left} #{@operator} #{@right}"
    end

    context 'options' do
      context 'minus' do
        before { subject.minus = true }
        it { subject.minus.should be_true }
        it { subject.should be_minu }
      end
      context 'parenthesis' do
        before { subject.parenthesis = true }
        it { subject.parenthesis.should be_true }
        it { subject.should be_parenthesis }
      end
    end
  end

  context 'expresion nodes' do
    before do
      @left = LiteralNode.new(1)
      @right = LiteralNode.new(1)
    end

    describe AdditionNode do
      subject { AdditionNode.new(@left, @right) }
      it 'operator is :+' do
        subject.operator.should == :+
      end
    end

    describe SubtractionNode do
      subject { SubtractionNode.new(@left, @right) }
      it 'operator is :-' do
        subject.operator.should == :-
      end
    end

    describe MultipleNode do
      context 'general' do
        subject { MultipleNode.new(@left, @right) }
        it 'operator is :*' do
          subject.operator.should == :*
        end
      end
      [
        LiteralNode.new('x'),
        RootNode.new('2'),
        FactorialNode.new(LiteralNode.new('x'), LiteralNode.new('2')),
      ].each do |right|
        context "with #{right.to_s}" do
          subject { MultipleNode.new(LiteralNode.new(2), right) }
          it { subject.to_latex.should match(/\A2#{Regexp.escape(right.to_latex)}\z/) }
        end
      end
    end

    describe DivisionNode do
      subject { DivisionNode.new(@left, @right) }
      it 'operator is :div' do
        subject.operator.should == :div
      end
    end

    describe FractionNode do
      before do
        @left = LiteralNode.new(5); @left.minus = true
        @right = LiteralNode.new(5)
      end
      subject { FractionNode.new(@left, @right) }
      it 'operator is /' do
        subject.operator.should == :/
      end
      it do
        fraction = FractionNode.new(@left, @right)
        fraction.to_s.should == '-5 / 5'
      end
      it do
        left = AdditionNode.new(@right, @right)
        left.minus = true
        fraction = FractionNode.new(left, @right)
        fraction.to_s.should == '-(5 + 5) / 5'
      end
    end
  end

  describe LiteralNode do
    [true, false].each do |minus|
      [
        'x', 'y', 'z',
        1, 2, 3, 5, 8, 1000
      ].each do |num|
        context "#{num}(#{minus})" do
          def show(num, minus)
            return "(-#{num})" if minus
            num.to_s
          end
          subject { LiteralNode.new(num) }
          before { subject.minus = minus }
          it { subject.value.should == num }
          it { subject.to_s.should == show(num, minus) }
          it { subject.to_latex.should == show(num, minus) }
        end
      end
    end
  end

  describe DecimalNode do
    subject { DecimalNode.new(@number) }
    before(:all) {
      @number = '0.473201'
      @normalized = '0.473'
    }
    it 'has own number' do
      subject.value.should == @number
    end
    it 'shows itself as string' do
      subject.to_s.should == "#{@normalized}"
    end
  end

  describe FactorialNode do
    subject { FactorialNode.new(@number, @power) }
    before(:all) { @number, @power = 5, 2 }
    it 'has own value' do
      subject.value.should == @number
      subject.power.should == @power
    end
    it 'shows itself as string' do
      subject.to_s.should == "#{@number}^#{@power}"
    end
  end

  describe RootNode do
    subject { RootNode.new(@number) }
    before(:all) { @number = 2 }
    it 'has own number' do
      subject.value.should == @number
    end
    it 'shows itself as string' do
      subject.to_s.should == "sqrt(#{@number})"
    end
  end
end

