# -*- coding: utf-8 -*-

require File.dirname(__FILE__) + '/../spec_helper.rb'

module Mint::Generator
  describe SquareRootArithmetic do

    subject { SquareRootArithmetic.new }

    before(:all) { @opt_pattern = %r![\*]! }

    it_should_behave_like 'Arithmetic'

    context 'create square root number' do

      before(:all) { @defaults = { :min => 2, :max => 2, :single_term_max => 1 } }

      [
        [{:min => 0, :max => 0}, /\A[2-9]\z/],
        [{:min => 1, :max => 1}, /\A[2-9]\z/],
        [{}, /\Asqrt\(2\)\z/],
        [{:use_coefficient => true}, /\A[2-9]?sqrt\(2\)\z/],

      ].each do |options, pattern|
        context options do
          before do
            settings = @defaults.merge(options)
            subject.__send__(:options=, settings)
            @root_number = subject.__send__(:create_square_root_number)
          end
          it { @root_number.should match(pattern) }
        end
      end
    end

    context 'coefficient' do
      {
        :use => [
          true,
          [
            [-1, :should_not],
            [0, :should_not],
            [1, :should_not],
            [2, :should_not]]
        ],
        :not_use => [
          false,
          [
            [-1, :should],
            [0, :should_not],
            [1, :should_not],
            [2, :should]]
        ],
      }.each do |name, (option, params)|
        context name do
          before { subject.__send__(:options=, :use_coefficient => option) }
          params.each do |num, meth|
            it { subject.__send__(:coefficient, num).__send__(meth, be_nil) }
          end
        end
      end
    end

    context 'create expression' do

      context 'single term' do
        before do
          settings = { :min => 2, :single_term_max => 1, :single_term_min => 1 }
          @problem = subject.generate(settings)
          @expected_pattern = /\Asqrt\(\d\d?\) #{@opt_pattern} sqrt\(\d\d?\)\z/
        end
        it { @problem.first.should match(@expected_pattern) }
      end

      context 'some term' do

        context 'min 2' do
          before(:all) { @common_settings = { :min => 2, :single_term_min => 2, :single_term_max => 2, } }

          term_pattern = /\(sqrt\(\d\d?\) [\-\+] sqrt\(\d\d?\)\)/
          [
            [{}, /\A#{term_pattern} [\*] #{term_pattern}\z/],
            [{:operators => ['+']}, /\A#{term_pattern} \+ #{term_pattern}\z/],
            [{:operators => ['-']}, /\A#{term_pattern} \- #{term_pattern}\z/],
            [{:use_power => true}, /\A#{term_pattern}(?:\^2)? [\*] #{term_pattern}(?:\^2)?\z/],

          ].each do |options, pattern|
            context options.inspect do
              before do
                settings = @common_settings.merge(options)
                @problem = subject.generate(settings)
              end
              it { @problem.first.should match(pattern) }
            end
          end
        end

        context 'min 11' do
          before(:all) { @common_settings = { :min => 11, :single_term_min => 2, :single_term_max => 2, } }

          term_pattern = /\(sqrt\(\d\d?\) [\-\+] sqrt\(\d\d?\)\)/
          [
            # MEMO: have_common_prime_divisor only runs when use_power is true.
            [{:use_power => true}, /\A#{term_pattern}(?:\^2)? [\*] #{term_pattern}(?:\^2)?\z/],

          ].each do |options, pattern|
            context options.inspect do
              before do
                settings = @common_settings.merge(options)
                @problem = subject.generate(settings)
              end
              it { @problem.first.should match(pattern) }
              it do
                @problem.first.scan(/sqrt\((\d*)\)/).each_cons(2) do |pair|
                  pair = pair.flatten.map(&:to_i)
                  subject.__send__(:have_common_prime_divisor?, pair).should be_true
                end
              end
            end
          end
        end

        context 'max 1 with stub' do

          before(:all) do
            @common_settings = {}
            subject.stub(:create_integer).and_return(2, 5, 1, 2, 2, 1, 5, 2)
          end

          term_pattern = /\(\d\d? [\-\+] sqrt\(\d\d?\)\)/
          [
            [{}, /\A#{term_pattern} [\*] #{term_pattern}\z/],
            [{:operators => ['+']}, /\A#{term_pattern} \+ #{term_pattern}\z/],
            [{:operators => ['-']}, /\A#{term_pattern} \- #{term_pattern}\z/],
            [{:use_power => true}, /\A#{term_pattern}(?:\^2)? [\*] #{term_pattern}(?:\^2)?\z/],

          ].each do |options, pattern|
            context options.inspect do
              before do
                settings = @common_settings.merge(options)
                @problem = subject.generate(settings)
              end
              it { @problem.first.should match(pattern) }
            end
          end
        end

        context 'max 1' do
          before(:all) { @common_settings = { :max => 1 } }

          term_pattern = /\(\d\d? [\-\+] sqrt\(\d\d?\)\)/
          [
            [{}, /\A#{term_pattern} [\*] #{term_pattern}\z/],
            [{:operators => ['+']}, /\A#{term_pattern} \+ #{term_pattern}\z/],
            [{:operators => ['-']}, /\A#{term_pattern} \- #{term_pattern}\z/],
            [{:use_power => true}, /\A#{term_pattern}(?:\^2)? [\*] #{term_pattern}(?:\^2)?\z/],

          ].each do |options, pattern|
            context options.inspect do
              before do
                settings = @common_settings.merge(options)
                @problem = subject.generate(settings)
              end
              it { @problem.first.should match(pattern) }
            end
          end
        end

        context 'max 1 and term_number 3' do
          before(:all) do
            @common_settings = {
              :max => 1,
              :single_term_min => 3,
              :single_term_max => 3,
            }
          end

          term_pattern = /\(\d\d? [\-\+] \d\d? [\-\+] sqrt\(\d\d?\)\)/
          [
            [{}, /\A#{term_pattern} [\*] #{term_pattern}\z/],
            [{:operators => ['+']}, /\A#{term_pattern} \+ #{term_pattern}\z/],
            [{:operators => ['-']}, /\A#{term_pattern} \- #{term_pattern}\z/],
            [{:use_power => true}, /\A#{term_pattern}(?:\^2)? [\*] #{term_pattern}(?:\^2)?\z/],

          ].each do |options, pattern|
            context options.inspect do
              before do
                settings = @common_settings.merge(options)
                @problem = subject.generate(settings)
              end
              it { @problem.first.should match(pattern) }
            end
          end
        end
      end
    end
  end
end

