module ActsAsProblem
  def self.included(base)
    base.extend(ClassMethods)
  end

  module ClassMethods
    def acts_as_problem(options = {})
      default_options = { :operators => false, :term_number => false }
      options = default_options.update(options)
      column :unit_id
      column :level, :type => :integer
      column :amount, :type => :integer
      column :lock_version, :type => :integer

      validates_presence_of :unit_id
      validates_numericality_of :unit_id, :only_integer => true

      validates_presence_of :level
      validates_inclusion_of :level, :in => 0..100

      validates_presence_of :amount
      validates_inclusion_of :amount, :in => 1..100

      if options[:operators] == true
        options[:operators] = %w[plus minus times div]
      end
      if options[:operators]
        options[:operators].each do |o|
          column "operators_#{o}".to_sym, :type => :boolean
        end
      end

      if options[:term_number]
        column :term_number, :type => :integer
        validates_presence_of     :term_number
        validates_numericality_of :term_number, :only_integer => true, :greater_than_or_equal_to => 2, :allow_nil => true
      end

      include InstanceMethods
    end
  end

  module InstanceMethods

    def unit
      Unit.find(unit_id)
    end

    def attributes_for_model
      {
        :unit_id      => unit_id,
        :level        => level,
        :amount       => amount,
        :options      => options_hash.to_json,
        :lock_version => lock_version,
      }
    end

    def check_min_max(min_column, max_column)
      min = __send__(min_column)
      max = __send__(max_column)
      return if min.nil? || max.nil?
      if max < min
        errors.add_to_base(_("%{fn1} must be greater than %{fn2}." %
                             { :fn1 => min_column.to_s.humanize,
                               :fn2 => max_column.to_s.humanize }))
      end
    end

    def check_operators
      if self.operators.empty?
        errors.add_to_base _("No operators specified.")
      end
    end

    def split_x(x)
      x.split(/,\s*/) if x
    end

    def operators
      arr = []
      arr << "+" if self.class.column_names.include?("operators_plus") && operators_plus
      arr << "-" if self.class.column_names.include?("operators_minus") && operators_minus
      arr << "*" if self.class.column_names.include?("operators_times") && operators_times
      arr << "div" if self.class.column_names.include?("operators_div") && operators_div
      arr << "/" if self.class.column_names.include?("operators_fraction") && operators_fraction
      arr
    end

    def operators=(arr)
      self.operators_plus = arr.include?("+") if self.class.column_names.include?("operators_plus")
      self.operators_minus = arr.include?("-") if self.class.column_names.include?("operators_minus")
      self.operators_times = arr.include?("*") if self.class.column_names.include?("operators_times")
      self.operators_div = arr.include?("div") if self.class.column_names.include?("operators_div")
      self.operators_fraction = arr.include?("/") if self.class.column_names.include?("operators_fraction")
    end
  end
end

ActiveForm.__send__(:include, ActsAsProblem)
