//
// ResourceManager.cpp
//

#include <fstream>
#include "ResourceManager.hpp"
#include "../common/Logger.hpp"
#include <boost/filesystem.hpp>
#include "unicode.hpp"

int ResourceManager::default_font_handle_ = -1;
int ResourceManager::default_font_handle()
{
    if (default_font_handle_ < 0) {
        //TCHAR font_name[] = CHAT_FONT_NAME;
        const TCHAR* font_name = CHAT_FONT_NAME;
        default_font_handle_ = CreateFontToHandle(font_name, CHAT_FONT_SIZE, CHAT_FONT_THICK, CHAT_FONT_TYPE);
    }

    return default_font_handle_;

}

int ResourceManager::default_font_size()
{
    return CHAT_FONT_SIZE;
}

std::unordered_map<std::string, ImageHandle> ResourceManager::graph_handles_;
std::unordered_map<std::string, std::vector<ImageHandle>> ResourceManager::div_graph_handles_;
ImageHandle ResourceManager::LoadCachedGraph(const std::string& filename)
{
    ImageHandle handle;
    if(graph_handles_.find(filename) == graph_handles_.end()) {
        handle = ImageHandle(DxLib::LoadGraph(filename.c_str()));
        graph_handles_[filename] = handle;
    } else {
        handle = graph_handles_[filename];
    }
    return ImageHandle(handle);
}

void ResourceManager::ClearCache()
{
    graph_handles_.clear();
    div_graph_handles_.clear();

    model_names_.clear();
    model_handles_.clear();
    model_name_tree_.clear();

    InitGraph();
    MV1InitModel();
}

std::unordered_map<std::string, std::string> ResourceManager::model_names_;
std::unordered_map<std::string, ModelHandle> ResourceManager::model_handles_;
ptree ResourceManager::model_name_tree_;
void ResourceManager::BuildModelFileTree()
{
    using namespace boost::filesystem;
    using namespace std;

    model_name_tree_.clear();

    path p("./resources/models");

    try {
        if (exists(p) && is_directory(p)) {
            for (auto it_dir = directory_iterator(p); it_dir != directory_iterator(); ++it_dir) {
                if (is_directory(*it_dir)) {
                    path json_path = it_dir->path() / "info.json";
                    if (exists(json_path)) {

                        path model_path;
                        for (auto it = directory_iterator(*it_dir); it != directory_iterator(); ++it) {
                            auto extension = it->path().extension().string();

                            if (extension == ".mv1" || extension == ".x"
                             || extension == ".pmd" || extension == ".pmx") {
                                model_path = it->path();
                                break;
                            }
                        }

                        if (!model_path.empty()) {
                            ptree pt_json;
                            read_json(json_path.string(), pt_json);

                            std::string name = pt_json.get<std::string>("name", "");
                            pt_json.put<std::string>("modelpath", unicode::sjis2utf8(model_path.string()));

                            if (name.size() > 0) {
                                model_name_tree_.put_child(ptree::path_type(name + ":_info_", ':'), pt_json);
                            }
                        }
                    }
                }
            }
        }
    } catch (const filesystem_error& ex) {
        Logger::Error("%s", ex.what());
    }

}

struct ReadFuncData {
        boost::filesystem3::path model_dir;
        std::list<std::pair<std::string, std::string>> motions;
        std::list<std::pair<std::string, std::string>>::iterator motions_it;
};

int LoadFile(const TCHAR *FilePath, void **FileImageAddr, int *FileSize)
{
    std::ifstream ifs(FilePath, std::ios::binary);

    if (!ifs) {
        *FileImageAddr = nullptr;
        return -1;
    }

    ifs.seekg (0, std::ios::end);
    *FileSize = ifs.tellg();
    ifs.seekg (0, std::ios::beg);

    auto buffer = new char[*FileSize];
    ifs.read(buffer, *FileSize);
    *FileImageAddr = buffer;

    return 0;
}

int FileReadFunc(const TCHAR *FilePath, void **FileImageAddr, int *FileSize, void *FileReadFuncData)
{
    ReadFuncData& funcdata = *static_cast<ReadFuncData*>(FileReadFuncData);

    using namespace boost::filesystem3;
    path filepath(FilePath);

    bool load_motion = false;
    if (funcdata.motions_it != funcdata.motions.end() &&
            filepath.string().find_last_of("L.vmd") != std::string::npos) {

        filepath = funcdata.motions_it->second;
        load_motion = true;
    }

    Logger::Debug("Request %s", filepath.string());

    auto full_path = funcdata.model_dir / filepath;
    if (!exists(full_path)) {
        if (load_motion) {
            full_path = "./resources/motions" / filepath;
        }
    }

    int result = LoadFile(full_path.string().c_str(), FileImageAddr, FileSize);

    if (load_motion) {
        // 読み込み失敗したモーションを削除
        if (result == -1) {
            funcdata.motions_it->second = "";
        }
        ++funcdata.motions_it;
    }

    return result;
}

int FileReleaseFunc(void *MemoryAddr, void *FileReadFuncData)
{
    delete static_cast<char*>(MemoryAddr);
    return 0;
}

ModelHandle ResourceManager::LoadModelFromName(const std::string& name)
{
    if (model_name_tree_.empty()) {
        BuildModelFileTree();
    }

    std::string filepath;
    ptree info;

    auto name_it = model_names_.find(name);
    if (name_it != model_names_.end()) {
        filepath = name_it->second;

    } else {
        ptree p;
        auto path = ptree::path_type(name, ':');

        p = model_name_tree_.get_child(path, ptree());

        // ルートで探索を打ち切る
        while (!path.single()) {
            if (p.empty()) {
                Logger::Debug("EMPTY %s", path.dump());
                // 親ノードを検索
                std::string path_str = path.dump();
                size_t separator_pos = path_str.find_last_of(':');
                assert(separator_pos != std::string::npos);

                path = ptree::path_type(path_str.substr(0, separator_pos), ':');
                p = model_name_tree_.get_child(path, ptree());
            } else {
                info = p.get_child("_info_", ptree());
                if (info.empty()) {
                    Logger::Debug("CHILD_FOUND");
                    // データがない場合は最初の子ノードへ移動
                    p = p.get_child(ptree::path_type(p.front().first, ':'), ptree());
                } else {
                    Logger::Debug("FOUND");
                    break;
                }
            }

        }

        filepath = info.get<std::string>("modelpath", "");
        Logger::Debug("ModelName to filepath %s -> %s", name, filepath);
        model_names_[name] = filepath;
    }

    if (filepath.size() > 0) {
        auto it = model_handles_.find(filepath);
        if (it != model_handles_.end()) {
            return it->second.Clone();
        } else {
            MV1SetLoadModelPhysicsWorldGravity(-100);
            MV1SetLoadModelUsePhysicsMode(DX_LOADMODEL_PHYSICS_LOADCALC);

            ReadFuncData funcdata;
            funcdata.model_dir = boost::filesystem3::path(unicode::utf82sjis(filepath)).parent_path();

            auto motions = info.get_child("character.motions", ptree());
            for (auto it = motions.begin(); it != motions.end(); ++it) {
                funcdata.motions.push_back(
                        std::pair<std::string, std::string>(it->first,
                                unicode::utf82sjis(
                                        it->second.get_value<std::string>())));
            }
            funcdata.motions_it = funcdata.motions.begin();

            void *FileImage ;
            int FileSize ;

            LoadFile(unicode::utf82sjis(filepath).c_str(), &FileImage, &FileSize );
            int handle = MV1LoadModelFromMem( FileImage, FileSize, FileReadFunc, FileReleaseFunc, &funcdata);

            // モーションの名前を設定
            int motion_index = 0;
            for (auto it = motions.begin(); it != motions.end(); ++it) {
                MV1SetAnimName(handle, motion_index, unicode::utf82sjis(it->first).c_str());
                Logger::Debug("Motion  %d", it->first);
                motion_index++;
            }

            auto model_handle = ModelHandle(handle, std::make_shared<ptree>(info));
            model_handles_[filepath] = model_handle;

            Logger::Debug("Model  %d", handle);
            return model_handle.Clone();
        }
    } else {
        return ModelHandle();
    }
}

void ResourceManager::CacheBakedModel()
{

}

ImageHandle::ImageHandle() :
                handle_(-1)
{

}

ImageHandle::ImageHandle(int handle) :
        handle_(handle)
{
}

ImageHandle::operator int() const
{
    return handle_;
}

ModelHandle::ModelHandle(int handle, const std::shared_ptr<ptree>& property, bool async_load) :
        handle_(handle),
        property_(property),
        name_(property_->get<std::string>("name", "")),
        async_load_(async_load)
{

}

ModelHandle ModelHandle::Clone()
{
    if (CheckHandleASyncLoad(handle_) == TRUE) {
        return ModelHandle(MV1DuplicateModel(handle_), property_, true);
    } else {
        return ModelHandle(MV1DuplicateModel(handle_), property_);
    }
}

ModelHandle::ModelHandle() :
        handle_(-1),
        property_(std::make_shared<ptree>())
{

}

ModelHandle::~ModelHandle()
{

}

int ModelHandle::handle() const
{
    return handle_;
}

const ptree& ModelHandle::property() const
{
    return *property_;
}

std::string ModelHandle::name() const
{
    return name_;
}
