//
// ResourceManager.hpp
//

#pragma once

#include <DxLib.h>
#include <string>
#include <array>
#include <map>
#include <memory>
#include <vector>
#include <unordered_map>
#include "unicode.hpp"
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/xml_parser.hpp>

using boost::property_tree::ptree;

class ImageHandle;
class ModelHandle;

class ResourceManager {

    public:
        static void ClearCache();

    // Fonts
    public:
        static int default_font_handle();
        static int default_font_size();

    // Graphics
    public:
        static ImageHandle LoadCachedGraph(const std::string&);

        template <std::size_t AllNum>
        static std::array<ImageHandle, AllNum> LoadCachedDivGraph(const std::string& filename, int XNum, int YNum, int XSize, int YSize)
        {
            if(div_graph_handles_.find(filename) == div_graph_handles_.end()) {
                std::array<int, AllNum> int_handles;
                if(DxLib::LoadDivGraph(filename.c_str(), AllNum, XNum, YNum, XSize, YSize, int_handles.data()) == -1){
                    return std::array<ImageHandle, AllNum>();
                }
                std::vector<ImageHandle> handle_vector;
                for (auto item : int_handles) {
                    handle_vector.push_back(ImageHandle(item));
                }
                div_graph_handles_[filename] = handle_vector;
            }

            std::array<ImageHandle, AllNum> handles;
            const std::vector<ImageHandle>& handle_vector = div_graph_handles_[filename];
            for (std::size_t i = 0; i < AllNum; i++) {
                handles[i] = handle_vector[i];
            }

            return handles;
        }

        // models
        static void BuildModelFileTree();
        static void CacheBakedModel();
        static ModelHandle LoadModelFromName(const std::string&);

    private:
        static int default_font_handle_;
        static std::unordered_map<std::string, ImageHandle> graph_handles_;
        static std::unordered_map<std::string, std::vector<ImageHandle>> div_graph_handles_;

        static ptree model_name_tree_;
        static std::unordered_map<std::string, std::string> model_names_;
        static std::unordered_map<std::string, ModelHandle> model_handles_;

    private:
        constexpr const static TCHAR* CHAT_FONT_NAME = _T("UmePlus P Gothic");
        constexpr static int CHAT_FONT_SIZE = 15;
        constexpr static int CHAT_FONT_THICK = 1;
        constexpr static int CHAT_FONT_TYPE = DX_FONTTYPE_ANTIALIASING;
};

class ImageHandle {
    public:
        ImageHandle();
        explicit ImageHandle(int handle);
        operator int() const;

    private:
        int handle_;
};

class ModelHandle {
    friend class ResourceManager;

    private:
        ModelHandle(int handle, const std::shared_ptr<ptree>& property, bool async_load = false);
        ModelHandle Clone();

    public:
        ModelHandle();
        ~ModelHandle();

        int handle() const;
        const ptree& property() const;
        std::string name() const;

    private:
        int handle_;
        std::shared_ptr<ptree> property_;
        std::string name_;
        bool async_load_;
};
