//
// PortMapper.cpp
//

#include "PortMapper.hpp"
#include "../network/Utils.hpp"
#include "../Logger.hpp"
#include <sstream>
#include <miniupnpc/miniupnpc.h>
#include <miniupnpc/upnpcommands.h>
#include <miniupnpc/upnperrors.h>
//#include <winsock2.h>
#include <boost/format.hpp>

namespace PortMapper {

// ポート開放の最大試行回数
const int MAX_TRY_TIME = 5;

bool TryMap(UPNPUrls& urls, IGDdatas& data, const char* eport,
        const char* iport, const char* iaddr, const char* desc,
        const char* proto, const char* leaseDuration)
{
    int r;
    char intClient[40] = {0};
    char intPort[6] = {0};
    char duration[16] = {0};
    char description[256] = {0};

    std::string formatted_desc = (boost::format("%d at %d:%d") % desc % iaddr % iport).str();

    r = UPNP_GetSpecificPortMappingEntry(urls.controlURL,
            data.first.servicetype, eport, proto, intClient, intPort,
            description, nullptr/*enabled*/, duration);
    if (r != UPNPCOMMAND_SUCCESS) {
        Logger::Debug("GetSpecificPortMappingEntry() failed with code %d (%s)", r, strupnperror(r));
    } else {
        if (std::string(description) != formatted_desc) {
            Logger::Debug("Port %s is used by other app: %s", eport, description);
            return false;
        }
    }

    r = UPNP_AddPortMapping(urls.controlURL, data.first.servicetype, eport,
            iport, iaddr, formatted_desc.c_str(), proto, 0, leaseDuration);
    if (r != UPNPCOMMAND_SUCCESS) {
        Logger::Debug("AddPortMapping(%s, %s, %s) failed with code %d (%s)", eport, iport, iaddr, r, strupnperror(r));
        return false;
    }

    r = UPNP_GetSpecificPortMappingEntry(urls.controlURL,
            data.first.servicetype, eport, proto, intClient, intPort,
            nullptr, nullptr/*enabled*/, duration);
    if (r != UPNPCOMMAND_SUCCESS) {
        Logger::Debug("GetSpecificPortMappingEntry() failed with code %d (%s)", r, strupnperror(r));
        return false;
    }

    if (intClient[0]) {
        Logger::Debug("InternalIP:Port = %s:%s", intClient, intPort);
        Logger::Debug("%s external %d %s is redirected to internal %s:%s (duration=%s)",
                description, eport, proto, intClient, intPort, duration);

        return true;
    } else {
        Logger::Error("Cannot open external port %d", eport);
        return false;
    }
}

uint16_t Map(uint16_t* port, std::string* global_ip, bool udp)
{
    struct UPNPDev * devlist = 0;
    char lanaddr[64]; /* my ip address on the LAN */
    int i;
    const char * rootdescurl = 0;
    const char * multicastif = 0;
    const char * minissdpdpath = 0;
    int error = 0;
    int ipv6 = 0;

    /*
    WSADATA wsaData;
    int nResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (nResult != NO_ERROR) {
        Logger::Error("WSAStartup() failed.");
        return 0;
    }
    */

    if (rootdescurl
            || (devlist = upnpDiscover(2000, multicastif, minissdpdpath,
                    0/*sameport*/, ipv6, &error))) {
        struct UPNPDev * device;
        struct UPNPUrls urls;
        struct IGDdatas data;
        if (devlist) {
            for (device = devlist; device; device = device->pNext) {
                Logger::Debug(" desc: %s\n st: %s\n", device->descURL, device->st);
            }
        } else {
            Logger::Error("upnpDiscover() error code= %d", error);
        }
        i = 1;
        if ((rootdescurl
                && UPNP_GetIGDFromUrl(rootdescurl, &urls, &data, lanaddr,
                        sizeof(lanaddr)))
                || (i = UPNP_GetValidIGD(devlist, &urls, &data, lanaddr,
                        sizeof(lanaddr)))) {
            switch (i) {
                case 1:
                    Logger::Debug("Found valid IGD : %s", urls.controlURL);
                case 2:
                    Logger::Debug("Found a (not connected?) IGD : %s", urls.controlURL);
                    Logger::Debug("Trying to continue anyway");
                    break;
                case 3:
                    Logger::Debug("UPnP device found. Is it an IGD ? : %s", urls.controlURL);
                    Logger::Debug("Trying to continue anyway");
                    break;
                default:
                    Logger::Debug("Found device (igd ?) : %s", urls.controlURL);
                    Logger::Debug("Trying to continue anyway");
                    break;
            }

            Logger::Debug("Local LAN ip address : %s", lanaddr);

            /*
            if (!network::Utils::IsPrivateAddress(lanaddr)) {
                Logger::Debug("Local LAN has global ip address.");
                *port = 0;
                return true;
            }
            */

            char externalIPAddress[40] = {0};
            // int r;

            const char proto_udp[4] = "UDP";
            const char proto_tcp[4] = "TCP";
            const char* proto = udp ? proto_udp : proto_tcp;
            const char* desc = "MMO Client";
            const char* iaddr = lanaddr;
            const char* leaseDuration = "";

            UPNP_GetExternalIPAddress(urls.controlURL, data.first.servicetype,
                    externalIPAddress);
            if (externalIPAddress[0]) {
                Logger::Debug("ExternalIPAddress = %s", externalIPAddress);
                *global_ip = externalIPAddress;
            } else {
                Logger::Error("GetExternalIPAddress failed.");
            }

            uint32_t try_port = 39391;

            for (int i = 0; i < MAX_TRY_TIME; i++) {
                std::stringstream port_str;
                port_str << try_port;

                Logger::Debug("Trying open port %d ...", try_port);
                if ( TryMap(urls, data, port_str.str().c_str(), port_str.str().c_str(), iaddr, desc, proto, leaseDuration) ) {
                    *port = try_port;
                    return true;
                }
                try_port++;
            }

//            r = UPNP_AddPortMapping(urls.controlURL, data.first.servicetype,
//                    eport, iport, iaddr, desc, proto, 0, leaseDuration);
//            if (r != UPNPCOMMAND_SUCCESS) {
//                if (auto log = syslog.lock())
//                    log->Error((boost::format("AddPortMapping(%s, %s, %s) failed with code %d (%s)")
//                        % eport % iport % iaddr % r % strupnperror(r)).str());
//            }
//
//            r = UPNP_GetSpecificPortMappingEntry(urls.controlURL,
//                    data.first.servicetype, eport, proto, intClient, intPort,
//                    nullptr/*desc*/, nullptr/*enabled*/, duration);
//            if (r != UPNPCOMMAND_SUCCESS) {
//                if (auto log = syslog.lock())
//                    log->Error((boost::format("GetSpecificPortMappingEntry() failed with code %d (%s)")
//                        % r % strupnperror(r)).str());
//            }

            FreeUPNPUrls(&urls);

        } else {
            Logger::Error("No valid UPNP Internet Gateway Device found.");
        }
        freeUPNPDevlist(devlist);
        devlist = 0;
    } else {
        Logger::Error("No IGD UPnP Device found on the network !");
    }

    return false;
}

bool MapTCP(uint16_t* port, std::string* global_ip)
{
    return Map(port, global_ip, false);
}

bool MapUDP(uint16_t* port, std::string* global_ip)
{
    return Map(port, global_ip, true);
}

}
