//************************************************************************//
// Copyright (C) 2011-2012 Mikiya Fujii                                   // 
//                                                                        // 
// This file is part of MolDS.                                            // 
//                                                                        // 
// MolDS is free software: you can redistribute it and/or modify          // 
// it under the terms of the GNU General Public License as published by   // 
// the Free Software Foundation, either version 3 of the License, or      // 
// (at your option) any later version.                                    // 
//                                                                        // 
// MolDS is distributed in the hope that it will be useful,               // 
// but WITHOUT ANY WARRANTY; without even the implied warranty of         // 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          // 
// GNU General Public License for more details.                           // 
//                                                                        // 
// You should have received a copy of the GNU General Public License      // 
// along with MolDS.  If not, see <http://www.gnu.org/licenses/>.         // 
//************************************************************************//
#include<stdio.h>
#include<stdlib.h>
#include<iostream>
#include<sstream>
#include<math.h>
#include<string>
#include<vector>
#include<stdexcept>
#include<omp.h>
#include<boost/format.hpp>
#include"../base/PrintController.h"
#include"../base/MolDSException.h"
#include"../base/Uncopyable.h"
#include"../wrappers/Lapack.h"
#include"../base/Enums.h"
#include"../base/MathUtilities.h"
#include"../base/MallocerFreer.h"
#include"../base/EularAngle.h"
#include"../base/Parameters.h"
#include"../base/atoms/Atom.h"
#include"../base/atoms/Hatom.h"
#include"../base/atoms/Liatom.h"
#include"../base/atoms/Catom.h"
#include"../base/atoms/Natom.h"
#include"../base/atoms/Oatom.h"
#include"../base/atoms/Satom.h"
#include"../base/Molecule.h"
#include"../base/GTOExpansionSTO.h"
#include"../base/RealSphericalHarmonicsIndex.h"
#include"../base/loggers/MOLogger.h"
#include"../base/ElectronicStructure.h"
#include"Cndo2.h"
#include"ReducedOverlapParameters.h"
using namespace std;
using namespace MolDS_base;
using namespace MolDS_base_atoms;

namespace MolDS_cndo{

/***
 *  Refferences for Cndo2 are [PB_1970], [PSS_1965], and [PS_1965].
 */
Cndo2::Cndo2(){
   this->theory = CNDO2;
   this->SetMessages();
   this->SetEnableAtomTypes();
   this->gammaAB = NULL;
   this->overlap = NULL;
   this->cartesianMatrix = NULL;
   this->electronicTransitionDipoleMoments = NULL;
   this->coreDipoleMoment = NULL;
   this->twoElecTwoCore = NULL;
   this->orbitalElectronPopulation = NULL;
   this->atomicElectronPopulation = NULL;
   this->fockMatrix = NULL;
   this->energiesMO = NULL;
   this->molecule = NULL;
   this->matrixForce = NULL;
   this->bondingAdjustParameterK[0] = 1.000; //see (3.79) in J. A. Pople book
   this->bondingAdjustParameterK[1] = 0.750; //see (3.79) in J. A. Pople book
   this->elecSCFEnergy = 0.0;
   this->coreRepulsionEnergy = 0.0;
   this->vdWCorrectionEnergy = 0.0;
   this->matrixCIS = NULL;
   this->excitedEnergies = NULL;
   this->freeExcitonEnergiesCIS = NULL;
   this->matrixCISdimension = 0;
   //this->OutputLog("Cndo created\n");
}

Cndo2::~Cndo2(){
   MallocerFreer::GetInstance()->Free<double>(&this->gammaAB, 
                                              this->molecule->GetNumberAtoms(),
                                              this->molecule->GetNumberAtoms());
   MallocerFreer::GetInstance()->Free<double>(&this->overlap, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(&this->cartesianMatrix, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs(),
                                              CartesianType_end);
   int electronicTransitionDipoleMomentsDim = 1;
   if(Parameters::GetInstance()->RequiresCIS()){
      electronicTransitionDipoleMomentsDim += Parameters::GetInstance()->GetNumberExcitedStatesCIS();
   }
   MallocerFreer::GetInstance()->Free<double>(&this->electronicTransitionDipoleMoments, 
                                              electronicTransitionDipoleMomentsDim,
                                              electronicTransitionDipoleMomentsDim,
                                              CartesianType_end);
   MallocerFreer::GetInstance()->Free<double>(&this->coreDipoleMoment, 
                                              CartesianType_end);
   MallocerFreer::GetInstance()->Free<double>(&this->orbitalElectronPopulation, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(&this->atomicElectronPopulation, 
                                              this->molecule->GetNumberAtoms());
   MallocerFreer::GetInstance()->Free<double>(&this->fockMatrix, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(&this->energiesMO, 
                                              this->molecule->GetTotalNumberAOs());
   //this->OutputLog("cndo deleted\n");
}

void Cndo2::SetMessages(){
   this->errorMessageSCFNotConverged 
      = "Error in cndo::Cndo2::DoSCF: SCF did not met convergence criterion. maxIterationsSCF=";
   this->errorMessageMoleculeNotSet 
      = "Error in cndo::Cndo2::DoSCF: A molecule is not set.\n";
   this->errorMessageOddTotalValenceElectrions 
      = "Error in cndo::Cndo2::SetMolecule: Total number of valence electrons is odd. totalNumberValenceElectrons=";
   this->errorMessageNotEnebleAtomType  
      = "Error in cndo::Cndo2::ChecEnableAtomType: Non available atom is contained.\n";
   this->errorMessageAtomA = "Atom A is:\n";
   this->errorMessageAtomB = "Atom B is:\n";
   this->errorMessageAtomType = "\tatom type = ";
   this->errorMessageOrbitalType = "\torbital type = ";
   this->errorMessageCartesianType = "\tcartesian type = ";
   this->errorMessageMolecularIntegralElement
      = "Error in cndo::Cndo2::GetMolecularIntegralElement: Non available orbital is contained.\n";
   this->errorMessageGetDiatomCoreRepulsion2ndDerivativeNotImplemented
      = "Error in cndo::Cndo2::GetDiatomCoreRepulsion2ndDerivative: Second derivative is not implemented for CNDO2.\n";
   this->errorMessageGetGaussianCartesianMatrixBadOrbital 
      = "Error in cndo::Cndo2::GetGaussianCartesianMatrix: Untreatable orbital is contained in atom A or B.\n";
   this->errorMessageGetGaussianOverlapBadOrbital 
      = "Error in cndo::Cndo2::GetGaussianOverlap: Untreatable orbital is contained in atom A or B.\n";
   this->errorMessageGetGaussianOverlap1stDerivativeOrbitalD 
      = "Error in cndo::Cndo2::GetGaussianOverlap1stDerivative: d-orbital is not treatable. The d-orbital is contained in atom A or B.\n";
   this->errorMessageCISNotImplemented 
      = "Error in cndo::Cndo2: CIS is not implemented for CNDO2.\n";
   this->errorMessageCalcForceNotImplemented
      = "Error in cndo::Cndo2::CalcForce: Force is not available in CNDO2.\n";
   this->errorMessageGetElectronicEnergyNumberCISStates 
      = "\tNumber of calculated CIS states (excluding ground state) = ";
   this->errorMessageGetElectronicEnergySetElecState
      = "\tSet Electronic state = ";
   this->errorMessageGetElectronicEnergyEnergyNotCalculated
      = "Error in cndo::Cndo2::GetElectronicEnergy: Set electronic state is not calculated by CIS.\n";
   this->errorMessageGetElectronicEnergyNULLCISEnergy 
      = "Error in cndo::Cndo2::GetElectronicEnergy: excitedEnergies is NULL\n";
   this->errorMessageCalDiaOverlapDiaFrameNullMatrix 
      = "Error in cndo::Cndo2::CalcDiatomicOverlapInDiatomicFrame: diatomicOverlap is NULL.\n";
   this->errorMessageCalcRotatingMatrixNullRotMatrix 
      = "Error in cndo::Cndo2::CalcRotatingMatrix: rotatingMatrix is NULL.\n";
   this->errorMessageRotDiaOverlapToSpaceFrameNullDiaMatrix 
      = "Error in cndo::Cndo2::RotateDiatmicOverlapToSpaceFrame diatomicOverlap is NULL.\n";
   this->errorMessageRotDiaOverlapToSpaceFrameNullRotMatrix 
      = "Error in cndo::Cndo2::RotateDiatmicOverlapToSpaceFrame: rotatingMatrix is NULL.\n";
   this->errorMessageSetOverlapElementNullDiaMatrix 
      = "Error in cndo::Cndo2::SetOverlapElement: diatomicOverlap is NULL.\n";
   this->errorMessageGetElectronicTransitionDipoleMomentBadState
      = "Error in cndo::Cndo2::GetElectronicTransitionDipoleMoment: Bad eigen state is set. In SCF module, the transition dipole moment of only between ground states can be calculated. Note taht state=0 means the ground state and other state = i means the i-th excited state in below.\n";
   this->errorMessageFromState = "\tfrom state = ";
   this->errorMessageToState = "\tto state = ";
   this->messageSCFMetConvergence = "\n\n\n\t\tCNDO/2-SCF met convergence criterion(^^b\n\n\n";
   this->messageStartSCF = "**********  START: CNDO/2-SCF  **********\n";
   this->messageDoneSCF = "**********  DONE: CNDO/2-SCF  **********\n\n\n";
   this->messageOmpElapsedTimeSCF = "\tElapsed time(omp) for the SCF = ";
   this->messageIterSCF = "SCF iter=";
   this->messageDensityRMS = ": RMS density=";
   this->messageEnergyMO = "\tEnergy of MO:";
   this->messageEnergyMOTitle = "\t\t\t| i-th | occ/unocc |  e[a.u.]  |  e[eV]  | \n";
   this->messageOcc = "occ";
   this->messageUnOcc = "unocc";
   this->messageMullikenAtoms = "\tMulliken charge:";
   this->messageMullikenAtomsTitle = "\t\t\t\t| i-th | atom type | core charge[a.u.] | Mulliken charge[a.u.]| \n";
   this->messageElecEnergy = "\tElectronic energy(SCF):";
   this->messageNoteElecEnergy = "\tNote that this electronic energy includs core-repulsions.\n\n";
   this->messageNoteElecEnergyVdW = "\tNote that this electronic energy includs core-repulsions and vdW correction.\n\n";
   this->messageElecEnergyTitle = "\t\t\t\t|   [a.u.]   |   [eV]   |\n";
   this->messageUnitSec = "[s].";
   this->messageCoreRepulsionTitle = "\t\t\t\t|   [a.u.]   |   [eV]   |\n";
   this->messageCoreRepulsion = "\tCore repulsion energy:";
   this->messageVdWCorrectionTitle = "\t\t\t\t\t\t|   [a.u.]   |   [eV]   |\n";
   this->messageVdWCorrection = "\tEmpirical van der Waals correction:";
   this->messageElectronicDipoleMomentTitle = "\t\t\t\t\t|  x[a.u.]  |  y[a.u.]  |  z[a.u.]  |  magnitude[a.u.]  |\t\t|  x[debye]  |  y[debye]  |  z[debye]  |  magnitude[debye]  |\n";
   this->messageElectronicDipoleMoment = "\tElectronic Dipole moment(SCF):";
   this->messageCoreDipoleMomentTitle = "\t\t\t\t\t|  x[a.u.]  |  y[a.u.]  |  z[a.u.]  |  magnitude[a.u.]  |\t\t|  x[debye]  |  y[debye]  |  z[debye]  |  magnitude[debye]  |\n";
   this->messageCoreDipoleMoment = "\tCore Dipole moment:";
   this->messageTotalDipoleMomentTitle = "\t\t\t\t\t|   x[a.u.]   |   y[a.u.]   |   z[a.u.]   |  magnitude[a.u.]  |\t\t|  x[debye]  |  y[debye]  |  z[debye]  |  magnitude[debye]  |\n";
   this->messageTotalDipoleMoment = "\tTotal Dipole moment(SCF):";
}

void Cndo2::SetEnableAtomTypes(){
   this->enableAtomTypes.clear();
   this->enableAtomTypes.push_back(H);
   this->enableAtomTypes.push_back(Li);
   //this->enableAtomTypes.push_back(Be);
   //this->enableAtomTypes.push_back(B);
   this->enableAtomTypes.push_back(C);
   this->enableAtomTypes.push_back(N);
   this->enableAtomTypes.push_back(O);
   //this->enableAtomTypes.push_back(F);
   //this->enableAtomTypes.push_back(Na);
   //this->enableAtomTypes.push_back(Mg);
   //this->enableAtomTypes.push_back(Al);
   //this->enableAtomTypes.push_back(Si);
   //this->enableAtomTypes.push_back(P);
   this->enableAtomTypes.push_back(S);
   //this->enableAtomTypes.push_back(Cl);
}

TheoryType Cndo2::GetTheoryType() const{
   return this->theory;
}

void Cndo2::SetMolecule(Molecule* molecule){
   // check of number of valence electrons
   this->CheckNumberValenceElectrons(*molecule);

   // check enable atom type
   this->CheckEnableAtomType(*molecule);

   // set molecule and malloc
   this->molecule = molecule;
   if(this->theory == CNDO2 || this->theory == INDO){
      MallocerFreer::GetInstance()->Malloc<double>(&this->gammaAB,
                                                   this->molecule->GetNumberAtoms(), 
                                                   this->molecule->GetNumberAtoms());
   }
   MallocerFreer::GetInstance()->Malloc<double>(&this->overlap, 
                                                this->molecule->GetTotalNumberAOs(), 
                                                this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Malloc<double>(&this->cartesianMatrix, 
                                                this->molecule->GetTotalNumberAOs(), 
                                                this->molecule->GetTotalNumberAOs(),
                                                CartesianType_end);
   int electronicTransitionDipoleMomentsDim = 1;
   if(Parameters::GetInstance()->RequiresCIS()){
      electronicTransitionDipoleMomentsDim += Parameters::GetInstance()->GetNumberExcitedStatesCIS();
   }
   MallocerFreer::GetInstance()->Malloc<double>(&this->electronicTransitionDipoleMoments, 
                                                electronicTransitionDipoleMomentsDim,
                                                electronicTransitionDipoleMomentsDim,
                                                CartesianType_end);
   MallocerFreer::GetInstance()->Malloc<double>(&this->coreDipoleMoment, 
                                                CartesianType_end);
   MallocerFreer::GetInstance()->Malloc<double>(&this->orbitalElectronPopulation,
                                                this->molecule->GetTotalNumberAOs(), 
                                                this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Malloc<double>(&this->atomicElectronPopulation,
                                                this->molecule->GetNumberAtoms());
   MallocerFreer::GetInstance()->Malloc<double>(&this->fockMatrix,
                                                this->molecule->GetTotalNumberAOs(), 
                                                this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Malloc<double>(&this->energiesMO,
                                                this->molecule->GetTotalNumberAOs());
}

void Cndo2::CheckNumberValenceElectrons(const Molecule& molecule) const{
   if(molecule.GetTotalNumberValenceElectrons() % 2 == 1){
      stringstream ss;
      ss << this->errorMessageOddTotalValenceElectrions << molecule.GetTotalNumberValenceElectrons() << "\n";
      throw MolDSException(ss.str());
   }
}

void Cndo2::CheckEnableAtomType(const Molecule& molecule) const{
   for(int i=0; i<molecule.GetNumberAtoms(); i++){
      AtomType atomType = molecule.GetAtom(i)->GetAtomType();
      bool enable = false;
      for(int j=0; j<this->enableAtomTypes.size(); j++){
         if(atomType == this->enableAtomTypes[j]){
            enable = true;
            break;
         }
      }
      if(!enable){
         stringstream ss;
         ss << this->errorMessageNotEnebleAtomType;
         ss << this->errorMessageAtomType << AtomTypeStr(atomType) << endl;
         throw MolDSException(ss.str());
      }
   }
}

void Cndo2::CalcCoreRepulsionEnergy(){
   double energy = 0.0;
   for(int i=0; i<this->molecule->GetNumberAtoms(); i++){
      for(int j=i+1; j<this->molecule->GetNumberAtoms(); j++){
         energy += this->GetDiatomCoreRepulsionEnergy(i, j);
      }
   }
   this->coreRepulsionEnergy = energy;
}

double Cndo2::GetDiatomCoreRepulsionEnergy(int indexAtomA, int indexAtomB) const{
   const Atom& atomA = *this->molecule->GetAtom(indexAtomA);
   const Atom& atomB = *this->molecule->GetAtom(indexAtomB);
   double distance = this->molecule->GetDistanceAtoms(indexAtomA, indexAtomB);
   return atomA.GetCoreCharge()*atomB.GetCoreCharge()/distance; 
}

// First derivative of diatomic core repulsion energy.
// This derivative is related to the coordinate of atomA.
double Cndo2::GetDiatomCoreRepulsion1stDerivative(int indexAtomA, int indexAtomB, 
                                                  CartesianType axisA) const{
   double value=0.0;
   const Atom& atomA = *this->molecule->GetAtom(indexAtomA);
   const Atom& atomB = *this->molecule->GetAtom(indexAtomB);
   double distance = this->molecule->GetDistanceAtoms(indexAtomA, indexAtomB);
   value = atomA.GetCoreCharge()*atomB.GetCoreCharge();
   value *= (atomA.GetXyz()[axisA] - atomB.GetXyz()[axisA])/distance;
   value *= -1.0/pow(distance,2.0);
   return value;
}

// Second derivative of diatomic core repulsion energy.
// Both derivatives are related to the coordinate of atomA.
double Cndo2::GetDiatomCoreRepulsion2ndDerivative(int indexAtomA,
                                                  int indexAtomB, 
                                                  CartesianType axisA1,
                                                  CartesianType axisA2) const{
   stringstream ss;
   ss << this->errorMessageGetDiatomCoreRepulsion2ndDerivativeNotImplemented;
   throw MolDSException(ss.str());
}

// See (2) in [G_2004] ((11) in [G_2006])
void Cndo2::CalcVdWCorrectionEnergy(){
   double value = 0.0;
   for(int i=0; i<this->molecule->GetNumberAtoms(); i++){
      for(int j=i+1; j<this->molecule->GetNumberAtoms(); j++){
         value += this->GetDiatomVdWCorrectionEnergy(i, j);
      }
   }
   this->vdWCorrectionEnergy = value;
}

// See damping function in (2) in [G_2004] ((11) in [G_2006])
double Cndo2::GetVdwDampingValue(double vdWDistance, double distance) const{
   double dampingFactor = Parameters::GetInstance()->GetVdWDampingFactorSCF();
   return 1.0/(1.0+exp(-1.0*dampingFactor*(distance/vdWDistance - 1.0)));
}

// See damping function in (2) in [G_2004] ((11) in [G_2006])
double Cndo2::GetVdwDampingValue1stDerivative(double vdWDistance, double distance) const{
   double dampingFactor = Parameters::GetInstance()->GetVdWDampingFactorSCF();
   return (dampingFactor/vdWDistance)
         *exp(-1.0*dampingFactor*(distance/vdWDistance - 1.0))
         *pow(1.0+exp(-1.0*dampingFactor*(distance/vdWDistance - 1.0)),-2.0);
}

// See damping function in (2) in [G_2004] ((11) in [G_2006])
double Cndo2::GetVdwDampingValue2ndDerivative(double vdWDistance, double distance) const{
   double dampingFactor = Parameters::GetInstance()->GetVdWDampingFactorSCF();
   double exponent = -1.0*dampingFactor*(distance/vdWDistance - 1.0);
   double pre = dampingFactor/vdWDistance;
   double dominator = 1.0+exp(exponent);
   return 2.0*pow(dominator,-3.0)*pre*pre*exp(2.0*exponent) 
         -    pow(dominator,-2.0)*pre*pre*exp(    exponent);
}

// See (2) in [G_2004] ((11) in [G_2006])
double Cndo2::GetDiatomVdWCorrectionEnergy(int indexAtomA, int indexAtomB) const{
   const Atom& atomA = *this->molecule->GetAtom(indexAtomA);
   const Atom& atomB = *this->molecule->GetAtom(indexAtomB);
   double distance = this->molecule->GetDistanceAtoms(indexAtomA, indexAtomB);
   double vdWDistance = atomA.GetVdWRadii() + atomB.GetVdWRadii();
   double vdWCoefficients = 2.0*atomA.GetVdWCoefficient()*atomB.GetVdWCoefficient()
                           /(atomA.GetVdWCoefficient()+atomB.GetVdWCoefficient());
   double damping = this->GetVdwDampingValue(vdWDistance, distance);
   double scalingFactor = Parameters::GetInstance()->GetVdWScalingFactorSCF();
   return -1.0*scalingFactor*vdWCoefficients*pow(distance,-6.0)*damping;
}

// First derivative of the vdW correction related to the coordinate of atom A.
// See (2) in [G_2004] ((11) in [G_2006]).
double Cndo2::GetDiatomVdWCorrection1stDerivative(int indexAtomA, int indexAtomB, 
                                                  CartesianType axisA) const{
   const Atom& atomA = *this->molecule->GetAtom(indexAtomA);
   const Atom& atomB = *this->molecule->GetAtom(indexAtomB);
   double distance = this->molecule->GetDistanceAtoms(indexAtomA, indexAtomB);
   double vdWDistance = atomA.GetVdWRadii() + atomB.GetVdWRadii();
   double vdWCoefficients = 2.0*atomA.GetVdWCoefficient()*atomB.GetVdWCoefficient()
                           /(atomA.GetVdWCoefficient()+atomB.GetVdWCoefficient());
   double dampingFactor = Parameters::GetInstance()->GetVdWDampingFactorSCF();
   double damping = this->GetVdwDampingValue(vdWDistance, distance);
   double damping1stDerivative = this->GetVdwDampingValue1stDerivative(vdWDistance, distance);
   double value=0.0;
   value += 6.0*pow(distance,-7.0)*damping - pow(distance,-6.0)*damping1stDerivative;
   value *= vdWCoefficients;
   value *= Parameters::GetInstance()->GetVdWScalingFactorSCF();
   value *= (atomA.GetXyz()[axisA] - atomB.GetXyz()[axisA])/distance;
   return value;
}

// Second derivative of the vdW correction.
// Both derivative sare related to the coordinate of atom A.
// See (2) in [G_2004] ((11) in [G_2006]).
double Cndo2::GetDiatomVdWCorrection2ndDerivative(int indexAtomA, 
                                                     int indexAtomB, 
                                                     CartesianType axisA1,
                                                     CartesianType axisA2) const{
   const Atom& atomA = *this->molecule->GetAtom(indexAtomA);
   const Atom& atomB = *this->molecule->GetAtom(indexAtomB);
   double distance = this->molecule->GetDistanceAtoms(indexAtomA, indexAtomB);
   double dCartesian1 = atomA.GetXyz()[axisA1] - atomB.GetXyz()[axisA1];
   double dCartesian2 = atomA.GetXyz()[axisA2] - atomB.GetXyz()[axisA2];
   double vdWDistance = atomA.GetVdWRadii() + atomB.GetVdWRadii();
   double vdWScalingFacotor = Parameters::GetInstance()->GetVdWScalingFactorSCF();
   double vdWCoefficients = 2.0*atomA.GetVdWCoefficient()*atomB.GetVdWCoefficient()
                           /(atomA.GetVdWCoefficient()+atomB.GetVdWCoefficient());
   double dampingFactor = Parameters::GetInstance()->GetVdWDampingFactorSCF();
   double damping = this->GetVdwDampingValue(vdWDistance, distance);
   double damping1stDerivative = this->GetVdwDampingValue1stDerivative(vdWDistance, distance);
   double damping2ndDerivative = this->GetVdwDampingValue2ndDerivative(vdWDistance, distance);

   double temp1 = -6.0*pow(distance,-7.0)*damping 
                  +    pow(distance,-6.0)*damping1stDerivative;
   double temp2 = 42.0*pow(distance,-8.0)*damping 
                 -12.0*pow(distance,-7.0)*damping1stDerivative
                 +     pow(distance,-6.0)*damping2ndDerivative;

   double pre1=0.0;
   double pre2=0.0;
   if(axisA1 != axisA2){
      pre1 = -dCartesian1*dCartesian2/pow(distance,3.0);
      pre2 =  dCartesian1*dCartesian2/pow(distance,2.0);
   }
   else{
      pre1 = 1.0/distance - dCartesian1*dCartesian1/pow(distance,3.0);
      pre2 = pow(dCartesian1/distance,2.0);
   }

   double value= pre1*temp1 + pre2*temp2;
   value *= -1.0*vdWScalingFacotor*vdWCoefficients;
   return value;
}

/*******
 *
 * Call Cndo2::SetMolecule(Molecule* molecule) at least once, 
 * before this function is called.
 *
 *****/
void Cndo2::DoSCF(bool requiresGuess){
   this->OutputLog(this->messageStartSCF);
   double ompStartTime = omp_get_wtime();

   if(this->molecule == NULL){
      stringstream ss;
      ss << this->errorMessageMoleculeNotSet;
      throw MolDSException(ss.str());
   }

   // temporary matrices for scf
   double**  oldOrbitalElectronPopulation = NULL;
   double*** diisStoredDensityMatrix = NULL;
   double*** diisStoredErrorVect = NULL;
   double**  diisErrorProducts = NULL;
   double*   diisErrorCoefficients = NULL;

   try{
      this->MallocSCFTemporaryMatrices(&oldOrbitalElectronPopulation,
                                       &diisStoredDensityMatrix,
                                       &diisStoredErrorVect,
                                       &diisErrorProducts,
                                       &diisErrorCoefficients);
      // calculate electron integral
      this->CalcGammaAB(this->gammaAB, *this->molecule);
      this->CalcOverlap(this->overlap, *this->molecule);
      this->CalcCartesianMatrixByGTOExpansion(this->cartesianMatrix, *this->molecule, STO6G);
      this->CalcTwoElecTwoCore(this->twoElecTwoCore, *this->molecule);

      // SCF
      double rmsDensity;
      int maxIterationsSCF = Parameters::GetInstance()->GetMaxIterationsSCF();
      bool isGuess=true;
      for(int iterationStep=0; iterationStep<maxIterationsSCF; iterationStep++){
         this->CalcAtomicElectronPopulation(this->atomicElectronPopulation, 
                                            this->orbitalElectronPopulation, 
                                            *this->molecule);
         this->UpdateOldOrbitalElectronPopulation(oldOrbitalElectronPopulation, 
                                                  this->orbitalElectronPopulation, 
                                                  this->molecule->GetTotalNumberAOs());
         isGuess = (iterationStep==0 && requiresGuess);
         this->CalcFockMatrix(this->fockMatrix, 
                              *this->molecule, 
                              this->overlap, 
                              this->gammaAB,
                              this->orbitalElectronPopulation, 
                              this->atomicElectronPopulation,
                              this->twoElecTwoCore,
                              isGuess);

         // diagonalization of the Fock matrix
         bool calcEigenVectors = true;
         MolDS_wrappers::Lapack::GetInstance()->Dsyevd(this->fockMatrix, 
                                                       this->energiesMO, 
                                                       this->molecule->GetTotalNumberAOs(), 
                                                       calcEigenVectors);

         this->CalcOrbitalElectronPopulation(this->orbitalElectronPopulation, 
                                             *this->molecule, 
                                             this->fockMatrix);

         // check convergence
         bool hasConverged = this->SatisfyConvergenceCriterion(oldOrbitalElectronPopulation, 
                                                               this->orbitalElectronPopulation,
                                                               this->molecule->GetTotalNumberAOs(), 
                                                               &rmsDensity, 
                                                               iterationStep);
         if(hasConverged){
            this->OutputLog(this->messageSCFMetConvergence);
            this->CalcSCFProperties();
            this->OutputSCFResults();
            break;
         }
         else{
            if(!isGuess){ 
               this->DoDamp(rmsDensity, 
                            this->orbitalElectronPopulation, 
                            oldOrbitalElectronPopulation, 
                            *this->molecule);
               this->DoDIIS(this->orbitalElectronPopulation,
                            oldOrbitalElectronPopulation,
                            diisStoredDensityMatrix,
                            diisStoredErrorVect,
                            diisErrorProducts,
                            diisErrorCoefficients,
                            Parameters::GetInstance()->GetDiisNumErrorVectSCF(),
                            *this->molecule,
                            iterationStep);
            }
         }

         // SCF fails
         if(iterationStep==maxIterationsSCF-1){
            stringstream ss;
            ss << this->errorMessageSCFNotConverged << maxIterationsSCF << "\n";
            throw MolDSException(ss.str());
         }
      }
   }
   catch(MolDSException ex){
      this->FreeSCFTemporaryMatrices(&oldOrbitalElectronPopulation,
                                     &diisStoredDensityMatrix,
                                     &diisStoredErrorVect,
                                     &diisErrorProducts,
                                     &diisErrorCoefficients);

      throw ex;
   }
   this->FreeSCFTemporaryMatrices(&oldOrbitalElectronPopulation,
                                  &diisStoredDensityMatrix,
                                  &diisStoredErrorVect,
                                  &diisErrorProducts,
                                  &diisErrorCoefficients);

   double ompEndTime = omp_get_wtime();
   this->OutputLog(boost::format("%s%lf%s\n%s") % this->messageOmpElapsedTimeSCF.c_str()
                                                % (ompEndTime - ompStartTime)
                                                % this->messageUnitSec.c_str()
                                                % this->messageDoneSCF.c_str());

}

void Cndo2::CalcSCFProperties(){
   this->CalcAtomicElectronPopulation(this->atomicElectronPopulation, 
                                      this->orbitalElectronPopulation, 
                                      *this->molecule);
   this->CalcCoreRepulsionEnergy();
   if(Parameters::GetInstance()->RequiresVdWSCF()){
      this->CalcVdWCorrectionEnergy();
   }
   this->CalcElecSCFEnergy(&this->elecSCFEnergy, 
                          *this->molecule, 
                          this->energiesMO, 
                          this->fockMatrix, 
                          this->gammaAB,
                          this->coreRepulsionEnergy,
                          this->vdWCorrectionEnergy);
   this->CalcCoreDipoleMoment(this->coreDipoleMoment, *this->molecule);
   this->CalcElectronicDipoleMomentGroundState(this->electronicTransitionDipoleMoments, 
                                               this->cartesianMatrix,
                                               *this->molecule, 
                                               this->orbitalElectronPopulation,
                                               this->overlap);
}

double Cndo2::GetBondingAdjustParameterK(ShellType shellA, ShellType shellB) const{
   double value=1.0;
   if(shellA >= m || shellB >= m){
      return this->bondingAdjustParameterK[1];
   }
   return value;
}

void Cndo2::DoCIS(){
   stringstream ss;
   ss << this->errorMessageCISNotImplemented;
   throw MolDSException(ss.str());
}

void Cndo2::OutputCISResults() const{
   stringstream ss;
   ss << this->errorMessageCISNotImplemented;
   throw MolDSException(ss.str());
}

void Cndo2::CalcCISProperties(){
   stringstream ss;
   ss << this->errorMessageCISNotImplemented;
   throw MolDSException(ss.str());
}

// elecState=0 means ground state
double Cndo2::GetElectronicEnergy(int elecState) const{
   int groundState = 0;
   if(elecState==groundState){
      return this->elecSCFEnergy;
   }
   else{
      if(this->excitedEnergies == NULL){
         stringstream ss;
         ss << this->errorMessageGetElectronicEnergyNULLCISEnergy;
         throw MolDSException(ss.str());
      }
      int numberExcitedStates = Parameters::GetInstance()->GetNumberExcitedStatesCIS();
      if(numberExcitedStates < elecState){
         stringstream ss;
         ss << this->errorMessageGetElectronicEnergyEnergyNotCalculated;
         ss << this->errorMessageGetElectronicEnergySetElecState << elecState << endl;
         ss << errorMessageGetElectronicEnergyNumberCISStates << numberExcitedStates << endl;
         throw MolDSException(ss.str());
      }
      return this->elecSCFEnergy + this->excitedEnergies[elecState-1];
   }
}

double Cndo2::GetCoreRepulsionEnergy() const{
   return this->coreRepulsionEnergy;
}

double Cndo2::GetVdWCorrectionEnergy() const{
   return this->vdWCorrectionEnergy;
}

double*** Cndo2::GetForce(const vector<int>& elecStates){
   this->CalcForce(elecStates);
   return this->matrixForce;
}

double** Cndo2::GetForce(int elecState){
   vector<int> elecStates;
   elecStates.push_back(elecState);
   this->CalcForce(elecStates);
   return this->matrixForce[0];
}

void Cndo2::CalcTwoElecTwoCore(double****** twoElecTwoCore, 
                               const Molecule& molecule) const{
   // do nothing for CNDO, INDO, and ZINDO/S.
   // two electron two core integrals are not needed for CNDO, INDO, and ZINDO/S.
}

void Cndo2::CalcForce(const vector<int>& elecStates){
   stringstream ss;
   ss << this->errorMessageCalcForceNotImplemented;
   throw MolDSException(ss.str());
}

void Cndo2::FreeSCFTemporaryMatrices(double*** oldOrbitalElectronPopulation,
                                     double**** diisStoredDensityMatrix,
                                     double**** diisStoredErrorVect,
                                     double*** diisErrorProducts,
                                     double** diisErrorCoefficients) const{

   int diisNumErrorVect = Parameters::GetInstance()->GetDiisNumErrorVectSCF();
   MallocerFreer::GetInstance()->Free<double>(oldOrbitalElectronPopulation, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(diisStoredDensityMatrix, 
                                              diisNumErrorVect, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(diisStoredErrorVect, 
                                              diisNumErrorVect, 
                                              this->molecule->GetTotalNumberAOs(),
                                              this->molecule->GetTotalNumberAOs());
   MallocerFreer::GetInstance()->Free<double>(diisErrorProducts, 
                                              diisNumErrorVect+1,
                                              diisNumErrorVect+1);
   MallocerFreer::GetInstance()->Free<double>(diisErrorCoefficients,
                                              diisNumErrorVect+1);
}

void Cndo2::MallocSCFTemporaryMatrices(double*** oldOrbitalElectronPopulation,
                                       double**** diisStoredDensityMatrix,
                                       double**** diisStoredErrorVect,
                                       double*** diisErrorProducts,
                                       double** diisErrorCoefficients){

   int diisNumErrorVect = Parameters::GetInstance()->GetDiisNumErrorVectSCF();
   MallocerFreer::GetInstance()->Malloc<double>(oldOrbitalElectronPopulation, 
                                                this->molecule->GetTotalNumberAOs(), 
                                                this->molecule->GetTotalNumberAOs());
   if(0<diisNumErrorVect){
      MallocerFreer::GetInstance()->Malloc<double>(diisStoredDensityMatrix,
                                                   diisNumErrorVect, 
                                                   this->molecule->GetTotalNumberAOs(), 
                                                   this->molecule->GetTotalNumberAOs());
      MallocerFreer::GetInstance()->Malloc<double>(diisStoredErrorVect,
                                                   diisNumErrorVect, 
                                                   this->molecule->GetTotalNumberAOs(), 
                                                   this->molecule->GetTotalNumberAOs());
      MallocerFreer::GetInstance()->Malloc<double>(diisErrorProducts, diisNumErrorVect+1, diisNumErrorVect+1);
      MallocerFreer::GetInstance()->Malloc<double>(diisErrorCoefficients, diisNumErrorVect+1);
   }
}

/***
 *
 *  see ref. [P_1980] for diis methods.
 *
 */
void Cndo2::DoDIIS(double** orbitalElectronPopulation,
                   double const* const* oldOrbitalElectronPopulation,
                   double*** diisStoredDensityMatrix,
                   double*** diisStoredErrorVect,
                   double** diisErrorProducts,
                   double* diisErrorCoefficients,
                   int diisNumErrorVect,
                   const Molecule& molecule,
                   int step) const{
   int totalNumberAOs = molecule.GetTotalNumberAOs();
   double diisStartError = Parameters::GetInstance()->GetDiisStartErrorSCF();
   double diisEndError = Parameters::GetInstance()->GetDiisEndErrorSCF();

   if( 0 < diisNumErrorVect){
      for(int m=0; m<diisNumErrorVect-1; m++){
         stringstream ompErrors;
#pragma omp parallel for schedule(auto)
         for(int j=0; j<totalNumberAOs; j++){
            try{
               for(int k=0; k<totalNumberAOs; k++){
                  diisStoredDensityMatrix[m][j][k] = diisStoredDensityMatrix[m+1][j][k];
                  diisStoredErrorVect[m][j][k] = diisStoredErrorVect[m+1][j][k];
               }
            }
            catch(MolDSException ex){
#pragma omp critical
               ompErrors << ex.what() << endl ;
            }
         }
         // Exception throwing for omp-region
         if(!ompErrors.str().empty()){
            throw MolDSException(ompErrors.str());
         }
      }
      {
         stringstream ompErrors;
#pragma omp parallel for schedule(auto)
         for(int j=0; j<totalNumberAOs; j++){
            try{
               for(int k=0; k<totalNumberAOs; k++){
                  diisStoredDensityMatrix[diisNumErrorVect-1][j][k] = orbitalElectronPopulation[j][k];
                  diisStoredErrorVect[diisNumErrorVect-1][j][k] = orbitalElectronPopulation[j][k] 
                                                                 -oldOrbitalElectronPopulation[j][k];
                     
               }
            }
            catch(MolDSException ex){
#pragma omp critical
               ompErrors << ex.what() << endl ;
            }
         }
         // Exception throwing for omp-region
         if(!ompErrors.str().empty()){
            throw MolDSException(ompErrors.str());
         }
      }
      for(int mi=0; mi<diisNumErrorVect-1; mi++){
         for(int mj=0; mj<diisNumErrorVect-1; mj++){
            diisErrorProducts[mi][mj] = diisErrorProducts[mi+1][mj+1];
         }
      }
               
      for(int mi=0; mi<diisNumErrorVect; mi++){
         double tempErrorProduct = 0.0;
         stringstream ompErrors;
#pragma omp parallel for schedule(auto) reduction(+:tempErrorProduct)
         for(int j=0; j<totalNumberAOs; j++){
            try{
               for(int k=0; k<totalNumberAOs; k++){
                  tempErrorProduct += diisStoredErrorVect[mi][j][k]
                                     *diisStoredErrorVect[diisNumErrorVect-1][j][k];
               }
            }
            catch(MolDSException ex){
#pragma omp critical
               ompErrors << ex.what() << endl ;
            }
         }
         // Exception throwing for omp-region
         if(!ompErrors.str().empty()){
            throw MolDSException(ompErrors.str());
         }
         diisErrorProducts[mi][diisNumErrorVect-1] = tempErrorProduct;
         diisErrorProducts[diisNumErrorVect-1][mi] = tempErrorProduct;
         diisErrorProducts[mi][diisNumErrorVect] = -1.0;
         diisErrorProducts[diisNumErrorVect][mi] = -1.0;
         diisErrorCoefficients[mi] = 0.0;
      }
      diisErrorProducts[diisNumErrorVect][diisNumErrorVect] = 0.0;
      diisErrorCoefficients[diisNumErrorVect] = -1.0;

      double eMax = 0;
      for(int j=0; j<totalNumberAOs; j++){
         for(int k=0; k<totalNumberAOs; k++){
            eMax = max(eMax, fabs(diisStoredErrorVect[diisNumErrorVect-1][j][k]));
         }
      }

      if(diisNumErrorVect <= step && diisEndError<eMax && eMax<diisStartError){
         MolDS_wrappers::Lapack::GetInstance()->Dsysv(diisErrorProducts, 
                                                      diisErrorCoefficients, 
                                                      diisNumErrorVect+1);
         for(int j=0; j<totalNumberAOs; j++){
            for(int k=0; k<totalNumberAOs; k++){
               orbitalElectronPopulation[j][k] = 0.0;
               for(int m=0; m<diisNumErrorVect; m++){
                  orbitalElectronPopulation[j][k] += diisErrorCoefficients[m]*diisStoredDensityMatrix[m][j][k];
               }
            }
         }
      }
   }
}

void Cndo2::DoDamp(double rmsDensity, 
                   double** orbitalElectronPopulation, 
                   double const* const* oldOrbitalElectronPopulation, 
                   const Molecule& molecule) const{
   double dampingThresh = Parameters::GetInstance()->GetDampingThreshSCF();
   double dampingWeight = Parameters::GetInstance()->GetDampingWeightSCF();
   if(0.0 < dampingWeight && dampingThresh < rmsDensity){
      stringstream ompErrors;
#pragma omp parallel for schedule(auto)
      for(int j=0; j<molecule.GetTotalNumberAOs(); j++){
         try{
            for(int k=0; k<molecule.GetTotalNumberAOs(); k++){
               orbitalElectronPopulation[j][k] *= (1.0 - dampingWeight);
               orbitalElectronPopulation[j][k] += dampingWeight*oldOrbitalElectronPopulation[j][k];
            }
         }
         catch(MolDSException ex){
#pragma omp critical
            ompErrors << ex.what() << endl ;
         }
      }
      // Exception throwing for omp-region
      if(!ompErrors.str().empty()){
         throw MolDSException(ompErrors.str());
      }
   } 

}

void Cndo2::OutputMOEnergies() const{
   double eV2AU = Parameters::GetInstance()->GetEV2AU();
   this->OutputLog(this->messageEnergyMOTitle);
   for(int mo=0; mo<this->molecule->GetTotalNumberAOs(); mo++){
      string occUnOcc = this->messageUnOcc;
      if(mo < this->molecule->GetTotalNumberValenceElectrons()/2){
         occUnOcc = this->messageOcc;
      }
      this->OutputLog(boost::format("%s\t%d\t%s\t%e\t%e\n") % this->messageEnergyMO
                                                            % mo
                                                            % occUnOcc
                                                            % this->energiesMO[mo] 
                                                            % (this->energiesMO[mo]/eV2AU) );
   }
   this->OutputLog("\n");
}

void Cndo2::OutputSCFEnergies() const{
   double eV2AU = Parameters::GetInstance()->GetEV2AU();

   // electronic energy
   this->OutputLog(this->messageElecEnergyTitle);
   this->OutputLog(boost::format("%s\t%e\t%e\n") % this->messageElecEnergy
                                                 % this->elecSCFEnergy
                                                 % (this->elecSCFEnergy/eV2AU));
   if(Parameters::GetInstance()->RequiresVdWSCF()){
      this->OutputLog(this->messageNoteElecEnergyVdW);
   }
   else{
      this->OutputLog(this->messageNoteElecEnergy);
   }

   // output core repulsion energy
   this->OutputLog(this->messageCoreRepulsionTitle);
   this->OutputLog(boost::format("%s\t%e\t%e\n\n") % this->messageCoreRepulsion
                                                   % this->coreRepulsionEnergy 
                                                   % (this->coreRepulsionEnergy/eV2AU));

   // output van der Waals correction 
   if(Parameters::GetInstance()->RequiresVdWSCF()){
      this->OutputLog(this->messageVdWCorrectionTitle);
      this->OutputLog(boost::format("%s\t%e\t%e\n\n") % this->messageVdWCorrection
                                                      % this->vdWCorrectionEnergy 
                                                      % (this->vdWCorrectionEnergy/eV2AU));
   }
}

void Cndo2::OutputSCFDipole() const{
   int groundState=0;
   double debye2AU = Parameters::GetInstance()->GetDebye2AU();
   double magnitude = 0.0;
   double temp = 0.0;

   // output total dipole moment 
   temp = 0.0;
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][XAxis]+this->coreDipoleMoment[XAxis],2.0);
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][YAxis]+this->coreDipoleMoment[YAxis],2.0);
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis]+this->coreDipoleMoment[ZAxis],2.0);
   magnitude = sqrt(temp);
   this->OutputLog(this->messageTotalDipoleMomentTitle);
   this->OutputLog(boost::format("%s\t%e\t%e\t%e\t%e\t\t%e\t%e\t%e\t%e\n\n") 
      % this->messageTotalDipoleMoment
      % (this->electronicTransitionDipoleMoments[groundState][groundState][XAxis]+this->coreDipoleMoment[XAxis])
      % (this->electronicTransitionDipoleMoments[groundState][groundState][YAxis]+this->coreDipoleMoment[YAxis])
      % (this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis]+this->coreDipoleMoment[ZAxis])
      % magnitude
      % ((this->electronicTransitionDipoleMoments[groundState][groundState][XAxis]+this->coreDipoleMoment[XAxis])/debye2AU)
      % ((this->electronicTransitionDipoleMoments[groundState][groundState][YAxis]+this->coreDipoleMoment[YAxis])/debye2AU)
      % ((this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis]+this->coreDipoleMoment[ZAxis])/debye2AU)
      % (magnitude/debye2AU));

   // output electronic dipole moment 
   temp = 0.0;
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][XAxis],2.0);
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][YAxis],2.0);
   temp += pow(this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis],2.0);
   magnitude = sqrt(temp);
   this->OutputLog(this->messageElectronicDipoleMomentTitle);
   this->OutputLog(boost::format("%s\t%e\t%e\t%e\t%e\t\t%e\t%e\t%e\t%e\n\n") 
      % this->messageElectronicDipoleMoment
      % this->electronicTransitionDipoleMoments[groundState][groundState][XAxis]
      % this->electronicTransitionDipoleMoments[groundState][groundState][YAxis]
      % this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis]
      % magnitude
      % (this->electronicTransitionDipoleMoments[groundState][groundState][XAxis]/debye2AU)
      % (this->electronicTransitionDipoleMoments[groundState][groundState][YAxis]/debye2AU)
      % (this->electronicTransitionDipoleMoments[groundState][groundState][ZAxis]/debye2AU)
      % (magnitude/debye2AU));

   // output core dipole moment 
   temp = 0.0;
   temp += pow(this->coreDipoleMoment[XAxis],2.0);
   temp += pow(this->coreDipoleMoment[YAxis],2.0);
   temp += pow(this->coreDipoleMoment[ZAxis],2.0);
   magnitude = sqrt(temp);
   this->OutputLog(this->messageCoreDipoleMomentTitle);
   this->OutputLog(boost::format("%s\t\t%e\t%e\t%e\t%e\t\t%e\t%e\t%e\t%e\n\n") 
      % this->messageCoreDipoleMoment
      % this->coreDipoleMoment[XAxis]
      % this->coreDipoleMoment[YAxis]
      % this->coreDipoleMoment[ZAxis]
      % magnitude
      % (this->coreDipoleMoment[XAxis]/debye2AU)
      % (this->coreDipoleMoment[YAxis]/debye2AU)
      % (this->coreDipoleMoment[ZAxis]/debye2AU)
      % (magnitude/debye2AU));
}

void Cndo2::OutputSCFMulliken() const{
   this->OutputLog(this->messageMullikenAtomsTitle);
   for(int a=0; a<this->molecule->GetNumberAtoms(); a++){
      Atom* atom = this->molecule->GetAtom(a);
      this->OutputLog(boost::format("%s\t%d\t%s\t%e\t%e\n") % this->messageMullikenAtoms
                                                            % a
                                                            % AtomTypeStr(atom->GetAtomType())
                                                            % atom->GetCoreCharge()
                                                            % (atom->GetCoreCharge()-atomicElectronPopulation[a]));
   }
   this->OutputLog("\n");
}

void Cndo2::OutputSCFResults() const{
   this->OutputMOEnergies();
   this->OutputSCFEnergies();
   this->OutputSCFDipole();
   this->OutputSCFMulliken();
   // ToDo: output eigen-vectors of the Hartree Fock matrix

   // output MOs
   if(Parameters::GetInstance()->RequiresMOPlot()){
      MolDS_base_loggers::MOLogger* moLogger = new MolDS_base_loggers::MOLogger(*this->molecule, 
                                                                                this->fockMatrix, 
                                                                                this->theory);
      moLogger->DrawMO(*(Parameters::GetInstance()->GetIndecesMOPlot()));
      delete moLogger;
   }
}

void Cndo2::CalcElecSCFEnergy(double* elecSCFEnergy, 
                             const Molecule& molecule, 
                             double const* energiesMO, 
                             double const* const* fockMatrix, 
                             double const* const* gammaAB, 
                             double coreRepulsionEnergy,
                             double vdWCorrectionEnergy) const{
   double electronicEnergy = 0.0;
   // use density matrix for electronic energy
   int totalNumberAOs = this->molecule->GetTotalNumberAOs();
   double** fMatrix = NULL;
   double** hMatrix = NULL;
   double** dammyOrbitalElectronPopulation = NULL;
   double* dammyAtomicElectronPopulation = NULL;

   try{
      MallocerFreer::GetInstance()->Malloc<double>(&fMatrix, totalNumberAOs, totalNumberAOs);
      MallocerFreer::GetInstance()->Malloc<double>(&hMatrix, totalNumberAOs,totalNumberAOs);
      MallocerFreer::GetInstance()->Malloc<double>(&dammyOrbitalElectronPopulation,
                                                   totalNumberAOs, 
                                                   totalNumberAOs);
      MallocerFreer::GetInstance()->Malloc<double>(&dammyAtomicElectronPopulation,
                                                   molecule.GetNumberAtoms());
      bool isGuess = false;
      this->CalcFockMatrix(fMatrix, 
                           molecule, 
                           this->overlap, 
                           this->gammaAB,
                           this->orbitalElectronPopulation, 
                           this->atomicElectronPopulation,
                           this->twoElecTwoCore,
                           isGuess);
      this->CalcFockMatrix(hMatrix, 
                           molecule, 
                           this->overlap, 
                           this->gammaAB,
                           dammyOrbitalElectronPopulation, 
                           dammyAtomicElectronPopulation,
                           this->twoElecTwoCore,
                           isGuess);

      for(int i=0; i<totalNumberAOs; i++){
         for(int j=i+1; j<totalNumberAOs; j++){
            fMatrix[j][i] = fMatrix[i][j];
            hMatrix[j][i] = hMatrix[i][j];
         }
      }

      for(int i=0; i<totalNumberAOs; i++){
         for(int j=0; j<totalNumberAOs; j++){
            electronicEnergy += this->orbitalElectronPopulation[j][i]*
                                 (fMatrix[i][j] + hMatrix[i][j]);
         }
      }
      electronicEnergy *= 0.5;
   }
   catch(MolDSException ex){
      this->FreeElecEnergyMatrices(&fMatrix, 
                                   &hMatrix, 
                                   &dammyOrbitalElectronPopulation, 
                                   &dammyAtomicElectronPopulation );
      throw ex;
   }
   this->FreeElecEnergyMatrices(&fMatrix, 
                                &hMatrix, 
                                &dammyOrbitalElectronPopulation, 
                                &dammyAtomicElectronPopulation );

   // use two electrons integrals for electronic energy
   /*
   for(int mo=0; mo<molecule.GetTotalNumberValenceElectrons()/2; mo++){
      electronicEnergy += 2.0*energiesMO[mo];
   }

   for(int moA=0; moA<molecule.GetTotalNumberValenceElectrons()/2; moA++){
      for(int moB=0; moB<molecule.GetTotalNumberValenceElectrons()/2; moB++){

         electronicEnergy -= 2.0*this->GetMolecularIntegralElement(moA, moA, moB, moB, 
                                                              molecule, fockMatrix, gammaAB);
         electronicEnergy += 1.0*this->GetMolecularIntegralElement(moA, moB, moB, moA, 
                                                              molecule, fockMatrix, gammaAB);
      }
   }
   */

   *elecSCFEnergy = electronicEnergy + coreRepulsionEnergy + vdWCorrectionEnergy;
}

void Cndo2::FreeElecEnergyMatrices(double*** fMatrix, 
                                   double*** hMatrix, 
                                   double*** dammyOrbitalElectronPopulation, 
                                   double**  dammyAtomicElectronPopulation ) const{
   int totalNumberAOs = this->molecule->GetTotalNumberAOs();
   MallocerFreer::GetInstance()->Free<double>(fMatrix, totalNumberAOs, totalNumberAOs);
   MallocerFreer::GetInstance()->Free<double>(hMatrix, totalNumberAOs, totalNumberAOs);
   MallocerFreer::GetInstance()->Free<double>(dammyOrbitalElectronPopulation, 
                                              totalNumberAOs,
                                              totalNumberAOs);
   MallocerFreer::GetInstance()->Free<double>(dammyAtomicElectronPopulation,
                                              this->molecule->GetNumberAtoms());
}

// The order of moI, moJ, moK, moL is consistent with Eq. (9) in [RZ_1973]
double Cndo2::GetMolecularIntegralElement(int moI, int moJ, int moK, int moL, 
                                          const Molecule& molecule, 
                                          double const* const* fockMatrix, 
                                          double const* const* gammaAB) const{
   double value = 0.0;
   for(int A=0; A<molecule.GetNumberAtoms(); A++){
      const Atom& atomA = *molecule.GetAtom(A);
      int firstAOIndexA = atomA.GetFirstAOIndex();
      int lastAOIndexA  = atomA.GetLastAOIndex();

      for(int B=0; B<molecule.GetNumberAtoms(); B++){
         const Atom& atomB = *molecule.GetAtom(B);
         int firstAOIndexB = atomB.GetFirstAOIndex();
         int lastAOIndexB  = atomB.GetLastAOIndex();
         double gamma = gammaAB[A][B];

         for(int mu=firstAOIndexA; mu<=lastAOIndexA; mu++){
            for(int nu=firstAOIndexB; nu<=lastAOIndexB; nu++){

               value += gamma*fockMatrix[moI][mu]*fockMatrix[moJ][mu]*fockMatrix[moK][nu]*fockMatrix[moL][nu];
            }
         }

      }
   }
   return value;
}

void Cndo2::UpdateOldOrbitalElectronPopulation(double** oldOrbitalElectronPopulation, 
                                               double const* const* orbitalElectronPopulation,
                                               int numberAOs) const{
   for(int i=0; i<numberAOs; i++){
      for(int j=0; j<numberAOs; j++){
         oldOrbitalElectronPopulation[i][j] = orbitalElectronPopulation[i][j];
      }
   }
}

bool Cndo2::SatisfyConvergenceCriterion(double const* const * oldOrbitalElectronPopulation,
                                        double const* const * orbitalElectronPopulation,
                                        int numberAOs,
                                        double* rmsDensity,
                                        int times) const{
   bool satisfy = false;
   double change = 0.0;
   stringstream ompErrors;
#pragma omp parallel for schedule(auto) reduction(+:change)
   for(int i=0; i<numberAOs; i++){
      try{
         for(int j=0; j<numberAOs; j++){
            change += pow(oldOrbitalElectronPopulation[i][j] - orbitalElectronPopulation[i][j], 2.0);
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
   *rmsDensity = sqrt(change);
  
   this->OutputLog(boost::format("%s%d%s%.15lf\n") % this->messageIterSCF.c_str()
                                                   % times
                                                   % this->messageDensityRMS.c_str()
                                                   % *rmsDensity);

   if(*rmsDensity < Parameters::GetInstance()->GetThresholdSCF()){
      satisfy = true;
   }

   return satisfy; 
}

/*********
 *
 *
 * Upper right part of the Fock matrix is only caluculated.
 *
 *
 * ******/
void Cndo2::CalcFockMatrix(double** fockMatrix, 
                           const Molecule& molecule, 
                           double const* const* overlap, 
                           double const* const* gammaAB,
                           double const* const* orbitalElectronPopulation, 
                           double const* atomicElectronPopulation,
                           double const* const* const* const* const* const* twoElecTwoCore, 
                           bool isGuess) const{
   MallocerFreer::GetInstance()->Initialize<double>(fockMatrix, 
                                                    molecule.GetTotalNumberAOs(), 
                                                    molecule.GetTotalNumberAOs());
   stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
   for(int A=0; A<molecule.GetNumberAtoms(); A++){
      try{
        const Atom& atomA = *molecule.GetAtom(A);
         int firstAOIndexA = atomA.GetFirstAOIndex();
         int lastAOIndexA  = atomA.GetLastAOIndex();
         for(int B=A; B<molecule.GetNumberAtoms(); B++){
            const Atom& atomB = *molecule.GetAtom(B);
            int firstAOIndexB = atomB.GetFirstAOIndex();
            int lastAOIndexB  = atomB.GetLastAOIndex();
            for(int mu=firstAOIndexA; mu<=lastAOIndexA; mu++){
               for(int nu=firstAOIndexB; nu<=lastAOIndexB; nu++){
                  if(mu == nu){
                     // diagonal part
                     fockMatrix[mu][mu] = this->GetFockDiagElement(atomA, 
                                                                   A, 
                                                                   mu, 
                                                                   molecule, 
                                                                   gammaAB,
                                                                   orbitalElectronPopulation, 
                                                                   atomicElectronPopulation,
                                                                   twoElecTwoCore,
                                                                   isGuess);
                  }
                  else if(mu < nu){
                     // upper right part
                     fockMatrix[mu][nu] = this->GetFockOffDiagElement(atomA, 
                                                                      atomB,
                                                                      A, 
                                                                      B, 
                                                                      mu, 
                                                                      nu, 
                                                                      molecule, 
                                                                      gammaAB,
                                                                      overlap,
                                                                      orbitalElectronPopulation, 
                                                                      twoElecTwoCore,
                                                                      isGuess);
                  }
                  else{
                     // lower left part (not calculated)
                  }

               }
            }
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
   /*  
   this->OutputLog("fock matrix\n");
   for(int o=0; o<this->molecule.GetTotalNumberAOs(); o++){
      for(int p=0; p<this->molecule.GetTotalNumberAOs(); p++){
         this->OutputLog(boost::format("%lf\t") % fockMatrix[o][p]);
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n\n");
   */
}

double Cndo2::GetFockDiagElement(const Atom& atomA, 
                                 int indexAtomA, 
                                 int mu, 
                                 const Molecule& molecule, 
                                 double const* const* gammaAB,
                                 double const* const* orbitalElectronPopulation, 
                                 double const* atomicElectronPopulation,
                                 double const* const* const* const* const* const* twoElecTwoCore, 
                                 bool isGuess) const{
   double value;
   int firstAOIndexA = atomA.GetFirstAOIndex();
   value = atomA.GetCoreIntegral(atomA.GetValence(mu-firstAOIndexA), 
                                 gammaAB[indexAtomA][indexAtomA], 
                                 isGuess, this->theory);
   if(!isGuess){
      double temp = atomicElectronPopulation[indexAtomA] 
                   -0.5*orbitalElectronPopulation[mu][mu];
      value += temp*gammaAB[indexAtomA][indexAtomA];

      temp = 0.0;
      for(int BB=0; BB<molecule.GetNumberAtoms(); BB++){
         if(BB != indexAtomA){
            const Atom& atomBB = *molecule.GetAtom(BB);
            temp += ( atomicElectronPopulation[BB] - atomBB.GetCoreCharge()  )
                     *gammaAB[indexAtomA][BB];
         }
      }
      value += temp;
   }

   return value;
}

double Cndo2::GetFockOffDiagElement(const Atom& atomA, 
                                    const Atom& atomB, 
                                    int indexAtomA, 
                                    int indexAtomB, 
                                    int mu, 
                                    int nu, 
                                    const Molecule& molecule, 
                                    double const* const* gammaAB, 
                                    double const* const* overlap,
                                    double const* const* orbitalElectronPopulation, 
                                    double const* const* const* const* const* const* twoElecTwoCore, 
                                    bool isGuess) const{
   double value;
   double K = this->GetBondingAdjustParameterK(atomA.GetValenceShellType(), atomB.GetValenceShellType());
   double bondParameter = 0.5*K*(atomA.GetBondingParameter() + atomB.GetBondingParameter()); 
   value =  bondParameter*overlap[mu][nu];
   if(!isGuess){
      value -= 0.5*orbitalElectronPopulation[mu][nu]*gammaAB[indexAtomA][indexAtomB];
   }
   return value;
}

void Cndo2::CalcOrbitalElectronPopulation(double** orbitalElectronPopulation, 
                                          const Molecule& molecule, 
                                          double const* const* fockMatrix) const{
   int totalNumberAOs = molecule.GetTotalNumberAOs();
   MallocerFreer::GetInstance()->Initialize<double>(orbitalElectronPopulation, totalNumberAOs, totalNumberAOs);

   double** transposedFockMatrix = NULL;
   try{
      MallocerFreer::GetInstance()->Malloc<double>(&transposedFockMatrix, totalNumberAOs, totalNumberAOs);
      for(int mu=0; mu<totalNumberAOs; mu++){
         for(int nu=0; nu<totalNumberAOs; nu++){
            transposedFockMatrix[mu][nu] = fockMatrix[nu][mu];
         }
      }
   
      int numberTotalValenceElectrons = molecule.GetTotalNumberValenceElectrons();
      stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
      for(int mu=0; mu<totalNumberAOs; mu++){
         try{
            for(int nu=mu; nu<totalNumberAOs; nu++){
               double value = 0.0;
               for(int mo=0; mo<numberTotalValenceElectrons/2; mo++){
                  value += transposedFockMatrix[mu][mo]*transposedFockMatrix[nu][mo];
               }
               orbitalElectronPopulation[mu][nu] = 2.0*value;
            }
         }
         catch(MolDSException ex){
#pragma omp critical
            ompErrors << ex.what() << endl ;
         }
      }
      // Exception throwing for omp-region
      if(!ompErrors.str().empty()){
         throw MolDSException(ompErrors.str());
      }
   }
   catch(MolDSException ex){
      MallocerFreer::GetInstance()->Free<double>(&transposedFockMatrix, totalNumberAOs, totalNumberAOs);
      throw ex;
   }
   MallocerFreer::GetInstance()->Free<double>(&transposedFockMatrix, totalNumberAOs, totalNumberAOs);
   
   for(int mu=0; mu<totalNumberAOs; mu++){
      for(int nu=mu+1; nu<totalNumberAOs; nu++){
         orbitalElectronPopulation[nu][mu] = orbitalElectronPopulation[mu][nu];
      }
   }

   /* 
   this->OutputLog("orbital population\n");
   for(int mu=0; mu<totalNumberAOs; mu++){
      for(int nu=0; nu<totalNumberAOs; nu++){
         this->OutputLog(boost::format("%lf\t") % orbitalElectronPopulation[mu][nu]);
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n");
   */
}

void Cndo2::CalcAtomicElectronPopulation(double* atomicElectronPopulation,
                                         double const* const* orbitalElectronPopulation, 
                                         const Molecule& molecule) const{
   int totalNumberAtoms = molecule.GetNumberAtoms();
   MallocerFreer::GetInstance()->Initialize<double>(atomicElectronPopulation, totalNumberAtoms);

   int firstAOIndex = 0;
   int numberAOs = 0;
   for(int A=0; A<totalNumberAtoms; A++){
      firstAOIndex = molecule.GetAtom(A)->GetFirstAOIndex();
      numberAOs = molecule.GetAtom(A)->GetValenceSize();
      atomicElectronPopulation[A] = 0.0;
      for(int i=firstAOIndex; i<firstAOIndex+numberAOs; i++){
         atomicElectronPopulation[A] += orbitalElectronPopulation[i][i];
      }
      //this->OutputLog(boost::format("P_AA[%d]=%lf\n") % A % atomicElectronPopulation[A]);
   }
}

// calculate gammaAB matrix. (B.56) and (B.62) in J. A. Pople book.
void Cndo2::CalcGammaAB(double** gammaAB, const Molecule& molecule) const{
   int totalAtomNumber = molecule.GetNumberAtoms();
   stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
   for(int A=0; A<totalAtomNumber; A++){
      try{
         const Atom& atomA = *molecule.GetAtom(A);
         int na = atomA.GetValenceShellType() + 1;
         double orbitalExponentA = atomA.GetOrbitalExponent(
                                         atomA.GetValenceShellType(), s, this->theory);
         for(int B=A; B<totalAtomNumber; B++){
            const Atom& atomB = *molecule.GetAtom(B);
            int nb = atomB.GetValenceShellType() + 1;
            double orbitalExponentB = atomB.GetOrbitalExponent(
                                            atomB.GetValenceShellType(), s, this->theory);

            double value = 0.0;
            double R = molecule.GetDistanceAtoms(A, B);
            double temp = 0.0;
            if(R>0.0){
               // (B.56)
               value = pow(0.5*R, 2.0*na);
               value *= this->GetReducedOverlap(2*na-1, 0, 2.0*orbitalExponentA*R, 0);

               for(int l=1; l<=2*nb; l++){
                  temp = 0.0;
                  temp = l;
                  temp *= pow(2.0*orbitalExponentB, 2*nb-l);
                  temp /= Factorial(2*nb-l)*2.0*nb;
                  temp *= pow(0.5*R, 2.0*nb-l+2.0*na);
                  temp *= this->GetReducedOverlap(2*na-1, 
                                                  2*nb-l, 
                                                  2.0*orbitalExponentA*R, 
                                                  2.0*orbitalExponentB*R);
                  value -= temp;
               }

               value *= pow(2.0*orbitalExponentA, 2.0*na+1.0);
               value /= Factorial(2*na);
            }
            else{
               // (B.62)
               value =  Factorial(2*na-1);
               value /= pow(2.0*orbitalExponentA, 2.0*na);

               for(int l=1; l<=2*nb; l++){
                  temp = l;
                  temp *= pow(2.0*orbitalExponentB, 2*nb-l);
                  temp *= Factorial(2*na+2*nb-l-1);
                  temp /= Factorial(2*nb-l);
                  temp /= 2.0*nb;
                  temp /= pow( 2.0*orbitalExponentA + 2.0*orbitalExponentB, 2.0*(na+nb)-l );
                  value -= temp;
               }
               value *= pow(2.0*orbitalExponentA, 2.0*na+1);
               value /= Factorial(2*na);
            }
            gammaAB[A][B] = value;
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }

#pragma omp parallel for schedule(auto)
   for(int A=0; A<totalAtomNumber; A++){
      for(int B=0; B<A; B++){
         gammaAB[A][B] = gammaAB[B][A];
      }
   }
   
   /* 
   this->OutputLog("gamma matrix\n");
   for(int A=0; A<totalAtomNumber; A++){
      for(int B=0; B<totalAtomNumber; B++){
         this->OutputLog(boost::format("gammaAB[%d][%d]=%lf\n") % A % B % gammaAB[A][B]);
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n");
   */

}

void Cndo2::CalcCoreDipoleMoment(double* coreDipoleMoment,
                                 const Molecule& molecule) const{

   for(int i=0; i<CartesianType_end; i++){
      coreDipoleMoment[i] = 0.0;
      for(int A=0; A<molecule.GetNumberAtoms(); A++){
         coreDipoleMoment[i] += molecule.GetAtom(A)->GetCoreCharge()
                               *(molecule.GetAtom(A)->GetXyz()[i] - molecule.GetXyzCOC()[i]);
      }
   }
}

void Cndo2::CalcElectronicDipoleMomentGroundState(double*** electronicTransitionDipoleMoments,
                                                  double const* const* const* cartesianMatrix,
                                                  const Molecule& molecule,
                                                  double const* const* orbitalElectronPopulation,
                                                  double const* const* overlap) const{
   int groundState = 0;
   stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
   for(int axis=0; axis<CartesianType_end; axis++){
      try{
         electronicTransitionDipoleMoments[groundState][groundState][axis] = this->GetElectronicTransitionDipoleMoment(
                                                                                   groundState,
                                                                                   groundState,
                                                                                   static_cast<CartesianType>(axis),
                                                                                   NULL,
                                                                                   NULL,
                                                                                   cartesianMatrix,
                                                                                   molecule,
                                                                                   orbitalElectronPopulation,
                                                                                   overlap,
                                                                                   NULL);
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
}

double Cndo2::GetElectronicTransitionDipoleMoment(int to, int from, CartesianType axis,
                                                  double const* const* fockMatrix,
                                                  double const* const* matrixCIS,
                                                  double const* const* const* cartesianMatrix,
                                                  const MolDS_base::Molecule& molecule, 
                                                  double const* const* orbitalElectronPopulation,
                                                  double const* const* overlap,
                                                  double const* groundStateDipole) const{
   double value = 0.0;
   int groundState = 0;
   if(from == groundState && to == groundState){
      int totalAONumber = molecule.GetTotalNumberAOs();
      for(int mu=0; mu<totalAONumber; mu++){
         for(int nu=0; nu<totalAONumber; nu++){
            value -= orbitalElectronPopulation[mu][nu]
                    *(cartesianMatrix[mu][nu][axis]-molecule.GetXyzCOC()[axis]*overlap[mu][nu]);
         }
      }
   }
   else{
      stringstream ss;
      ss << this->errorMessageGetElectronicTransitionDipoleMomentBadState;
      ss << this->errorMessageFromState << from << endl;
      ss << this->errorMessageToState << to << endl;
      ss << this->errorMessageCartesianType << CartesianTypeStr(axis) << endl;
      throw MolDSException(ss.str());
   }
   return value;
}

// calculate Cartesian matrix between atomic orbitals. 
// The analytic Cartesian matrix is calculated with Gaussian expansion technique written in [DY_1977]
void Cndo2::CalcCartesianMatrixByGTOExpansion(double*** cartesianMatrix, 
                                              const Molecule& molecule, 
                                              STOnGType stonG) const{
   int totalAONumber = molecule.GetTotalNumberAOs();
   int totalAtomNumber = molecule.GetNumberAtoms();

   stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
   for(int A=0; A<totalAtomNumber; A++){
      try{
         const Atom& atomA = *molecule.GetAtom(A);
         int firstAOIndexAtomA = atomA.GetFirstAOIndex();
         for(int B=0; B<totalAtomNumber; B++){
            const Atom& atomB = *molecule.GetAtom(B);
            int firstAOIndexAtomB = atomB.GetFirstAOIndex();
            for(int a=0; a<atomA.GetValenceSize(); a++){
               for(int b=0; b<atomB.GetValenceSize(); b++){
                  int mu = firstAOIndexAtomA + a;      
                  int nu = firstAOIndexAtomB + b;      
                  for(int i=0; i<CartesianType_end; i++){
                     double value = this->GetCartesianMatrixElementByGTOExpansion(atomA, 
                                                                                  a, 
                                                                                  atomB, 
                                                                                  b, 
                                                                                  static_cast<CartesianType>(i),
                                                                                  stonG);
                     cartesianMatrix[mu][nu][i] = value;
                  }
               }
            }
            
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
   /* 
   this->OutputLog("cartesian matrix\n"); 
   for(int o=0; o<molecule.GetTotalNumberAOs(); o++){
      for(int p=0; p<molecule.GetTotalNumberAOs(); p++){
         for(int i=0; i<CartesianType_end; i++){
            this->OutputLog(boost::format("%lf\t") % cartesianMatrix[o][p][i]);
         }
         this->OutputLog("\n");
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n");
   */
}

// Calculate elements of Cartesian matrix between atomic orbitals. 
// The analytic Cartesian matrix is calculated with Gaussian expansion technique written in [DY_1977]
double Cndo2::GetCartesianMatrixElementByGTOExpansion(const Atom& atomA, int valenceIndexA, 
                                                      const Atom& atomB, int valenceIndexB,
                                                      CartesianType axis,
                                                      STOnGType stonG) const{
   double value = 0.0;
   ShellType shellTypeA = atomA.GetValenceShellType();
   ShellType shellTypeB = atomB.GetValenceShellType();
   OrbitalType valenceOrbitalA = atomA.GetValence(valenceIndexA);
   OrbitalType valenceOrbitalB = atomB.GetValence(valenceIndexB);
   double orbitalExponentA = atomA.GetOrbitalExponent(atomA.GetValenceShellType(), 
                                                      valenceOrbitalA, 
                                                      this->theory);
   double orbitalExponentB = atomB.GetOrbitalExponent(atomB.GetValenceShellType(), 
                                                      valenceOrbitalB, 
                                                      this->theory);
   double gaussianExponentA = 0.0;
   double gaussianExponentB = 0.0;
   double Rab = sqrt( pow(atomA.GetXyz()[XAxis]-atomB.GetXyz()[XAxis], 2.0) 
                     +pow(atomA.GetXyz()[YAxis]-atomB.GetXyz()[YAxis], 2.0) 
                     +pow(atomA.GetXyz()[ZAxis]-atomB.GetXyz()[ZAxis], 2.0) );
   double temp = 0.0;
   for(int i=0; i<=stonG; i++){
      for(int j=0; j<=stonG; j++){
         temp = GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                               shellTypeA, 
                                                               valenceOrbitalA, 
                                                               i); 
         temp *= GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                                shellTypeB, 
                                                                valenceOrbitalB, 
                                                                j); 
         gaussianExponentA = pow(orbitalExponentA, 2.0) *
                             GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeA, 
                                                                         valenceOrbitalA, 
                                                                         i);
         gaussianExponentB = pow(orbitalExponentB, 2.0) *
                             GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeB, 
                                                                         valenceOrbitalB, 
                                                                         j);
         temp *= this->GetGaussianCartesianMatrix(atomA.GetAtomType(), 
                                                  valenceOrbitalA, 
                                                  gaussianExponentA, 
                                                  atomA.GetXyz(),
                                                  atomB.GetAtomType(), 
                                                  valenceOrbitalB, 
                                                  gaussianExponentB,
                                                  atomB.GetXyz(), 
                                                  Rab,
                                                  axis);
         value += temp;
      }
   }
   return value;
}

// calculate gaussian Caretesian integrals. 
double Cndo2::GetGaussianCartesianMatrix(AtomType atomTypeA, 
                                         OrbitalType valenceOrbitalA, 
                                         double gaussianExponentA, 
                                         double const* xyzA,
                                         AtomType atomTypeB, 
                                         OrbitalType valenceOrbitalB, 
                                         double gaussianExponentB,
                                         double const* xyzB,
                                         double Rab,
                                         CartesianType axis) const{

   double value = 0.0;
   double beta = gaussianExponentA + gaussianExponentB;
   double dxyz[CartesianType_end] = {xyzA[XAxis] - xyzB[XAxis], 
                                     xyzA[YAxis] - xyzB[YAxis], 
                                     xyzA[ZAxis] - xyzB[ZAxis]};
   if(valenceOrbitalA == s && valenceOrbitalB == s){
      value = gaussianExponentA*xyzA[axis] + gaussianExponentB*xyzB[axis];
      value /= beta;
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value *= sasb;
   }
   else if( (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == s && axis == XAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == pz) ||
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == s && axis == YAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == pz) ||
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == dxxyy) ||
            (valenceOrbitalA == s && axis == ZAxis && valenceOrbitalB == dzz) ){
      OrbitalType pOrbital;
      if(axis == XAxis){
         pOrbital = px;
      }
      else if(axis == YAxis){
         pOrbital = py;
      }
      else if(axis == ZAxis){
         pOrbital = pz;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double overlap2 = this->GetGaussianOverlap(atomTypeA,
                                                 pOrbital, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      value = overlap2/(2.0*sqrt(gaussianExponentA))+xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == px    && axis == XAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == py    && axis == XAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == pz    && axis == XAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == dxy   && axis == XAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == dyz   && axis == XAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == dzx   && axis == XAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dzz   && axis == XAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == px    && axis == YAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == py    && axis == YAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == pz    && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dxy   && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dyz   && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dzx   && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dzz   && axis == YAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == px    && axis == ZAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == py    && axis == ZAxis && valenceOrbitalB == s) || 
            (valenceOrbitalA == pz    && axis == ZAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dxy   && axis == ZAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dyz   && axis == ZAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dzx   && axis == ZAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == s) ||
            (valenceOrbitalA == dzz   && axis == ZAxis && valenceOrbitalB == s) ){
      OrbitalType pOrbital;
      if(axis == XAxis){
         pOrbital = px;
      }
      else if(axis == YAxis){
         pOrbital = py;
      }
      else if(axis == ZAxis){
         pOrbital = pz;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double overlap2 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 pOrbital, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      value = overlap2/(2.0*sqrt(gaussianExponentB))+xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) ){
      double temp1 = gaussianExponentA*xyzA[axis] + gaussianExponentB*xyzB[axis];
      double temp2 = gaussianExponentA*xyzA[axis] - gaussianExponentA*xyzB[axis];
      double temp3 = gaussianExponentB*xyzA[axis] - gaussianExponentB*xyzB[axis];
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(temp1+temp2-temp3);
      value -= temp1*temp2*temp3*pow(beta,-1.0);
      value *= 4.0*sqrt(gaussianExponentA*gaussianExponentB)*pow(beta,-2.0);
      value *= sasb;
   }
   else if( (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == valenceOrbitalA) ){
      CartesianType piDirection;
      if(valenceOrbitalA == px){
         piDirection = XAxis;
      }
      else if(valenceOrbitalA == py){
         piDirection = YAxis;
      }
      else if(valenceOrbitalA == pz){
         piDirection = ZAxis;
      }
      double temp1 = gaussianExponentA*xyzA[piDirection] - gaussianExponentA*xyzB[piDirection];
      double temp2 = gaussianExponentB*xyzA[piDirection] - gaussianExponentB*xyzB[piDirection];
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5 - temp1*temp2*pow(beta,-1.0);
      value *= gaussianExponentA*xyzA[axis] + gaussianExponentB*xyzB[axis];
      value *= 4.0*sqrt(gaussianExponentA*gaussianExponentB)*pow(beta,-2.0);
      value *= sasb;
   }
   else if( (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == py) ){
      CartesianType piDirectionA;
      if(valenceOrbitalA == px){
         piDirectionA = XAxis;
      }
      else if(valenceOrbitalA == py){
         piDirectionA = YAxis;
      }
      else if(valenceOrbitalA == pz){
         piDirectionA = ZAxis;
      }
      double temp1 = gaussianExponentA*xyzA[axis] + gaussianExponentB*xyzB[axis];
      double temp2 = gaussianExponentA*xyzA[axis] - gaussianExponentA*xyzB[axis];
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5 + temp1*temp2*pow(beta,-1.0);
      value *= gaussianExponentB*xyzA[piDirectionA] - gaussianExponentB*xyzB[piDirectionA];
      value *= -4.0*sqrt(gaussianExponentA*gaussianExponentB)*pow(beta,-2.0);
      value *= sasb;
   }
   else if( (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == py) ){
      CartesianType piDirectionB;
      if(valenceOrbitalB == px){
         piDirectionB = XAxis;
      }
      else if(valenceOrbitalB == py){
         piDirectionB = YAxis;
      }
      else if(valenceOrbitalB == pz){
         piDirectionB = ZAxis;
      }
      double temp1 = gaussianExponentA*xyzA[axis] + gaussianExponentB*xyzB[axis];
      double temp2 = gaussianExponentB*xyzA[axis] - gaussianExponentB*xyzB[axis];
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5 - temp1*temp2*pow(beta,-1.0);
      value *= gaussianExponentA*xyzA[piDirectionB] - gaussianExponentA*xyzB[piDirectionB];
      value *= 4.0*sqrt(gaussianExponentA*gaussianExponentB)*pow(beta,-2.0);
      value *= sasb;
   }
   else if( (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == py) ||
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == py) ||
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == px) ||
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == pz) ){
      CartesianType piDirectionA;
      CartesianType piDirectionB;
      if(valenceOrbitalA == px){
         piDirectionA = XAxis;
      }
      else if(valenceOrbitalA == py){
         piDirectionA = YAxis;
      }
      else if(valenceOrbitalA == pz){
         piDirectionA = ZAxis;
      }
      if(valenceOrbitalB == px){
         piDirectionB = XAxis;
      }
      else if(valenceOrbitalB == py){
         piDirectionB = YAxis;
      }
      else if(valenceOrbitalB == pz){
         piDirectionB = ZAxis;
      }
      double temp1 = gaussianExponentB*xyzA[piDirectionA] - gaussianExponentB*xyzB[piDirectionA];
      double temp2 = gaussianExponentA*xyzA[axis]         + gaussianExponentB*xyzB[axis];
      double temp3 = gaussianExponentA*xyzA[piDirectionB] - gaussianExponentA*xyzB[piDirectionB];
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -4.0*sqrt(gaussianExponentA*gaussianExponentB)*pow(beta,-3.0);
      value *= temp1*temp2*temp3;
      value *= sasb;
   }
   else if( (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == dzx) ){
      CartesianType anotherAxis;
      if(valenceOrbitalB == dxy){
         if(axis == XAxis){
            anotherAxis = YAxis;
         }
         else{
            anotherAxis = XAxis;
         }
      }
      else if(valenceOrbitalB == dyz){
         if(axis == YAxis){
            anotherAxis = ZAxis;
         }
         else{
            anotherAxis = YAxis;
         }
      }
      else if(valenceOrbitalB == dzx){
         if(axis == ZAxis){
            anotherAxis = XAxis;
         }
         else{
            anotherAxis = ZAxis;
         }
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*gaussianExponentA*dxyz[axis]
             -gaussianExponentB*dxyz[axis]
             +pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[axis],3.0)/beta;
      value *= 8.0*pow(gaussianExponentA, 1.5)*gaussianExponentB*pow(beta,-3.0)*dxyz[anotherAxis]*sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == py) ||
            (valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == py) ||
            (valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == pz) ||
            (valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == pz) ||
            (valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == px) ){
      CartesianType anotherAxis;
      if(valenceOrbitalA == dxy){
         if(axis == XAxis){
            anotherAxis = YAxis;
         }
         else{
            anotherAxis = XAxis;
         }
      }
      else if(valenceOrbitalA == dyz){
         if(axis == YAxis){
            anotherAxis = ZAxis;
         }
         else{
            anotherAxis = YAxis;
         }
      }
      else if(valenceOrbitalA == dzx){
         if(axis == ZAxis){
            anotherAxis = XAxis;
         }
         else{
            anotherAxis = ZAxis;
         }
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*gaussianExponentB*dxyz[axis]
             -gaussianExponentA*dxyz[axis]
             +pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[axis],3.0)/beta;
      value *= 8.0*pow(gaussianExponentB, 1.5)*gaussianExponentA*pow(beta,-3.0)*dxyz[anotherAxis]*sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == dzx) ){
      CartesianType anotherAxis1;
      CartesianType anotherAxis2;
      if(axis == XAxis){
         anotherAxis1 = YAxis;
         anotherAxis2 = ZAxis;
      }
      else if(axis == YAxis){
         anotherAxis1 = XAxis;
         anotherAxis2 = ZAxis;
      }
      else if(axis == ZAxis){
         anotherAxis1 = XAxis;
         anotherAxis2 = YAxis;
      }
      double overlap1=0.0; 
      overlap1 = this->GetGaussianOverlap(atomTypeA,
                                          valenceOrbitalA, 
                                          gaussianExponentA, 
                                          atomTypeB, 
                                          valenceOrbitalB, 
                                          gaussianExponentB, 
                                          dxyz[XAxis], 
                                          dxyz[YAxis], 
                                          dxyz[ZAxis], 
                                          Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5+pow(gaussianExponentB*dxyz[axis], 2.0)/beta;
      value *= 8.0*pow(gaussianExponentA, 2.5)*gaussianExponentB*pow(beta, -3.0)*sasb;
      value *= dxyz[anotherAxis1]*dxyz[anotherAxis2];
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == ZAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dyz && axis == XAxis && valenceOrbitalB == px) ||
            (valenceOrbitalA == dzx && axis == YAxis && valenceOrbitalB == py) ){
      CartesianType anotherAxis1;
      CartesianType anotherAxis2;
      if(axis == XAxis){
         anotherAxis1 = YAxis;
         anotherAxis2 = ZAxis;
      }
      else if(axis == YAxis){
         anotherAxis1 = XAxis;
         anotherAxis2 = ZAxis;
      }
      else if(axis == ZAxis){
         anotherAxis1 = XAxis;
         anotherAxis2 = YAxis;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB, 
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5+pow(gaussianExponentA*dxyz[axis], 2.0)/beta;
      value *= 8.0*pow(gaussianExponentB, 2.5)*gaussianExponentA*pow(beta, -3.0)*sasb;
      value *= dxyz[anotherAxis1]*dxyz[anotherAxis2];
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == dxy) || 
            (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == dyz) || 
            (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == dzx) || 
            (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == dxxyy) || 
            (valenceOrbitalA == px && axis == YAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == px && axis == ZAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == py && axis == XAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == py && axis == ZAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == pz && axis == XAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == pz && axis == YAxis && valenceOrbitalB == dzz) ){
      OrbitalType dOrbital;
      if( (valenceOrbitalA == py && axis == XAxis) || 
          (valenceOrbitalA == px && axis == YAxis) ){
         dOrbital = dxy;
      }
      else if( (valenceOrbitalA == py && axis == ZAxis) || 
               (valenceOrbitalA == pz && axis == YAxis) ){
         dOrbital = dyz;
      }
      else if( (valenceOrbitalA == px && axis == ZAxis) || 
               (valenceOrbitalA == pz && axis == XAxis) ){
         dOrbital = dzx;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double overlap2 = this->GetGaussianOverlap(atomTypeA,
                                                 dOrbital, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      value = overlap2/(2.0*sqrt(gaussianExponentA))+xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dxy   && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dxy   && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dxy   && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dxy   && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dxy   && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dxy   && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dyz   && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dyz   && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dyz   && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dyz   && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dyz   && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dyz   && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dzx   && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dzx   && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dzx   && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dzx   && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dzx   && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dzx   && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dzz   && axis == YAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dzz   && axis == ZAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dzz   && axis == XAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dzz   && axis == ZAxis && valenceOrbitalB == py) || 
            (valenceOrbitalA == dzz   && axis == XAxis && valenceOrbitalB == pz) || 
            (valenceOrbitalA == dzz   && axis == YAxis && valenceOrbitalB == pz) ){
      OrbitalType dOrbital;
      if( (valenceOrbitalB == py && axis == XAxis) || 
          (valenceOrbitalB == px && axis == YAxis) ){
         dOrbital = dxy;
      }
      else if( (valenceOrbitalB == py && axis == ZAxis) || 
               (valenceOrbitalB == pz && axis == YAxis) ){
         dOrbital = dyz;
      }
      else if( (valenceOrbitalB == px && axis == ZAxis) || 
               (valenceOrbitalB == pz && axis == XAxis) ){
         dOrbital = dzx;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double overlap2 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 dOrbital, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      value = overlap2/(2.0*sqrt(gaussianExponentB))+xyzB[axis]*overlap1;
   }
   else if(valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == dxxyy){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[XAxis],2.0)/beta;
      value += 0.5*pow(gaussianExponentA,2.0)*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta;
      value += pow(gaussianExponentA*gaussianExponentB*dxyz[XAxis]/beta,2.0)
              *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentA,0.5)*gaussianExponentB*pow(beta,-2.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == px){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[XAxis],2.0)/beta;
      value += 0.5*pow(gaussianExponentB,2.0)*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta;
      value += pow(gaussianExponentA*gaussianExponentB*dxyz[XAxis]/beta,2.0)
              *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentB,0.5)*gaussianExponentA*pow(beta,-2.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if(valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == dxxyy){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[YAxis],2.0)/beta;
      value += 0.5*pow(gaussianExponentA,2.0)*(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/beta;
      value += pow(gaussianExponentA*gaussianExponentB*dxyz[YAxis]/beta,2.0)
              *(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0));
      value *= -4.0*pow(gaussianExponentA,0.5)*gaussianExponentB*pow(beta,-2.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == py){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[YAxis],2.0)/beta;
      value += 0.5*pow(gaussianExponentB,2.0)*(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/beta;
      value += pow(gaussianExponentA*gaussianExponentB*dxyz[YAxis]/beta,2.0)
              *(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0));
      value *= -4.0*pow(gaussianExponentB,0.5)*gaussianExponentA*pow(beta,-2.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if(valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == dxxyy){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value += pow(gaussianExponentB*dxyz[ZAxis],2.0)
              *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))
              /beta;
      value *= 4.0*pow(gaussianExponentA,2.5)*gaussianExponentB*pow(beta,-3.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == pz){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value += pow(gaussianExponentA*dxyz[ZAxis],2.0)
              *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))
              /beta;
      value *= 4.0*pow(gaussianExponentB,2.5)*gaussianExponentA*pow(beta,-3.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == px && axis == XAxis && valenceOrbitalB == dzz) || 
            (valenceOrbitalA == py && axis == YAxis && valenceOrbitalB == dzz) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5
             +2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[axis],2.0)/beta
             +0.5*pow(gaussianExponentA,2.0)
                 *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA*gaussianExponentB*dxyz[axis]/beta,2.0)
              *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentA,0.5)*gaussianExponentB*pow(beta,-2.0)/sqrt(3.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == px) || 
            (valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == py) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5
             +2.0*gaussianExponentA*gaussianExponentB*pow(dxyz[axis],2.0)/beta
             +0.5*pow(gaussianExponentB,2.0)
                 *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA*gaussianExponentB*dxyz[axis]/beta,2.0)
              *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentB,0.5)*gaussianExponentA*pow(beta,-2.0)/sqrt(3.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if(valenceOrbitalA == pz && axis == ZAxis && valenceOrbitalB == dzz){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 1.0
             -4.0*gaussianExponentA*gaussianExponentB*pow(dxyz[axis],2.0)/beta
             +0.5*pow(gaussianExponentA,2.0)
                 *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA*gaussianExponentB*dxyz[axis]/beta,2.0)
              *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentA,0.5)*gaussianExponentB*pow(beta,-2.0)/sqrt(3.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == pz){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 1.0
             -4.0*gaussianExponentA*gaussianExponentB*pow(dxyz[axis],2.0)/beta
             +0.5*pow(gaussianExponentB,2.0)
                 *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA*gaussianExponentB*dxyz[axis]/beta,2.0)
              *(2.0*pow(dxyz[ZAxis],2.0)-pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0));
      value *= 4.0*pow(gaussianExponentB,0.5)*gaussianExponentA*pow(beta,-2.0)/sqrt(3.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == valenceOrbitalA) ){
      CartesianType anotherAxis;
      if(valenceOrbitalB == dxy){
         if(axis == XAxis){
            anotherAxis = YAxis;
         }
         else{
            anotherAxis = XAxis;
         }
      }
      else if(valenceOrbitalB == dyz){
         if(axis == YAxis){
            anotherAxis = ZAxis;
         }
         else{
            anotherAxis = YAxis;
         }
      }
      else if(valenceOrbitalB == dzx){
         if(axis == ZAxis){
            anotherAxis = XAxis;
         }
         else{
            anotherAxis = ZAxis;
         }
      }
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(gaussianExponentA-gaussianExponentB)*dxyz[axis]
             +(gaussianExponentA-gaussianExponentB)*(gaussianExponentA*gaussianExponentB)
              *dxyz[axis]*pow(dxyz[anotherAxis],2.0)/beta;
      value *= 8.0*(gaussianExponentA*gaussianExponentB)*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dxy   && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dyz   && axis == XAxis && valenceOrbitalB == valenceOrbitalA) || 
            (valenceOrbitalA == dzx   && axis == YAxis && valenceOrbitalB == valenceOrbitalA) ||
            (valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) ){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      value = axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == valenceOrbitalA) ||
            (valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == valenceOrbitalA) ){
      CartesianType anotherAxis;
      if(axis == XAxis){
         anotherAxis = YAxis;
      }
      else{
         anotherAxis = XAxis;
      }
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = (gaussianExponentA-gaussianExponentB)*dxyz[axis]
             -(gaussianExponentA-gaussianExponentB)*(gaussianExponentA*gaussianExponentB)
              *(pow(dxyz[axis],2.0) - pow(dxyz[anotherAxis],2.0))*dxyz[axis]/beta;
      value *= 4.0*(gaussianExponentA*gaussianExponentB)*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == valenceOrbitalA) ||
            (valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == valenceOrbitalA) ){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = (gaussianExponentA-gaussianExponentB)*dxyz[axis]
             -(gaussianExponentA-gaussianExponentB)*(gaussianExponentA*gaussianExponentB)
              *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))*dxyz[axis]/beta;
      value *= 4.0*(gaussianExponentA*gaussianExponentB)*pow(beta,-3.0)/3.0;
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == valenceOrbitalA) {
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 2.0*(gaussianExponentA-gaussianExponentB)*dxyz[axis]
             -(gaussianExponentA-gaussianExponentB)*(gaussianExponentA*gaussianExponentB)
              *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))*dxyz[axis]/beta;
      value *= 8.0*(gaussianExponentA*gaussianExponentB)*pow(beta,-3.0)/3.0;
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == dzx) ){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -8.0*pow(gaussianExponentA*gaussianExponentB,2.0)*dxyz[XAxis]*dxyz[YAxis]*dxyz[ZAxis]
             *(gaussianExponentA-gaussianExponentB)*pow(beta,-4.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == dzx) ){
      CartesianType anotherAxis1;
      CartesianType anotherAxis2;
      if(valenceOrbitalA == dxy && valenceOrbitalB == dyz){
         anotherAxis1 = YAxis;
         anotherAxis2 = ZAxis;
      }
      else if(valenceOrbitalA == dyz && valenceOrbitalB == dzx){
         anotherAxis1 = ZAxis;
         anotherAxis2 = XAxis;
      }
      else if(valenceOrbitalA == dzx && valenceOrbitalB == dxy){
         anotherAxis1 = XAxis;
         anotherAxis2 = YAxis;
      }
      else if(valenceOrbitalA == dyz && valenceOrbitalB == dxy){
         anotherAxis1 = YAxis;
         anotherAxis2 = XAxis;
      }
      else if(valenceOrbitalA == dzx && valenceOrbitalB == dyz){
         anotherAxis1 = ZAxis;
         anotherAxis2 = YAxis;
      }
      else if(valenceOrbitalA == dxy && valenceOrbitalB == dzx){
         anotherAxis1 = XAxis;
         anotherAxis2 = ZAxis;
      }
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-gaussianExponentA*gaussianExponentB*pow(dxyz[anotherAxis1],2.0)/beta;
      value *= 8.0*pow(gaussianExponentA,2.0)*gaussianExponentB*pow(beta,-3.0)*dxyz[anotherAxis2];
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dyz && axis == XAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dzx && axis == YAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dxy && axis == ZAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dxy && axis == ZAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dyz && axis == XAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dzx && axis == YAxis && valenceOrbitalB == dxy) ){
      CartesianType anotherAxis1;
      CartesianType anotherAxis2;
      if(valenceOrbitalA == dyz && valenceOrbitalB == dxy){
         anotherAxis1 = YAxis;
         anotherAxis2 = ZAxis;
      }
      else if(valenceOrbitalA == dzx && valenceOrbitalB == dyz){
         anotherAxis1 = ZAxis;
         anotherAxis2 = XAxis;
      }
      else if(valenceOrbitalA == dxy && valenceOrbitalB == dzx){
         anotherAxis1 = XAxis;
         anotherAxis2 = YAxis;
      }
      else if(valenceOrbitalA == dxy && valenceOrbitalB == dyz){
         anotherAxis1 = YAxis;
         anotherAxis2 = XAxis;
      }
      else if(valenceOrbitalA == dyz && valenceOrbitalB == dzx){
         anotherAxis1 = ZAxis;
         anotherAxis2 = YAxis;
      }
      else if(valenceOrbitalA == dzx && valenceOrbitalB == dxy){
         anotherAxis1 = XAxis;
         anotherAxis2 = ZAxis;
      }

      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5-gaussianExponentA*gaussianExponentB*pow(dxyz[anotherAxis1],2.0)/beta;
      value *= -8.0*pow(gaussianExponentB,2.0)*gaussianExponentA*pow(beta,-3.0)*dxyz[anotherAxis2];
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == dxy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*beta
             -pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[XAxis],2.0)/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= 8.0*gaussianExponentA*gaussianExponentB*pow(beta,-3.0)*dxyz[YAxis];
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*beta
             -pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[XAxis],2.0)/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= -8.0*gaussianExponentA*gaussianExponentB*pow(beta,-3.0)*dxyz[YAxis];
      value *= sasb;
      value += axisAverage*overlap1;
   } 
   else if(valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == dxy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[YAxis],2.0)/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= 8.0*gaussianExponentA*gaussianExponentB*pow(beta,-3.0)*dxyz[XAxis];
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[YAxis],2.0)/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= -8.0*gaussianExponentA*gaussianExponentB*pow(beta,-3.0)*dxyz[XAxis];
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == dxy) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 8.0*pow(gaussianExponentA,4.0)*pow(gaussianExponentB,3.0)
             *dxyz[XAxis]*dxyz[YAxis]*dxyz[ZAxis]
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/pow(beta,5.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dyz && axis == XAxis && valenceOrbitalB == dxxyy) ||
            (valenceOrbitalA == dzx && axis == YAxis && valenceOrbitalB == dxxyy) ||
            (valenceOrbitalA == dxy && axis == ZAxis && valenceOrbitalB == dxxyy) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -8.0*pow(gaussianExponentA,3.0)*pow(gaussianExponentB,4.0)
             *dxyz[XAxis]*dxyz[YAxis]*dxyz[ZAxis]
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/pow(beta,5.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == dyz){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*gaussianExponentA
             +pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[YAxis],2.0)/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= 8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*gaussianExponentB
             +pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[YAxis],2.0)/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/(2.0*beta);
      value *= -8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == dzx){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*gaussianExponentA
             +pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[XAxis],2.0)/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/(2.0*beta);
      value *= -8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = -0.5*gaussianExponentB
             +pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[YAxis],2.0)/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/(2.0*beta);
      value *= 8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == dyz){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA*gaussianExponentB*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta+1.0;
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB,2.0)*dxyz[YAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA*gaussianExponentB*(pow(dxyz[XAxis],2.0)-pow(dxyz[YAxis],2.0))/beta+1.0;
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA,2.0)*dxyz[YAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == dzx){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA*gaussianExponentB*(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/beta+1.0;
      value *= -4.0*gaussianExponentA*pow(gaussianExponentB,2.0)*dxyz[XAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA*gaussianExponentB*(pow(dxyz[YAxis],2.0)-pow(dxyz[XAxis],2.0))/beta+1.0;
      value *= 4.0*gaussianExponentB*pow(gaussianExponentA,2.0)*dxyz[XAxis]*pow(beta,-3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == dzx) ||
            (valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == dxy) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0);
      value *= dxyz[XAxis]*dxyz[YAxis]*dxyz[ZAxis];
      value *= 8.0*pow(gaussianExponentA,4.0)*pow(gaussianExponentB,3.0);
      value /= sqrt(3.0)*pow(beta,5.0);
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dyz && axis == XAxis && valenceOrbitalB == dzz) ||
            (valenceOrbitalA == dzx && axis == YAxis && valenceOrbitalB == dzz) ||
            (valenceOrbitalA == dxy && axis == ZAxis && valenceOrbitalB == dzz) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0);
      value *= dxyz[XAxis]*dxyz[YAxis]*dxyz[ZAxis];
      value *= -8.0*pow(gaussianExponentB,4.0)*pow(gaussianExponentA,3.0);
      value /= sqrt(3.0)*pow(beta,5.0);
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == dxy) ||
            (valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == dxy) ){
      CartesianType anotherAxis;
      if(axis == XAxis){
         anotherAxis = YAxis;
      }
      else if(axis == YAxis){
         anotherAxis = XAxis;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(gaussianExponentB-gaussianExponentA)
             +3.0*pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[axis],2.0)*pow(beta,-2.0)
             +gaussianExponentA*pow(gaussianExponentB,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentA,3.0)*pow(gaussianExponentB,2.0)*pow(dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))*pow(beta,-2.0);
      value *= 8.0*gaussianExponentA*gaussianExponentB*dxyz[anotherAxis]/(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dxy && axis == XAxis && valenceOrbitalB == dzz) ||
            (valenceOrbitalA == dxy && axis == YAxis && valenceOrbitalB == dzz) ){
      CartesianType anotherAxis;
      if(axis == XAxis){
         anotherAxis = YAxis;
      }
      else if(axis == YAxis){
         anotherAxis = XAxis;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = 0.5*(gaussianExponentA-gaussianExponentB)
             +3.0*pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[axis],2.0)*pow(beta,-2.0)
             +gaussianExponentB*pow(gaussianExponentA,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentB,3.0)*pow(gaussianExponentA,2.0)*pow(dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))*pow(beta,-2.0);
      value *= -8.0*gaussianExponentA*gaussianExponentB*dxyz[anotherAxis]/(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == dzx) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentB-0.5*gaussianExponentA
             +gaussianExponentA*pow(gaussianExponentB,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentA,3.0)*pow(gaussianExponentB*dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))
             *pow(beta,-2.0);
      value *= 8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]
              /(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dyz && axis == YAxis && valenceOrbitalB == dzz) ||
            (valenceOrbitalA == dzx && axis == XAxis && valenceOrbitalB == dzz) ){
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA-0.5*gaussianExponentB
             +gaussianExponentB*pow(gaussianExponentA,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentB,3.0)*pow(gaussianExponentA*dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))
             *pow(beta,-2.0);
      value *= -8.0*gaussianExponentA*gaussianExponentB*dxyz[ZAxis]
              /(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == dyz) ||
            (valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == dzx) ){
      CartesianType anotherAxis;
      if(valenceOrbitalB == dyz){
         anotherAxis = YAxis;
      }
      else if(valenceOrbitalB == dzx){
         anotherAxis = XAxis;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA-0.5*gaussianExponentB
             -3.0*pow(gaussianExponentA,2.0)*gaussianExponentB*pow(dxyz[axis]/beta,2.0)
             +gaussianExponentA*pow(gaussianExponentB,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentA,3.0)*pow(gaussianExponentB*dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))
             *pow(beta,-2.0);
      value *= 8.0*gaussianExponentA*gaussianExponentB*dxyz[anotherAxis]
              /(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzB[axis]*overlap1;
   }
   else if( (valenceOrbitalA == dyz && axis == ZAxis && valenceOrbitalB == dzz) ||
            (valenceOrbitalA == dzx && axis == ZAxis && valenceOrbitalB == dzz) ){
      CartesianType anotherAxis;
      if(valenceOrbitalA == dyz){
         anotherAxis = YAxis;
      }
      else if(valenceOrbitalA == dzx){
         anotherAxis = XAxis;
      }
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentB-0.5*gaussianExponentA
             -3.0*pow(gaussianExponentB,2.0)*gaussianExponentA*pow(dxyz[axis]/beta,2.0)
             +gaussianExponentB*pow(gaussianExponentA,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/(2.0*beta)
             +pow(gaussianExponentB,3.0)*pow(gaussianExponentA*dxyz[axis],2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))
             *pow(beta,-2.0);
      value *= -8.0*gaussianExponentA*gaussianExponentB*dxyz[anotherAxis]
              /(sqrt(3.0)*pow(beta,3.0));
      value *= sasb;
      value += xyzA[axis]*overlap1;
   }
   else if(valenceOrbitalA == dzz && axis == XAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentB - gaussianExponentA
             +gaussianExponentA*pow(gaussianExponentB,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta;
      value *= 4.0*gaussianExponentA*gaussianExponentB*dxyz[XAxis]*pow(beta,-3.0)/sqrt(3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == XAxis && valenceOrbitalB == dzz){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA - gaussianExponentB
             +gaussianExponentB*pow(gaussianExponentA,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta;
      value *= -4.0*gaussianExponentA*gaussianExponentB*dxyz[XAxis]*pow(beta,-3.0)/sqrt(3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dzz && axis == YAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentB - gaussianExponentA
             +gaussianExponentA*pow(gaussianExponentB,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentA,2.0)*gaussianExponentB
             *(pow(dxyz[YAxis],2.0) - pow(dxyz[XAxis],2.0))/beta;
      value *= -4.0*gaussianExponentA*gaussianExponentB*dxyz[YAxis]*pow(beta,-3.0)/sqrt(3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == YAxis && valenceOrbitalB == dzz){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = gaussianExponentA - gaussianExponentB
             +gaussianExponentB*pow(gaussianExponentA,2.0)
             *(2.0*pow(dxyz[ZAxis],2.0) - pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0))/beta
             +pow(gaussianExponentB,2.0)*gaussianExponentA
             *(pow(dxyz[YAxis],2.0) - pow(dxyz[XAxis],2.0))/beta;
      value *= 4.0*gaussianExponentA*gaussianExponentB*dxyz[YAxis]*pow(beta,-3.0)/sqrt(3.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dzz && axis == ZAxis && valenceOrbitalB == dxxyy){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0);
      value *= -8.0*pow(gaussianExponentA,3.0)*pow(gaussianExponentB,2.0)*dxyz[ZAxis];
      value /= sqrt(3.0)*pow(beta,4.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else if(valenceOrbitalA == dxxyy && axis == ZAxis && valenceOrbitalB == dzz){
      double axisAverage = (gaussianExponentA*xyzA[axis]+gaussianExponentB*xyzB[axis])/beta;
      double overlap1 = this->GetGaussianOverlap(atomTypeA,
                                                 valenceOrbitalA, 
                                                 gaussianExponentA, 
                                                 atomTypeB, 
                                                 valenceOrbitalB, 
                                                 gaussianExponentB,
                                                 dxyz[XAxis], 
                                                 dxyz[YAxis], 
                                                 dxyz[ZAxis], 
                                                 Rab);
      double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA, gaussianExponentB, Rab);
      value = pow(dxyz[XAxis],2.0) - pow(dxyz[YAxis],2.0);
      value *= 8.0*pow(gaussianExponentA,2.0)*pow(gaussianExponentB,3.0)*dxyz[ZAxis];
      value /= sqrt(3.0)*pow(beta,4.0);
      value *= sasb;
      value += axisAverage*overlap1;
   }
   else{
      stringstream ss;
      ss << this->errorMessageGetGaussianCartesianMatrixBadOrbital;
      ss << this->errorMessageAtomA;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeA) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalA) << endl;
      ss << this->errorMessageAtomB;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeB) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalB) << endl;
      ss << this->errorMessageCartesianType << CartesianTypeStr(axis) << endl;
      throw MolDSException(ss.str());
   }

   return value;
}

void Cndo2::FreeDiatomicOverlapAndRotatingMatrix(double*** diatomicOverlap, 
                                                 double*** rotatingMatrix) const{
   // free
   MallocerFreer::GetInstance()->Free<double>(diatomicOverlap, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(rotatingMatrix,  OrbitalType_end, OrbitalType_end);
}

// calculate Overlap matrix. E.g. S_{\mu\nu} in (3.74) in J. A. Pople book.
void Cndo2::CalcOverlap(double** overlap, const Molecule& molecule) const{
   int totalAONumber = molecule.GetTotalNumberAOs();
   int totalAtomNumber = molecule.GetNumberAtoms();

   stringstream ompErrors;
#pragma omp parallel 
   {
      double** diatomicOverlap = NULL;
      double** rotatingMatrix = NULL;
      try{
         // malloc
         MallocerFreer::GetInstance()->Malloc<double>(&diatomicOverlap,
                                                      OrbitalType_end, 
                                                      OrbitalType_end);
         MallocerFreer::GetInstance()->Malloc<double>(&rotatingMatrix,
                                                      OrbitalType_end, 
                                                      OrbitalType_end);
         // calculation overlap matrix
         for(int mu=0; mu<totalAONumber; mu++){
            overlap[mu][mu] = 1.0;
         }

#pragma omp for schedule(auto)
         for(int A=0; A<totalAtomNumber; A++){
            const Atom& atomA = *molecule.GetAtom(A);
            for(int B=A+1; B<totalAtomNumber; B++){
               const Atom& atomB = *molecule.GetAtom(B);
               this->CalcDiatomicOverlapInDiatomicFrame(diatomicOverlap, atomA, atomB);
               this->CalcRotatingMatrix(rotatingMatrix, atomA, atomB);
               this->RotateDiatmicOverlapToSpaceFrame(diatomicOverlap, rotatingMatrix);
               this->SetOverlapElement(overlap, diatomicOverlap, atomA, atomB);
            }
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
      this->FreeDiatomicOverlapAndRotatingMatrix(&diatomicOverlap, &rotatingMatrix);
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
   /* 
   this->OutputLog("overlap matrix\n"); 
   for(int o=0; o<molecule.GetTotalNumberAOs(); o++){
      for(int p=0; p<molecule.GetTotalNumberAOs(); p++){
         this->OutputLog(boost::format("%lf\t") % overlap[o][p]);
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n");
   */
}

// First derivative of diatomic overlap integrals between AOs in space fixed flame.
// The Overlap matrix is S_{\mu\nu} in (3.74) in J. A. Pople book.
// Note that this method can not treat d-obitals 
// because CalcRotatingMatrix1stDerivatives can not treat d-orbitals.
void Cndo2::CalcDiatomicOverlap1stDerivatives(double*** diatomicOverlap1stDerivs, 
                                              const Atom& atomA, 
                                              const Atom& atomB) const{
   double cartesian[CartesianType_end] = {atomA.GetXyz()[XAxis] - atomB.GetXyz()[XAxis], 
                                          atomA.GetXyz()[YAxis] - atomB.GetXyz()[YAxis],
                                          atomA.GetXyz()[ZAxis] - atomB.GetXyz()[ZAxis]};
   double R = sqrt( pow(cartesian[XAxis],2.0) + 
                    pow(cartesian[YAxis],2.0) + 
                    pow(cartesian[ZAxis],2.0) );
   
   double** diaOverlapInDiaFrame = NULL;  // diatomic overlap in diatomic frame
   double** diaOverlap1stDerivInDiaFrame = NULL; // first derivative of the diaOverlap. This derivative is related to the distance between two atoms.
   double**  rotMat = NULL; // rotating Matrix from the diatomic frame to space fixed frame.
   double*** rotMat1stDerivs = NULL; // first derivatives of the rotMat.

   try{
      this->MallocDiatomicOverlap1stDeriTemps(&diaOverlapInDiaFrame,
                                              &diaOverlap1stDerivInDiaFrame,
                                              &rotMat,
                                              &rotMat1stDerivs);
      this->CalcDiatomicOverlapInDiatomicFrame(diaOverlapInDiaFrame, atomA, atomB);
      this->CalcDiatomicOverlap1stDerivativeInDiatomicFrame(diaOverlap1stDerivInDiaFrame, atomA, atomB);
      this->CalcRotatingMatrix(rotMat, atomA, atomB);
      this->CalcRotatingMatrix1stDerivatives(rotMat1stDerivs, atomA, atomB);

      // rotate
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            for(int c=0; c<CartesianType_end; c++){
               diatomicOverlap1stDerivs[i][j][c] = 0.0;

               double temp1 = 0.0;
               double temp2 = 0.0;
               double temp3 = 0.0;
               for(int k=0; k<OrbitalType_end; k++){
                  for(int l=0; l<OrbitalType_end; l++){
                     temp1 += rotMat[i][k] 
                             *rotMat[j][l]
                             *(cartesian[c]/R)
                             *diaOverlap1stDerivInDiaFrame[k][l];
                     temp2 += rotMat1stDerivs[i][k][c] 
                             *rotMat[j][l]
                             *diaOverlapInDiaFrame[k][l];
                     temp3 += rotMat[i][k] 
                             *rotMat1stDerivs[j][l][c]
                             *diaOverlapInDiaFrame[k][l];
                  }
               }
               diatomicOverlap1stDerivs[i][j][c] = temp1 + temp2 + temp3;
            }
         }
      }
   }
   catch(MolDSException ex){
      this->FreeDiatomicOverlap1stDeriTemps(&diaOverlapInDiaFrame,
                                            &diaOverlap1stDerivInDiaFrame,
                                            &rotMat,
                                            &rotMat1stDerivs);
      throw ex;
   }
   // free
   this->FreeDiatomicOverlap1stDeriTemps(&diaOverlapInDiaFrame,
                                         &diaOverlap1stDerivInDiaFrame,
                                         &rotMat,
                                         &rotMat1stDerivs);
}

void Cndo2::CalcDiatomicOverlap1stDerivatives(double*** diatomicOverlap1stDerivs, 
                                              int indexAtomA, 
                                              int indexAtomB) const{
   this->CalcDiatomicOverlap1stDerivatives(diatomicOverlap1stDerivs,
                                           *this->molecule->GetAtom(indexAtomA),
                                           *this->molecule->GetAtom(indexAtomB));
}

// Second derivative of diatomic overlap integrals between AOs in space fixed flame.
// The Overlap matrix is S_{\mu\nu} in (3.74) in J. A. Pople book.
// Note that this method can not treat d-obitals 
// because CalcRotatingMatrix1stDerivatives can not treat d-orbitals.
void Cndo2::CalcDiatomicOverlap2ndDerivatives(double**** diatomicOverlap2ndDerivs, 
                                              const Atom& atomA, 
                                              const Atom& atomB) const{
   double cartesian[CartesianType_end] = {atomA.GetXyz()[XAxis] - atomB.GetXyz()[XAxis], 
                                          atomA.GetXyz()[YAxis] - atomB.GetXyz()[YAxis],
                                          atomA.GetXyz()[ZAxis] - atomB.GetXyz()[ZAxis]};
   double R = sqrt( pow(cartesian[XAxis],2.0) + 
                    pow(cartesian[YAxis],2.0) + 
                    pow(cartesian[ZAxis],2.0) );
   
   double** diaOverlapInDiaFrame = NULL;  // diatomic overlap in diatomic frame
   double** diaOverlap1stDerivInDiaFrame = NULL; // first derivative of the diaOverlap. This derivative is related to the distance between two atoms.
   double** diaOverlap2ndDerivInDiaFrame = NULL; // second derivative of the diaOverlap. This derivative is related to the distance between two atoms.
   double**   rotMat = NULL; // rotating Matrix from the diatomic frame to space fixed frame.
   double***  rotMat1stDerivatives = NULL; //first derivatives of the rotMat
   double**** rotMat2ndDerivatives = NULL; //second derivatives of the rotMat
   double***  tempDiaOverlap1stDerivs = NULL; // first derivatives of the diaOverlap. This derivatives are related to the all Cartesian coordinates.
   double**** tempDiaOverlap2ndDerivs = NULL; //sedond derivatives of the diaOverlap. This derivatives are related to the all Cartesian coordinates.

   try{
      this->MallocDiatomicOverlap2ndDeriTemps(&diaOverlapInDiaFrame,
                                              &diaOverlap1stDerivInDiaFrame,
                                              &diaOverlap2ndDerivInDiaFrame,
                                              &rotMat,
                                              &rotMat1stDerivatives,
                                              &rotMat2ndDerivatives,
                                              &tempDiaOverlap1stDerivs,
                                              &tempDiaOverlap2ndDerivs);
      this->CalcDiatomicOverlapInDiatomicFrame(diaOverlapInDiaFrame, atomA, atomB);
      this->CalcDiatomicOverlap1stDerivativeInDiatomicFrame(diaOverlap1stDerivInDiaFrame, atomA, atomB);
      this->CalcDiatomicOverlap2ndDerivativeInDiatomicFrame(diaOverlap2ndDerivInDiaFrame, atomA, atomB);
      this->CalcRotatingMatrix(rotMat, atomA, atomB);
      this->CalcRotatingMatrix1stDerivatives(rotMat1stDerivatives, atomA, atomB);
      this->CalcRotatingMatrix2ndDerivatives(rotMat2ndDerivatives, atomA, atomB);

      // calculate each element of first derivatives
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            for(int dimA1=0; dimA1<CartesianType_end; dimA1++){
               tempDiaOverlap1stDerivs[i][j][dimA1] = (cartesian[dimA1]/R)*diaOverlap1stDerivInDiaFrame[i][j];
            }
         }
      }

      // calculate each element of second derivatives
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            for(int dimA1=XAxis; dimA1<CartesianType_end; dimA1++){
               for(int dimA2=XAxis; dimA2<CartesianType_end; dimA2++){
                  tempDiaOverlap2ndDerivs[i][j][dimA1][dimA2] 
                     = this->Get2ndDerivativeElementFromDistanceDerivatives(diaOverlap1stDerivInDiaFrame[i][j],
                                                                            diaOverlap2ndDerivInDiaFrame[i][j],
                                                                            static_cast<CartesianType>(dimA1),
                                                                            static_cast<CartesianType>(dimA2),
                                                                            cartesian,
                                                                            R);
               }
            }
         }
      }

      // rotate
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            for(int dimA1=XAxis; dimA1<CartesianType_end; dimA1++){
               for(int dimA2=XAxis; dimA2<CartesianType_end; dimA2++){
                  diatomicOverlap2ndDerivs[i][j][dimA1][dimA2] = 0.0;
               
                  double temp1 = 0.0, temp2=0.0, temp3 = 0.0;
                  double temp4 = 0.0, temp5=0.0, temp6 = 0.0;
                  double temp7 = 0.0, temp8=0.0, temp9 = 0.0;
                  for(int k=0; k<OrbitalType_end; k++){
                     for(int l=0; l<OrbitalType_end; l++){
              
                        temp1 += rotMat2ndDerivatives   [i][k][dimA1][dimA2]
                                *rotMat                 [j][l]
                                *diaOverlapInDiaFrame   [k][l];
                        temp2 += rotMat                 [i][k]
                                *rotMat2ndDerivatives   [j][l][dimA1][dimA2]
                                *diaOverlapInDiaFrame   [k][l];
                        temp3 += rotMat                 [i][k]
                                *rotMat                 [j][l]
                                *tempDiaOverlap2ndDerivs[k][l][dimA1][dimA2];
                        temp4 += rotMat1stDerivatives   [i][k][dimA1] 
                                *rotMat1stDerivatives   [j][l][dimA2]
                                *diaOverlapInDiaFrame   [k][l];
                        temp5 += rotMat1stDerivatives   [i][k][dimA1] 
                                *rotMat                 [j][l]
                                *tempDiaOverlap1stDerivs[k][l][dimA2];
                        temp6 += rotMat1stDerivatives   [i][k][dimA2] 
                                *rotMat1stDerivatives   [j][l][dimA1]
                                *diaOverlapInDiaFrame   [k][l];
                        temp7 += rotMat                 [i][k] 
                                *rotMat1stDerivatives   [j][l][dimA1]
                                *tempDiaOverlap1stDerivs[k][l][dimA2];
                        temp8 += rotMat1stDerivatives   [i][k][dimA2] 
                                *rotMat                 [j][l]
                                *tempDiaOverlap1stDerivs[k][l][dimA1];
                        temp9 += rotMat                 [i][k] 
                                *rotMat1stDerivatives   [j][l][dimA2]
                                *tempDiaOverlap1stDerivs[k][l][dimA1];
                     }
                  }

                  diatomicOverlap2ndDerivs[i][j][dimA1][dimA2] = temp1+temp2+temp3 
                                                                +temp4+temp5+temp6 
                                                                +temp7+temp8+temp9;
               }
            }
         }
      }
      
   }
   catch(MolDSException ex){
      this->FreeDiatomicOverlap2ndDeriTemps(&diaOverlapInDiaFrame,
                                            &diaOverlap1stDerivInDiaFrame,
                                            &diaOverlap2ndDerivInDiaFrame,
                                            &rotMat,
                                            &rotMat1stDerivatives,
                                            &rotMat2ndDerivatives,
                                            &tempDiaOverlap1stDerivs,
                                            &tempDiaOverlap2ndDerivs);
      throw ex;
   }
   // free
   this->FreeDiatomicOverlap2ndDeriTemps(&diaOverlapInDiaFrame,
                                         &diaOverlap1stDerivInDiaFrame,
                                         &diaOverlap2ndDerivInDiaFrame,
                                         &rotMat,
                                         &rotMat1stDerivatives,
                                         &rotMat2ndDerivatives,
                                         &tempDiaOverlap1stDerivs,
                                         &tempDiaOverlap2ndDerivs);
   /*
   for(int i=0; i<OrbitalType_end; i++){
      for(int j=0; j<OrbitalType_end; j++){
         for(int dimA1=0; dimA1<CartesianType_end; dimA1++){
            for(int dimA2=0; dimA2<CartesianType_end; dimA2++){
               printf("i=%d j=%d dimA1=%d dimA2=%d: %e\n",i,j,dimA1,dimA2,overlap2ndDeri[i][j][dimA1][dimA2]);
            }
         }
      }
   }
   */
}

void Cndo2::CalcDiatomicOverlap2ndDerivatives(double**** diatomicOverlap2ndDerivs, 
                                              int indexAtomA, 
                                              int indexAtomB) const{
   this->CalcDiatomicOverlap2ndDerivatives(diatomicOverlap2ndDerivs,
                                           *this->molecule->GetAtom(indexAtomA),
                                           *this->molecule->GetAtom(indexAtomB));
}

double Cndo2::Get2ndDerivativeElementFromDistanceDerivatives(double firstDistanceDeri,
                                                             double secondDistanceDeri,
                                                             CartesianType axisA1,
                                                             CartesianType axisA2,
                                                             double* cartesian,
                                                             double Rab) const{
   double value=0.0;               
   if(axisA1 != axisA2){
      value = -1.0*pow(Rab, -3.0)*firstDistanceDeri;
      value += pow(Rab, -2.0)*secondDistanceDeri;
      value *= cartesian[axisA1]*cartesian[axisA2];
   }
   else{
      value = (pow(Rab,2.0) - pow(cartesian[axisA1],2.0))*pow(Rab, -3.0)*firstDistanceDeri;
      value += pow(cartesian[axisA1]/Rab, 2.0)*secondDistanceDeri;
   }
   return value;
}

void Cndo2::MallocDiatomicOverlap1stDeriTemps(double*** diaOverlapInDiaFrame, 
                                              double*** diaOverlap1stDerivInDiaFrame,
                                              double*** rotMat,
                                              double**** rotMat1stDerivs) const{
   MallocerFreer::GetInstance()->Malloc<double>(diaOverlapInDiaFrame,         OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(diaOverlap1stDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(rotMat,                       OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(rotMat1stDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end);
}

void Cndo2::MallocDiatomicOverlap2ndDeriTemps(double*** diaOverlapInDiaFrame, 
                                              double*** diaOverlap1stDerivInDiaFrame,
                                              double*** diaOverlap2ndDerivInDiaFrame,
                                              double***   rotMat,
                                              double****  rotMat1stDerivs,
                                              double***** rotMat2ndDerivs,
                                              double****  tempDiaOverlap1stDerivs,
                                              double***** tempDiaOverlap2ndDerivs) const{
   MallocerFreer::GetInstance()->Malloc<double>(diaOverlapInDiaFrame,         OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(diaOverlap1stDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(diaOverlap2ndDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(rotMat,                       OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Malloc<double>(rotMat1stDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Malloc<double>(rotMat2ndDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Malloc<double>(tempDiaOverlap1stDerivs,      OrbitalType_end, OrbitalType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Malloc<double>(tempDiaOverlap2ndDerivs,      OrbitalType_end, OrbitalType_end, CartesianType_end, CartesianType_end);
}

void Cndo2::FreeDiatomicOverlap1stDeriTemps(double*** diaOverlapInDiaFrame, 
                                            double*** diaOverlap1stDerivInDiaFrame,
                                            double*** rotMat,
                                            double**** rotMat1stDerivs) const{
   MallocerFreer::GetInstance()->Free<double>(diaOverlapInDiaFrame,         OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(diaOverlap1stDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(rotMat,                       OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(rotMat1stDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end);
}

void Cndo2::FreeDiatomicOverlap2ndDeriTemps(double*** diaOverlapInDiaFrame, 
                                            double*** diaOverlap1stDerivInDiaFrame,
                                            double*** diaOverlap2ndDerivInDiaFrame,
                                            double***   rotMat,
                                            double****  rotMat1stDerivs,
                                            double***** rotMat2ndDerivs,
                                            double****  tempDiaOverlap1stDerivs,
                                            double***** tempDiaOverlap2ndDerivs) const{
   MallocerFreer::GetInstance()->Free<double>(diaOverlapInDiaFrame,         OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(diaOverlap1stDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(diaOverlap2ndDerivInDiaFrame, OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(rotMat,                       OrbitalType_end, OrbitalType_end);
   MallocerFreer::GetInstance()->Free<double>(rotMat1stDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Free<double>(rotMat2ndDerivs,              OrbitalType_end, OrbitalType_end, CartesianType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Free<double>(tempDiaOverlap1stDerivs,      OrbitalType_end, OrbitalType_end, CartesianType_end);
   MallocerFreer::GetInstance()->Free<double>(tempDiaOverlap2ndDerivs,      OrbitalType_end, OrbitalType_end, CartesianType_end, CartesianType_end);
}

// calculate Overlap matrix. E.g. S_{\mu\nu} in (3.74) in J. A. Pople book by GTO expansion.
// See Eqs. (28) - (32) in [DY_1977]
void Cndo2::CalcOverlapByGTOExpansion(double** overlap, 
                                      const Molecule& molecule, 
                                      STOnGType stonG) const{
   int totalAONumber = molecule.GetTotalNumberAOs();
   int totalAtomNumber = molecule.GetNumberAtoms();

   // calculation overlap matrix
   for(int mu=0; mu<totalAONumber; mu++){
      overlap[mu][mu] = 1.0;
   }

   stringstream ompErrors;
#pragma omp parallel for schedule(auto) 
   for(int A=0; A<totalAtomNumber; A++){
      try{
         const Atom& atomA = *molecule.GetAtom(A);
         int firstAOIndexAtomA = atomA.GetFirstAOIndex();
         for(int B=A+1; B<totalAtomNumber; B++){
            const Atom& atomB = *molecule.GetAtom(B);
            int firstAOIndexAtomB = atomB.GetFirstAOIndex();
            for(int a=0; a<atomA.GetValenceSize(); a++){
               for(int b=0; b<atomB.GetValenceSize(); b++){
                  int mu = firstAOIndexAtomA + a;      
                  int nu = firstAOIndexAtomB + b;      
                  double value = this->GetOverlapElementByGTOExpansion(atomA, a, atomB, b, stonG);
                  overlap[mu][nu] = value;
                  overlap[nu][mu] = value;
               }
            }
         }
      }
      catch(MolDSException ex){
#pragma omp critical
         ompErrors << ex.what() << endl ;
      }
   }
   // Exception throwing for omp-region
   if(!ompErrors.str().empty()){
      throw MolDSException(ompErrors.str());
   }
   /* 
   this->OutputLog("overlap matrix by STOnG\n"); 
   for(int o=0; o<molecule.GetTotalNumberAOs(); o++){
      for(int p=0; p<molecule.GetTotalNumberAOs(); p++){
         this->OutputLog(boost::format("%lf\t") % overlap[o][p]);
      }
      this->OutputLog("\n");
   }
   this->OutputLog("\n");
   */   
}

// calculate elements of overlap matrix. 
// E.g. S_{\mu\nu} in (3.74) in J. A. Pople book by GTO expansion.
// See Eqs. (28) - (32) in [DY_1977]
double Cndo2::GetOverlapElementByGTOExpansion(const Atom& atomA, int valenceIndexA, 
                                              const Atom& atomB, int valenceIndexB,
                                              STOnGType stonG) const{
   double value = 0.0;
   double dx = atomA.GetXyz()[XAxis] - atomB.GetXyz()[XAxis];
   double dy = atomA.GetXyz()[YAxis] - atomB.GetXyz()[YAxis];
   double dz = atomA.GetXyz()[ZAxis] - atomB.GetXyz()[ZAxis];
   double Rab = sqrt( pow(dx, 2.0) + pow(dy, 2.0) + pow(dz,2.0) );
   ShellType shellTypeA = atomA.GetValenceShellType();
   ShellType shellTypeB = atomB.GetValenceShellType();
   OrbitalType valenceOrbitalA = atomA.GetValence(valenceIndexA);
   OrbitalType valenceOrbitalB = atomB.GetValence(valenceIndexB);
   double orbitalExponentA = atomA.GetOrbitalExponent(atomA.GetValenceShellType(), 
                                                      valenceOrbitalA, 
                                                      this->theory);
   double orbitalExponentB = atomB.GetOrbitalExponent(atomB.GetValenceShellType(), 
                                                      valenceOrbitalB, 
                                                      this->theory);
   double gaussianExponentA = 0.0;
   double gaussianExponentB = 0.0;

   double temp = 0.0;
   for(int i=0; i<=stonG; i++){
      for(int j=0; j<=stonG; j++){
         temp = GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                               shellTypeA, 
                                                               valenceOrbitalA, 
                                                               i); 
         temp *= GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                                shellTypeB, 
                                                                valenceOrbitalB, 
                                                                j); 
         gaussianExponentA = pow(orbitalExponentA, 2.0) *
                             GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeA, 
                                                                         valenceOrbitalA, 
                                                                         i);
         gaussianExponentB = pow(orbitalExponentB, 2.0) *
                             GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeB, 
                                                                         valenceOrbitalB, 
                                                                         j);
         temp *= this->GetGaussianOverlap(atomA.GetAtomType(), 
                                          valenceOrbitalA, 
                                          gaussianExponentA, 
                                          atomB.GetAtomType(), 
                                          valenceOrbitalB, 
                                          gaussianExponentB,
                                          dx, dy, dz, Rab);
         value += temp;
      }
   }
   return value;
}

// Calculate gaussian overlap integrals of Sa and Sb.
// That is, calculate (S_A|S_B). See Eq. (28) in [DY_1977].
double Cndo2::GetGaussianOverlapSaSb(double gaussianExponentA, 
                                     double gaussianExponentB,
                                     double Rab) const{
   double value;
   double temp1 = 0.0;
   double temp2 = 0.0;
   temp1 = 2.0*pow(gaussianExponentA*gaussianExponentB, 0.5)
            /(gaussianExponentA+gaussianExponentB);
   temp2 = -1.0* gaussianExponentA*gaussianExponentB
            /(gaussianExponentA+gaussianExponentB);
   value = pow(temp1, 1.5)*exp(temp2*pow(Rab, 2.0));
   return value;
}

// calculate gaussian overlap integrals. 
// See Eqs. (28) - (32) in [DY_1977].
// Although d-orbital is not calucluated in [DY_1977],
// the way to calculate overlap related to d-orbital is 
// same to the one written in [DY_1977].
double Cndo2::GetGaussianOverlap(AtomType atomTypeA, 
                                 OrbitalType valenceOrbitalA, 
                                 double gaussianExponentA, 
                                 AtomType atomTypeB, 
                                 OrbitalType valenceOrbitalB, 
                                 double gaussianExponentB,
                                 double dx, double dy, double dz, double Rab) const{

   double value = 0.0;
   if(valenceOrbitalA == s && valenceOrbitalB == s){
      value = 1.0;
   }

   else if(valenceOrbitalA == s && valenceOrbitalB == px){
      value = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)*dx;
      value /= (gaussianExponentA+gaussianExponentB);
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == py){
      value = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)*dy;
      value /= (gaussianExponentA+gaussianExponentB);
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == pz){
      value = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)*dz;
      value /= (gaussianExponentA+gaussianExponentB);
   }

   else if(valenceOrbitalA == px && valenceOrbitalB == s){
      value = -2.0*pow(gaussianExponentA, 0.5)*gaussianExponentB*dx;
      value /= (gaussianExponentA+gaussianExponentB);
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == s){
      value = -2.0*pow(gaussianExponentA, 0.5)*gaussianExponentB*dy;
      value /= (gaussianExponentA+gaussianExponentB);
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == s){
      value = -2.0*pow(gaussianExponentA, 0.5)*gaussianExponentB*dz;
      value /= (gaussianExponentA+gaussianExponentB);
   }

   else if(valenceOrbitalA == px && valenceOrbitalB == px){
      double temp = 0.0;
      temp = -1.0*pow(dx,2.0)*gaussianExponentA*gaussianExponentB;
      temp /= (gaussianExponentA+gaussianExponentB);
      temp += 0.5;
      value = 4.0*pow(gaussianExponentA*gaussianExponentB, 0.5);
      value /= (gaussianExponentA+gaussianExponentB);
      value *= temp;
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == py){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dx*dy;
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == pz){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dx*dz;
   }

   else if(valenceOrbitalA == py && valenceOrbitalB == px){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dy*dx;
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == py){
      double temp = 0.0;
      temp = -1.0*pow(dy,2.0)*gaussianExponentA*gaussianExponentB;
      temp /= (gaussianExponentA+gaussianExponentB);
      temp += 0.5;
      value = 4.0*pow(gaussianExponentA*gaussianExponentB, 0.5);
      value /= (gaussianExponentA+gaussianExponentB);
      value *= temp;
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == pz){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dy*dz;
   }

   else if(valenceOrbitalA == pz && valenceOrbitalB == px){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dz*dx;
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == py){
      value = -4.0*pow(gaussianExponentA*gaussianExponentB, 1.5);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= dz*dy;
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == pz){
      double temp = 0.0;
      temp = -1.0*pow(dz,2.0)*gaussianExponentA*gaussianExponentB;
      temp /= (gaussianExponentA+gaussianExponentB);
      temp += 0.5;
      value = 4.0*pow(gaussianExponentA*gaussianExponentB, 0.5);
      value /= (gaussianExponentA+gaussianExponentB);
      value *= temp;
   }

   else if(valenceOrbitalA == dxy && valenceOrbitalB == s){
      value = 4.0*gaussianExponentA;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentB*dx;
      value *= gaussianExponentB*dy;
   }
   else if(valenceOrbitalA == dyz && valenceOrbitalB == s){
      value = 4.0*gaussianExponentA;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentB*dy;
      value *= gaussianExponentB*dz;
   }
   else if(valenceOrbitalA == dzx && valenceOrbitalB == s){
      value = 4.0*gaussianExponentA;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentB*dz;
      value *= gaussianExponentB*dx;
   }

   else if(valenceOrbitalA == s && valenceOrbitalB == dxy){
      value = 4.0*gaussianExponentB;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentA*dx;
      value *= gaussianExponentA*dy;
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == dyz){
      value = 4.0*gaussianExponentB;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentA*dy;
      value *= gaussianExponentA*dz;
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == dzx){
      value = 4.0*gaussianExponentB;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= gaussianExponentA*dz;
      value *= gaussianExponentA*dx;
   }

   else if(valenceOrbitalA == dxy && valenceOrbitalB == px){
      double temp1 = -0.5*gaussianExponentB*dy;
      double temp2 = gaussianExponentB*dx*gaussianExponentA*dx*gaussianExponentB*dy;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dxy && valenceOrbitalB == py){
      double temp1 = -0.5*gaussianExponentB*dx;
      double temp2 = gaussianExponentB*dy*gaussianExponentA*dy*gaussianExponentB*dx;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dyz && valenceOrbitalB == py){
      double temp1 = -0.5*gaussianExponentB*dz;
      double temp2 = gaussianExponentB*dy*gaussianExponentA*dy*gaussianExponentB*dz;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dyz && valenceOrbitalB == pz){
      double temp1 = -0.5*gaussianExponentB*dy;
      double temp2 = gaussianExponentB*dz*gaussianExponentA*dz*gaussianExponentB*dy;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dzx && valenceOrbitalB == pz){
      double temp1 = -0.5*gaussianExponentB*dx;
      double temp2 = gaussianExponentB*dz*gaussianExponentA*dz*gaussianExponentB*dx;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dzx && valenceOrbitalB == px){
      double temp1 = -0.5*gaussianExponentB*dz;
      double temp2 = gaussianExponentB*dx*gaussianExponentA*dx*gaussianExponentB*dz;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }

   else if(valenceOrbitalA == px && valenceOrbitalB == dxy){
      double temp1 = 0.5*gaussianExponentA*dy;
      double temp2 = -1.0*gaussianExponentA*dx*gaussianExponentB*dx*gaussianExponentA*dy;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == dxy){
      double temp1 = 0.5*gaussianExponentA*dx;
      double temp2 = -1.0*gaussianExponentA*dy*gaussianExponentB*dy*gaussianExponentA*dx;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == dyz){
      double temp1 = 0.5*gaussianExponentA*dz;
      double temp2 = -1.0*gaussianExponentA*dy*gaussianExponentB*dy*gaussianExponentA*dz;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == dyz){
      double temp1 = 0.5*gaussianExponentA*dy;
      double temp2 = -1.0*gaussianExponentA*dz*gaussianExponentB*dz*gaussianExponentA*dy;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == dzx){
      double temp1 = 0.5*gaussianExponentA*dx;
      double temp2 = -1.0*gaussianExponentA*dz*gaussianExponentB*dz*gaussianExponentA*dx;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == dzx){
      double temp1 = 0.5*gaussianExponentA*dz;
      double temp2 = -1.0*gaussianExponentA*dx*gaussianExponentB*dx*gaussianExponentA*dz;
      temp2 /= gaussianExponentA+gaussianExponentB;
      value = temp1 + temp2;
      value *= 8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }

   else if(valenceOrbitalA == dxy && valenceOrbitalB == pz){
      value = 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentB*dx*gaussianExponentB*dy*gaussianExponentA*dz;
   }
   else if(valenceOrbitalA == dyz && valenceOrbitalB == px){
      value = 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentB*dy*gaussianExponentB*dz*gaussianExponentA*dx;
   }
   else if(valenceOrbitalA == dzx && valenceOrbitalB == py){
      value = 8.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentB*dz*gaussianExponentB*dx*gaussianExponentA*dy;
   }

   else if(valenceOrbitalA == pz && valenceOrbitalB == dxy){
      value = -8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentA*dx*gaussianExponentA*dy*gaussianExponentB*dz;
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == dyz){
      value = -8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentA*dy*gaussianExponentA*dz*gaussianExponentB*dx;
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == dzx){
      value = -8.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
      value *= gaussianExponentA*dz*gaussianExponentA*dx*gaussianExponentB*dy;
   }

   else if(valenceOrbitalA == dxxyy && valenceOrbitalB == s){
      value = 2.0*gaussianExponentA;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= pow(gaussianExponentB*dx, 2.0) - pow(gaussianExponentB*dy, 2.0);
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == dxxyy){
      value = 2.0*gaussianExponentB;
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= pow(gaussianExponentA*dx, 2.0) - pow(gaussianExponentA*dy, 2.0);
   }
   else if(valenceOrbitalA == dxxyy && valenceOrbitalB == px){
      value = gaussianExponentB*dx;
      value -= pow(gaussianExponentB*dx, 2.0)*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      value += pow(gaussianExponentB*dy, 2.0)*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      value *= -4.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == dxxyy){
      value = gaussianExponentA*dx;
      value -= pow(gaussianExponentA*dx, 2.0)*gaussianExponentB*dx/(gaussianExponentA+gaussianExponentB);
      value += pow(gaussianExponentA*dy, 2.0)*gaussianExponentB*dx/(gaussianExponentA+gaussianExponentB);
      value *= 4.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dxxyy && valenceOrbitalB == py){
      value = gaussianExponentB*dy;
      value += pow(gaussianExponentB*dx, 2.0)*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      value -= pow(gaussianExponentB*dy, 2.0)*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == dxxyy){
      value = gaussianExponentA*dy;
      value += pow(gaussianExponentA*dx, 2.0)*gaussianExponentB*dy/(gaussianExponentA+gaussianExponentB);
      value -= pow(gaussianExponentA*dy, 2.0)*gaussianExponentB*dy/(gaussianExponentA+gaussianExponentB);
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 2.0);
   }
   else if(valenceOrbitalA == dxxyy && valenceOrbitalB == pz){
      value = pow(gaussianExponentB*dx, 2.0) - pow(gaussianExponentB*dy, 2.0);
      value *= gaussianExponentA*dz;
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == dxxyy){
      value = pow(gaussianExponentA*dx, 2.0) - pow(gaussianExponentA*dy, 2.0);
      value *= gaussianExponentB*dz;
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA, 0.5);
      value /= pow(gaussianExponentA+gaussianExponentB, 3.0);
   }

   else if(valenceOrbitalA == dzz && valenceOrbitalB == s){
      double temp = 0.0;
      temp = 2.0*pow(gaussianExponentB*dz, 2.0) 
            -    pow(gaussianExponentB*dx, 2.0) 
            -    pow(gaussianExponentB*dy, 2.0);
      value = 2.0*gaussianExponentA/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= temp;
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == dzz){
      double temp = 0.0;
      temp = 2.0*pow(gaussianExponentA*dz, 2.0) 
            -    pow(gaussianExponentA*dx, 2.0) 
            -    pow(gaussianExponentA*dy, 2.0);
      value = 2.0*gaussianExponentB/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
      value *= temp;
   }
   else if(valenceOrbitalA == dzz && valenceOrbitalB == px){
      double temp = 0.0;
      temp = gaussianExponentB*dx;
      temp += 2.0*pow(gaussianExponentB*dz, 2.0)*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dx, 2.0)*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dy, 2.0)*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == dzz){
      double temp = 0.0;
      temp = gaussianExponentA*dx;
      temp += 2.0*pow(gaussianExponentA*dz, 2.0)*gaussianExponentB*dx/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dx, 2.0)*gaussianExponentB*dx/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dy, 2.0)*gaussianExponentB*dx/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dzz && valenceOrbitalB == py){
      double temp = 0.0;
      temp = gaussianExponentB*dy;
      temp += 2.0*pow(gaussianExponentB*dz, 2.0)*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dx, 2.0)*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dy, 2.0)*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == dzz){
      double temp = 0.0;
      temp = gaussianExponentA*dy;
      temp += 2.0*pow(gaussianExponentA*dz, 2.0)*gaussianExponentB*dy/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dx, 2.0)*gaussianExponentB*dy/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dy, 2.0)*gaussianExponentB*dy/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dzz && valenceOrbitalB == pz){
      double temp = 0.0;
      temp = -2.0*gaussianExponentB*dz;
      temp += 2.0*pow(gaussianExponentB*dz, 2.0)*gaussianExponentA*dz/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dx, 2.0)*gaussianExponentA*dz/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentB*dy, 2.0)*gaussianExponentA*dz/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= 4.0*gaussianExponentA*pow(gaussianExponentB, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == dzz){
      double temp = 0.0;
      temp = -2.0*gaussianExponentA*dz;
      temp += 2.0*pow(gaussianExponentA*dz, 2.0)*gaussianExponentB*dz/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dx, 2.0)*gaussianExponentB*dz/(gaussianExponentA+gaussianExponentB);
      temp -=     pow(gaussianExponentA*dy, 2.0)*gaussianExponentB*dz/(gaussianExponentA+gaussianExponentB);
      value = temp;
      value *= -4.0*gaussianExponentB*pow(gaussianExponentA, 0.5)/sqrt(3.0);
      value *= pow(gaussianExponentA+gaussianExponentB, -2.0);
   }

   else if(valenceOrbitalA == dxy && valenceOrbitalB == dxy){
      double temp = 0.25;
      temp -= 0.5*gaussianExponentB*dx*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      temp -= 0.5*gaussianExponentB*dy*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      temp += gaussianExponentB*dx*gaussianExponentA*dx
             *gaussianExponentB*dy*gaussianExponentA*dy
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
      value = 16.0*temp*gaussianExponentA*gaussianExponentB
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dyz && valenceOrbitalB == dyz){
      double temp = 0.25;
      temp -= 0.5*gaussianExponentB*dy*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      temp -= 0.5*gaussianExponentB*dz*gaussianExponentA*dz/(gaussianExponentA+gaussianExponentB);
      temp += gaussianExponentB*dy*gaussianExponentA*dy
             *gaussianExponentB*dz*gaussianExponentA*dz
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
      value = 16.0*temp*gaussianExponentA*gaussianExponentB
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dzx && valenceOrbitalB == dzx){
      double temp = 0.25;
      temp -= 0.5*gaussianExponentB*dz*gaussianExponentA*dz/(gaussianExponentA+gaussianExponentB);
      temp -= 0.5*gaussianExponentB*dx*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      temp += gaussianExponentB*dz*gaussianExponentA*dz
             *gaussianExponentB*dx*gaussianExponentA*dx
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
      value = 16.0*temp*gaussianExponentA*gaussianExponentB
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dxxyy && valenceOrbitalB == dxxyy){
      double temp1 = 1.0;
      temp1 -= 2.0*gaussianExponentB*dx*gaussianExponentA*dx/(gaussianExponentA+gaussianExponentB);
      temp1 -= 2.0*gaussianExponentB*dy*gaussianExponentA*dy/(gaussianExponentA+gaussianExponentB);
      double temp2 = gaussianExponentA*gaussianExponentB*(pow(dx,2.0)-pow(dy,2.0))
             /(gaussianExponentA+gaussianExponentB);
      temp1 += pow(temp2,2.0);
      value = 4.0*temp1*gaussianExponentA*gaussianExponentB
             *pow(gaussianExponentA+gaussianExponentB, -2.0);
   }
   else if(valenceOrbitalA == dzz && valenceOrbitalB == dzz){
      double temp = 3.0;
      temp -= gaussianExponentA*gaussianExponentB
             /(gaussianExponentA+gaussianExponentB)
             *(8.0*pow(dz,2.0)+2.0*pow(dx,2.0)+2.0*pow(dy,2.0));
      temp += pow(gaussianExponentA*gaussianExponentB,2.0)
             *pow(gaussianExponentA+gaussianExponentB, -2.0)
             *(4.0*pow(dz,4.0)
                  +pow(dx,4.0)
                  +pow(dy,4.0)
              -4.0*pow(dx*dz,2.0)
              -4.0*pow(dy*dz,2.0)
              +2.0*pow(dx*dy,2.0));
      value = 4.0*temp*gaussianExponentA*gaussianExponentB
             *pow(gaussianExponentA+gaussianExponentB, -2.0)
             /3.0;
   }

   else if((valenceOrbitalA == dxy && valenceOrbitalB == dyz) ||
           (valenceOrbitalA == dyz && valenceOrbitalB == dxy)){
      double temp = 0.5;
      temp -= gaussianExponentA*gaussianExponentB*pow(dy,2.0)/(gaussianExponentA+gaussianExponentB);
      value = -16.0*pow(gaussianExponentA*gaussianExponentB,2.0)
             *pow(gaussianExponentA+gaussianExponentB,-3.0)
             *dx*dz*temp;
   }
   else if((valenceOrbitalA == dyz && valenceOrbitalB == dzx) ||
           (valenceOrbitalA == dzx && valenceOrbitalB == dyz)){
      double temp = 0.5;
      temp -= gaussianExponentA*gaussianExponentB*pow(dz,2.0)/(gaussianExponentA+gaussianExponentB);
      value = -16.0*pow(gaussianExponentA*gaussianExponentB,2.0)
             *pow(gaussianExponentA+gaussianExponentB,-3.0)
             *dy*dx*temp;
   }
   else if((valenceOrbitalA == dzx && valenceOrbitalB == dxy) ||
           (valenceOrbitalA == dxy && valenceOrbitalB == dzx)){
      double temp = 0.5;
      temp -= gaussianExponentA*gaussianExponentB*pow(dx,2.0)/(gaussianExponentA+gaussianExponentB);
      value = -16.0*pow(gaussianExponentA*gaussianExponentB,2.0)
             *pow(gaussianExponentA+gaussianExponentB,-3.0)
             *dz*dy*temp;
   }

   else if((valenceOrbitalA == dxxyy && valenceOrbitalB == dxy) ||
           (valenceOrbitalA == dxy && valenceOrbitalB == dxxyy)){
      double temp = 2.0*gaussianExponentA*gaussianExponentB;
      value = pow(temp,3.0)*(dy*pow(dx,3.0)-dx*pow(dy,3.0))
             *pow(gaussianExponentA+gaussianExponentB,-4.0);
   }
   else if((valenceOrbitalA == dxxyy && valenceOrbitalB == dyz) ||
           (valenceOrbitalA == dyz && valenceOrbitalB == dxxyy)){
      double temp = 2.0*gaussianExponentA*gaussianExponentB;
      value = pow(temp,3.0)*(dy*dz*(gaussianExponentA+gaussianExponentB)
                            /(gaussianExponentA*gaussianExponentB)
                             +(pow(dx,2.0)*dy*dz - pow(dy,3.0)*dz))
             *pow(gaussianExponentA+gaussianExponentB,-4.0);
   }
   else if((valenceOrbitalA == dxxyy && valenceOrbitalB == dzx) ||
           (valenceOrbitalA == dzx && valenceOrbitalB == dxxyy)){
      double temp = 2.0*gaussianExponentA*gaussianExponentB;
      value = -1.0*pow(temp,3.0)*(dx*dz*(gaussianExponentA+gaussianExponentB)
                            /(gaussianExponentA*gaussianExponentB)
                             +(pow(dy,2.0)*dx*dz - pow(dx,3.0)*dz))
             *pow(gaussianExponentA+gaussianExponentB,-4.0);
   }

   else if((valenceOrbitalA == dzz && valenceOrbitalB == dxy) ||
           (valenceOrbitalA == dxy && valenceOrbitalB == dzz)){
      double temp = 2.0*dx*dy*pow(dz,2.0) - pow(dx,3.0)*dy - dx*pow(dy,3.0);
      temp *= gaussianExponentA*gaussianExponentB/(gaussianExponentA+gaussianExponentB);
      temp += 2.0*dx*dy;
      value = 8.0*pow(gaussianExponentA*gaussianExponentB,2.0)*temp;
      value /= sqrt(3.0)*pow(gaussianExponentA+gaussianExponentB,3.0);
   }
   else if((valenceOrbitalA == dzz && valenceOrbitalB == dyz) ||
           (valenceOrbitalA == dyz && valenceOrbitalB == dzz)){
      double temp1 = -1.0*dy*dz;
      double temp2 = 2.0*dy*pow(dz,3.0) - pow(dy,3.0)*dz - pow(dx,2.0)*dy*dz;
      temp2 *= gaussianExponentA*gaussianExponentB/(gaussianExponentA+gaussianExponentB);
      temp1 += temp2;
      value = 8.0*pow(gaussianExponentA*gaussianExponentB,2.0)*temp1;
      value /= sqrt(3.0)*pow(gaussianExponentA+gaussianExponentB,3.0);
   }
   else if((valenceOrbitalA == dzz && valenceOrbitalB == dzx) ||
           (valenceOrbitalA == dzx && valenceOrbitalB == dzz)){
      double temp1 = -1.0*dx*dz;
      double temp2 = 2.0*dx*pow(dz,3.0) - pow(dx,3.0)*dz - pow(dy,2.0)*dx*dz;
      temp2 *= gaussianExponentA*gaussianExponentB/(gaussianExponentA+gaussianExponentB);
      temp1 += temp2;
      value = 8.0*pow(gaussianExponentA*gaussianExponentB,2.0)*temp1;
      value /= sqrt(3.0)*pow(gaussianExponentA+gaussianExponentB,3.0);
   }
   else if((valenceOrbitalA == dxxyy && valenceOrbitalB == dzz) ||
           (valenceOrbitalA == dzz && valenceOrbitalB == dxxyy)){
      double temp = 2.0*pow(dz,2.0)-pow(dx,2.0)-pow(dy,2.0);
      temp *= gaussianExponentA*gaussianExponentB/(gaussianExponentA+gaussianExponentB);
      temp += 2.0;
      value = 4.0*pow(gaussianExponentA*gaussianExponentB,2.0);
      value /= sqrt(3.0)*pow(gaussianExponentA+gaussianExponentB,3.0);
      value *= (pow(dx,2.0)-pow(dy,2.0))*temp;
   }

   else{
      stringstream ss;
      ss << this->errorMessageGetGaussianOverlapBadOrbital;
      ss << this->errorMessageAtomA;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeA) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalA) << endl;
      ss << this->errorMessageAtomB;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeB) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalB) << endl;
      throw MolDSException(ss.str());
      value = 0.0;
   }
   double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA,
                                              gaussianExponentB, Rab);
   value *= sasb;

   return value;
}

// calculate elements of analytic first derivative of the overlap matrix. 
// The derivative is carried out related to the coordinate of atom A.
// See Eqs. (34) - (44) in [DY_1977]
double Cndo2::GetOverlapElement1stDerivativeByGTOExpansion(const Atom& atomA, 
                                                           int valenceIndexA, 
                                                           const Atom& atomB, 
                                                           int valenceIndexB,
                                                           STOnGType stonG, 
                                                           CartesianType axisA) const{

   double value = 0.0;
   double dx = atomA.GetXyz()[XAxis] - atomB.GetXyz()[XAxis];
   double dy = atomA.GetXyz()[YAxis] - atomB.GetXyz()[YAxis];
   double dz = atomA.GetXyz()[ZAxis] - atomB.GetXyz()[ZAxis];
   double Rab = sqrt( pow(dx, 2.0) + pow(dy, 2.0) + pow(dz,2.0) );
   ShellType shellTypeA = atomA.GetValenceShellType();
   ShellType shellTypeB = atomB.GetValenceShellType();
   OrbitalType valenceOrbitalA = atomA.GetValence(valenceIndexA);
   OrbitalType valenceOrbitalB = atomB.GetValence(valenceIndexB);
   double orbitalExponentA = atomA.GetOrbitalExponent(atomA.GetValenceShellType(), 
                                                      valenceOrbitalA, 
                                                      this->theory);
   double orbitalExponentB = atomB.GetOrbitalExponent(atomB.GetValenceShellType(), 
                                                      valenceOrbitalB, 
                                                      this->theory);
   double gaussianExponentA = 0.0;
   double gaussianExponentB = 0.0;

   double temp = 0.0;
   for(int i=0; i<=stonG; i++){
      for(int j=0; j<=stonG; j++){
         temp = GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                               shellTypeA, 
                                                               valenceOrbitalA, 
                                                               i); 
         temp *= GTOExpansionSTO::GetInstance()->GetCoefficient(stonG, 
                                                                shellTypeB, 
                                                                valenceOrbitalB, 
                                                                j); 
         gaussianExponentA = pow(orbitalExponentA, 2.0) 
                            *GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeA, 
                                                                         valenceOrbitalA, 
                                                                         i);
         gaussianExponentB = pow(orbitalExponentB, 2.0)
                            *GTOExpansionSTO::GetInstance()->GetExponent(stonG, 
                                                                         shellTypeB, 
                                                                         valenceOrbitalB, 
                                                                         j);
         temp *= this->GetGaussianOverlap1stDerivative(atomA.GetAtomType(), 
                                                       valenceOrbitalA, 
                                                       gaussianExponentA, 
                                                       atomB.GetAtomType(), 
                                                       valenceOrbitalB, 
                                                       gaussianExponentB,
                                                       dx, 
                                                       dy, 
                                                       dz, 
                                                       Rab, 
                                                       axisA);
         value += temp;
      }
   }
   return value;
}

// calculate first derivative of gaussian overlap integrals. 
// See Eqs. (35) - (44) in [DY_1977]
double Cndo2::GetGaussianOverlap1stDerivative(AtomType atomTypeA, 
                                              OrbitalType valenceOrbitalA, 
                                              double gaussianExponentA, 
                                              AtomType atomTypeB, 
                                              OrbitalType valenceOrbitalB, 
                                              double gaussianExponentB,
                                              double dx, double dy, double dz, double Rab, 
                                              CartesianType axisA) const{
   double value = 0.0;

   if(valenceOrbitalA == s && valenceOrbitalB == s){
      double temp = -2.0*gaussianExponentA*gaussianExponentB
                     /(gaussianExponentA+gaussianExponentB);
      value = temp;
      if(axisA == XAxis){
         value *= dx;
      }
      else if(axisA == YAxis){
         value *= dy;
      }
      else if(axisA == ZAxis){
         value *= dz;
      }
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == px){
      double temp1 = 4.0*pow(gaussianExponentA,2.0)*pow(gaussianExponentB, 1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         double temp2 = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)
                        /(gaussianExponentA+gaussianExponentB);
         value = temp2-temp1*dx*dx;
      }
      else if(axisA == YAxis){
         value = -1.0*temp1*dx*dy;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp1*dx*dz;
      }
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == py){
      double temp1 = 4.0*pow(gaussianExponentA,2.0)*pow(gaussianExponentB, 1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp1*dx*dy;
      }
      else if(axisA == YAxis){
         double temp2 = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)
                        /(gaussianExponentA+gaussianExponentB);
         value = temp2-temp1*dy*dy;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp1*dy*dz;
      }
   }
   else if(valenceOrbitalA == s && valenceOrbitalB == pz){
      double temp1 = 4.0*pow(gaussianExponentA,2.0)*pow(gaussianExponentB, 1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp1*dx*dz;
      }
      else if(axisA == YAxis){
         value = -1.0*temp1*dy*dz;
      }
      else if(axisA == ZAxis){
         double temp2 = 2.0*gaussianExponentA*pow(gaussianExponentB, 0.5)
                        /(gaussianExponentA+gaussianExponentB);
         value = temp2-temp1*dz*dz;
      }
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == s){
      double temp1 = 4.0*pow(gaussianExponentA,1.5)*pow(gaussianExponentB, 2.0)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         double temp2 = 2.0*pow(gaussianExponentA,0.5)*gaussianExponentB
                        /(gaussianExponentA+gaussianExponentB);
         value = -1.0*temp2+temp1*dx*dx;
      }
      else if(axisA == YAxis){
         value = temp1*dx*dy;
      }
      else if(axisA == ZAxis){
         value = temp1*dx*dz;
      }
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == s){
      double temp1 = 4.0*pow(gaussianExponentA,1.5)*pow(gaussianExponentB, 2.0)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = temp1*dx*dy;
      }
      else if(axisA == YAxis){
         double temp2 = 2.0*pow(gaussianExponentA,0.5)*gaussianExponentB
                        /(gaussianExponentA+gaussianExponentB);
         value = -1.0*temp2+temp1*dy*dy;
      }
      else if(axisA == ZAxis){
         value = temp1*dy*dz;
      }
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == s){
      double temp1 = 4.0*pow(gaussianExponentA,1.5)*pow(gaussianExponentB, 2.0)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = temp1*dx*dz;
      }
      else if(axisA == YAxis){
         value = temp1*dy*dz;
      }
      else if(axisA == ZAxis){
         double temp2 = 2.0*pow(gaussianExponentA,0.5)*gaussianExponentB
                        /(gaussianExponentA+gaussianExponentB);
         value = -1.0*temp2+temp1*dz*dz;
      }
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == py){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp2*dy+temp1*dx*dx*dy;
      }
      else if(axisA == YAxis){
         value = -1.0*temp2*dx+temp1*dx*dy*dy;
      }
      else if(axisA == ZAxis){
         value = temp1*dx*dy*dz;
      }
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == px){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp2*dy+temp1*dy*dx*dx;
      }
      else if(axisA == YAxis){
         value = -1.0*temp2*dx+temp1*dy*dy*dx;
      }
      else if(axisA == ZAxis){
         value = temp1*dx*dy*dz;
      }
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == pz){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp2*dz+temp1*dx*dx*dz;
      }
      else if(axisA == YAxis){
         value = temp1*dx*dy*dz;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp2*dx+temp1*dx*dz*dz;
      }
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == px){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = -1.0*temp2*dz+temp1*dz*dx*dx;
      }
      else if(axisA == YAxis){
         value = temp1*dx*dy*dz;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp2*dx+temp1*dz*dz*dx;
      }
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == pz){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = temp1*dx*dy*dz;
      }
      else if(axisA == YAxis){
         value = -1.0*temp2*dz+temp1*dy*dy*dz;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp2*dy+temp1*dy*dz*dz;
      }
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == py){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,2.5)
                     /pow(gaussianExponentA+gaussianExponentB,3.0);
      double temp2 = 4.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      if(axisA == XAxis){
         value = temp1*dx*dy*dz;
      }
      else if(axisA == YAxis){
         value = -1.0*temp2*dz+temp1*dz*dy*dy;
      }
      else if(axisA == ZAxis){
         value = -1.0*temp2*dy+temp1*dz*dz*dy;
      }
   }
   else if(valenceOrbitalA == px && valenceOrbitalB == px){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      double temp2 = gaussianExponentA*gaussianExponentB
                     /(gaussianExponentA+gaussianExponentB); 
      if(axisA == XAxis){
         value = -1.0*temp1*dx*(1.5-temp2*dx*dx);
      }
      else if(axisA == YAxis){
         value = -1.0*temp1*dy*(0.5-temp2*dx*dx);
      }
      else if(axisA == ZAxis){
         value = -1.0*temp1*dz*(0.5-temp2*dx*dx);
      }
   }
   else if(valenceOrbitalA == py && valenceOrbitalB == py){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      double temp2 = gaussianExponentA*gaussianExponentB
                     /(gaussianExponentA+gaussianExponentB); 
      if(axisA == XAxis){
         value = -1.0*temp1*dx*(0.5-temp2*dy*dy);
      }
      else if(axisA == YAxis){
         value = -1.0*temp1*dy*(1.5-temp2*dy*dy);
      }
      else if(axisA == ZAxis){
         value = -1.0*temp1*dz*(0.5-temp2*dy*dy);
      }
   }
   else if(valenceOrbitalA == pz && valenceOrbitalB == pz){
      double temp1 = 8.0*pow(gaussianExponentA*gaussianExponentB,1.5)
                     /pow(gaussianExponentA+gaussianExponentB,2.0);
      double temp2 = gaussianExponentA*gaussianExponentB
                     /(gaussianExponentA+gaussianExponentB); 
      if(axisA == XAxis){
         value = -1.0*temp1*dx*(0.5-temp2*dz*dz);
      }
      else if(axisA == YAxis){
         value = -1.0*temp1*dy*(0.5-temp2*dz*dz);
      }
      else if(axisA == ZAxis){
         value = -1.0*temp1*dz*(1.5-temp2*dz*dz);
      }
   }
   else{
      stringstream ss;
      ss << this->errorMessageGetGaussianOverlap1stDerivativeOrbitalD;
      ss << this->errorMessageAtomA;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeA) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalA) << endl;
      ss << this->errorMessageAtomB;
      ss << this->errorMessageAtomType << AtomTypeStr(atomTypeB) << endl;
      ss << this->errorMessageOrbitalType << OrbitalTypeStr(valenceOrbitalB) << endl;
      throw MolDSException(ss.str());
   }

   double sasb = this->GetGaussianOverlapSaSb(gaussianExponentA,
                                              gaussianExponentB, Rab);
   value *= sasb;
   return value;
}

// see J. Mol. Struc. (Theochem), 419, 19 (1997) (ref. [BFB_1997])
// we set gamma=0 always.
void Cndo2::CalcRotatingMatrix(double** rotatingMatrix, 
                               const Atom& atomA, 
                               const Atom& atomB) const{
   if(rotatingMatrix==NULL){
      stringstream ss;
      ss << this->errorMessageCalcRotatingMatrixNullRotMatrix;
      throw MolDSException(ss.str());
   }
   MallocerFreer::GetInstance()->Initialize<double>(rotatingMatrix,  OrbitalType_end, OrbitalType_end);

   double x = atomB.GetXyz()[0] - atomA.GetXyz()[0];
   double y = atomB.GetXyz()[1] - atomA.GetXyz()[1];
   double z = atomB.GetXyz()[2] - atomA.GetXyz()[2];

   EularAngle eularAngle(x, y, z);
   double alpha = eularAngle.GetAlpha();
   double beta  = eularAngle.GetBeta();

   // rotating matrix for s-function
   rotatingMatrix[s][s] = 1.0;

   // rotating matrix for p-function
   // dMatrix is (53) with gamma=0 in J. Mol. Strct. 419, 19(1997) (ref. [BFB_1997])
   rotatingMatrix[py][py] = cos(alpha);
   rotatingMatrix[py][pz] = sin(alpha)*sin(beta);
   rotatingMatrix[py][px] = sin(alpha)*cos(beta);

   rotatingMatrix[pz][py] = 0.0;
   rotatingMatrix[pz][pz] = cos(beta);
   rotatingMatrix[pz][px] = -1.0*sin(beta);

   rotatingMatrix[px][py] = -1.0*sin(alpha);
   rotatingMatrix[px][pz] = cos(alpha)*sin(beta);
   rotatingMatrix[px][px] = cos(alpha)*cos(beta);

   // rotating matrix for d-function
   // dMatrix is (37) in J. Mol. Strct. 419, 19(1997) (ref. [BFB_1997])
   double** dMatrix = NULL;
   try{
      MallocerFreer::GetInstance()->Malloc<double>(&dMatrix, OrbitalType_end, OrbitalType_end);
      dMatrix[dzz][dzz] = 0.5*(3.0*pow(cos(beta),2.0) - 1.0);
      dMatrix[dxxyy][dxxyy] = pow(cos(0.5*beta),4.0);
      dMatrix[dzx][dzx] = (2.0*cos(beta)-1.0)*pow(cos(0.5*beta),2.0);
      dMatrix[dxxyy][dzx] = -2.0*sin(0.5*beta)*pow(cos(0.5*beta),3.0);
      dMatrix[dxxyy][dzz] = sqrt(6.0)*pow(sin(0.5*beta),2.0)*pow(cos(0.5*beta),2.0);
      dMatrix[dxxyy][dyz] = -2.0*pow(sin(0.5*beta),3.0)*pow(cos(0.5*beta),1.0);
      dMatrix[dxxyy][dxy] = pow(sin(0.5*beta),4.0);
      dMatrix[dzx][dzz] = -sqrt(6.0)*cos(beta)*cos(0.5*beta)*sin(0.5*beta);
      dMatrix[dzx][dyz] = (2.0*cos(beta)+1.0)*pow(sin(0.5*beta),2.0);

      rotatingMatrix[dxy][dxy] = cos(2.0*alpha)*            (dMatrix[dxxyy][dxxyy] - dMatrix[dxxyy][dxy]);
      rotatingMatrix[dxy][dyz] = cos(2.0*alpha)*            (-1.0*dMatrix[dxxyy][dzx] - dMatrix[dxxyy][dyz]);
      rotatingMatrix[dxy][dzz] = sqrt(2.0)*sin(2.0*alpha)*  dMatrix[dxxyy][dzz];
      rotatingMatrix[dxy][dzx] = sin(2.0*alpha)*            (-1.0*dMatrix[dxxyy][dzx] + dMatrix[dxxyy][dyz]);
      rotatingMatrix[dxy][dxxyy] = sin(2.0*alpha)*          (dMatrix[dxxyy][dxxyy] + dMatrix[dxxyy][dxy]);

      rotatingMatrix[dyz][dxy] = cos(alpha)*                (dMatrix[dxxyy][dzx] + dMatrix[dxxyy][dyz]);
      rotatingMatrix[dyz][dyz] = cos(alpha)*                (dMatrix[dzx][dzx] + dMatrix[dzx][dyz]);
      rotatingMatrix[dyz][dzz] = -1.0*sqrt(2.0)*sin(alpha)* dMatrix[dzx][dzz];
      rotatingMatrix[dyz][dzx] = sin(alpha)*                (dMatrix[dzx][dzx] - dMatrix[dzx][dyz]);
      rotatingMatrix[dyz][dxxyy] = sin(alpha)*              (dMatrix[dxxyy][dzx] - dMatrix[dxxyy][dyz]);

      rotatingMatrix[dzz][dxy] = 0.0;
      rotatingMatrix[dzz][dyz] = 0.0;
      rotatingMatrix[dzz][dzz] = dMatrix[dzz][dzz];
      rotatingMatrix[dzz][dzx] = sqrt(2.0)*dMatrix[dzx][dzz];
      rotatingMatrix[dzz][dxxyy] = sqrt(2.0)*dMatrix[dxxyy][dzz];

      rotatingMatrix[dzx][dxy] = -1.0*sin(alpha)*           (dMatrix[dxxyy][dzx] + dMatrix[dxxyy][dyz]);
      rotatingMatrix[dzx][dyz] = -1.0*sin(alpha)*           (dMatrix[dzx][dzx] + dMatrix[dzx][dyz]);
      rotatingMatrix[dzx][dzz] = -1.0*sqrt(2.0)*cos(alpha)* dMatrix[dzx][dzz];
      rotatingMatrix[dzx][dzx] = cos(alpha)*                (dMatrix[dzx][dzx] - dMatrix[dzx][dyz]);
      rotatingMatrix[dzx][dxxyy] = cos(alpha)*              (dMatrix[dxxyy][dzx] - dMatrix[dxxyy][dyz]);

      rotatingMatrix[dxxyy][dxy] = -1.0*sin(2.0*alpha)*     (dMatrix[dxxyy][dxxyy] - dMatrix[dxxyy][dxy]);
      rotatingMatrix[dxxyy][dyz] = -1.0*sin(2.0*alpha)*     (-1.0*dMatrix[dxxyy][dzx] - dMatrix[dxxyy][dyz]);
      rotatingMatrix[dxxyy][dzz] = sqrt(2.0)*cos(2.0*alpha)*dMatrix[dxxyy][dzz];
      rotatingMatrix[dxxyy][dzx] = cos(2.0*alpha)*          (-1.0*dMatrix[dxxyy][dzx] + dMatrix[dxxyy][dyz]);
      rotatingMatrix[dxxyy][dxxyy] = cos(2.0*alpha)*        (dMatrix[dxxyy][dxxyy] + dMatrix[dxxyy][dxy]);
   }
   catch(MolDSException ex){
      MallocerFreer::GetInstance()->Free<double>(&dMatrix, OrbitalType_end, OrbitalType_end);
      throw ex;
   }
   MallocerFreer::GetInstance()->Free<double>(&dMatrix, OrbitalType_end, OrbitalType_end);
}

// First derivative of rotating matirx. 
// This derivative is related to a coordinate of atom A.
// This method can not calculate d-orbital yet.
// For rotating matirxi, see J. Mol. Struc. (Theochem), 419, 19 (1997) (ref. [BFB_1997])
// we set gamma=0 always.
void Cndo2::CalcRotatingMatrix1stDerivatives(double*** rotMat1stDerivatives, 
                                             const Atom& atomA, 
                                             const Atom& atomB) const{

   MallocerFreer::GetInstance()->Initialize<double>(
                                 rotMat1stDerivatives,  
                                 OrbitalType_end, 
                                 OrbitalType_end,
                                 CartesianType_end);

   double x = atomB.GetXyz()[0] - atomA.GetXyz()[0];
   double y = atomB.GetXyz()[1] - atomA.GetXyz()[1];
   double z = atomB.GetXyz()[2] - atomA.GetXyz()[2];
   double r = sqrt( pow(x,2.0) + pow(y,2.0) );
   double R = sqrt( pow(x,2.0) + pow(y,2.0) + pow(z,2.0) );

   // for s-function
   rotMat1stDerivatives[s][s][XAxis] = 0.0;
   rotMat1stDerivatives[s][s][YAxis] = 0.0;
   rotMat1stDerivatives[s][s][ZAxis] = 0.0;

   // for p-function
   rotMat1stDerivatives[py][py][XAxis] = -1.0/r + pow(x,2.0)/pow(r,3.0);
   rotMat1stDerivatives[py][pz][XAxis] = x*y/pow(R,3.0);
   rotMat1stDerivatives[py][px][XAxis] = (1.0/(pow(r,3.0)*R) + 1.0/(pow(R,3.0)*r))*x*y*z;

   rotMat1stDerivatives[pz][py][XAxis] = 0.0;
   rotMat1stDerivatives[pz][pz][XAxis] = x*z/pow(R,3.0);
   rotMat1stDerivatives[pz][px][XAxis] = x/(r*R) - x*r/pow(R,3.0);

   rotMat1stDerivatives[px][py][XAxis] = -1.0*x*y/pow(r,3.0);
   rotMat1stDerivatives[px][pz][XAxis] = -1.0/R + x*x/pow(R,3.0); 
   rotMat1stDerivatives[px][px][XAxis] = -1.0*z/(r*R) + 
                                  (1.0/(pow(r,3.0)*R) + 1.0/(pow(R,3.0)*r))*x*x*z;

   rotMat1stDerivatives[py][py][YAxis] = x*y/pow(r,3.0);
   rotMat1stDerivatives[py][pz][YAxis] = -1.0/R + y*y/pow(R,3.0);
   rotMat1stDerivatives[py][px][YAxis] = -1.0*z/(r*R) +
                                  (1.0/(pow(r,3.0)*R) + 1.0/(pow(R,3.0)*r))*y*y*z;

   rotMat1stDerivatives[pz][py][YAxis] = 0.0;
   rotMat1stDerivatives[pz][pz][YAxis] = y*z/pow(R,3.0); 
   rotMat1stDerivatives[pz][px][YAxis] = y/(r*R) - y*r/pow(R,3.0);

   rotMat1stDerivatives[px][py][YAxis] = 1.0/r - y*y/pow(r,3.0);
   rotMat1stDerivatives[px][pz][YAxis] = x*y/pow(R,3.0);
   rotMat1stDerivatives[px][px][YAxis] = (1.0/(pow(r,3.0)*R) + 1.0/(pow(R,3.0)*r))*x*y*z;

   rotMat1stDerivatives[py][py][ZAxis] = 0.0;
   rotMat1stDerivatives[py][pz][ZAxis] = y*z/pow(R,3.0);
   rotMat1stDerivatives[py][px][ZAxis] = -1.0*y/(r*R) + y*z*z/(r*pow(R,3.0));

   rotMat1stDerivatives[pz][py][ZAxis] = 0.0;
   rotMat1stDerivatives[pz][pz][ZAxis] = -1.0/R + z*z/pow(R,3.0); 
   rotMat1stDerivatives[pz][px][ZAxis] = -1.0*z*r/pow(R,3.0);

   rotMat1stDerivatives[px][py][ZAxis] = 0.0;
   rotMat1stDerivatives[px][pz][ZAxis] = x*z/pow(R,3.0);
   rotMat1stDerivatives[px][px][ZAxis] = -1.0*x/(r*R) + x*z*z/(r*pow(R,3.0));

   // for d-function
   // ToDo: First derivative of rotating matrix for d-orbital...

}

// Second derivative of rotating matirx. 
// Both derivatives are related to a coordinate of atom A.
// This method can not calculate d-orbital yet.
// For rotating matirxi, see J. Mol. Struc. (Theochem), 419, 19 (1997) (ref. [BFB_1997])
// we set gamma=0 always.
void Cndo2::CalcRotatingMatrix2ndDerivatives(double**** rotMat2ndDerivatives, 
                                                const Atom& atomA, 
                                                const Atom& atomB) const{

   MallocerFreer::GetInstance()->Initialize<double>(
                                 rotMat2ndDerivatives,  
                                 OrbitalType_end, 
                                 OrbitalType_end,
                                 CartesianType_end,
                                 CartesianType_end);

   double x = atomB.GetXyz()[0] - atomA.GetXyz()[0];
   double y = atomB.GetXyz()[1] - atomA.GetXyz()[1];
   double z = atomB.GetXyz()[2] - atomA.GetXyz()[2];
   double r = sqrt( pow(x,2.0) + pow(y,2.0) );
   double R = sqrt( pow(x,2.0) + pow(y,2.0) + pow(z,2.0) );

   // for s-function
   rotMat2ndDerivatives[s][s][XAxis][XAxis] = 0.0;
   rotMat2ndDerivatives[s][s][XAxis][YAxis] = 0.0;
   rotMat2ndDerivatives[s][s][XAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[s][s][YAxis][XAxis] = 0.0;
   rotMat2ndDerivatives[s][s][YAxis][YAxis] = 0.0;
   rotMat2ndDerivatives[s][s][YAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[s][s][ZAxis][XAxis] = 0.0;
   rotMat2ndDerivatives[s][s][ZAxis][YAxis] = 0.0;
   rotMat2ndDerivatives[s][s][ZAxis][ZAxis] = 0.0;

   // for p-function, xx-derivatives
   rotMat2ndDerivatives[py][py][XAxis][XAxis] = -3.0*x*pow(r,-3.0) + 3.0*pow(x,3.0)*pow(r,-5.0);
   rotMat2ndDerivatives[py][pz][XAxis][XAxis] = -1.0*y*pow(R,-3.0) + 3.0*pow(x,2.0)*y*pow(R,-5.0);
   rotMat2ndDerivatives[py][px][XAxis][XAxis] = -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*y*z
                                                   +(2.0*pow(r*R,-3.0) + 3.0/(pow(r,5.0)*R) + 3.0/(r*pow(R,5.0)))*pow(x,2.0)*y*z;
                                                
   rotMat2ndDerivatives[pz][py][XAxis][XAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][XAxis][XAxis] = -1.0*z*pow(R,-3.0) + 3.0*pow(x,2.0)*z*pow(R,-5.0);
   rotMat2ndDerivatives[pz][px][XAxis][XAxis] = -1.0*pow(r*R,-1.0) + (1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*pow(x,2.0)
                                                   +r*pow(R,-3.0) - 3.0*pow(x,2.0)*r*pow(R,-5.0) + pow(x,2.0)*pow(r,-1.0)*pow(R,-3.0);
                                                
   rotMat2ndDerivatives[px][py][XAxis][XAxis] = y*pow(r,-3.0) - 3.0*pow(x,2.0)*y*pow(r,-5.0);
   rotMat2ndDerivatives[px][pz][XAxis][XAxis] = -3.0*x*pow(R,-3.0) + 3.0*pow(x,3.0)*pow(R,-5.0);
   rotMat2ndDerivatives[px][px][XAxis][XAxis] = -3.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*z
                                                   +(2.0*pow(r*R,-3.0) + 3.0/(pow(r,5.0)*R) + 3.0/(r*pow(R,5.0)))*pow(x,3.0)*z;

   // for p-function, xy-derivatives
   rotMat2ndDerivatives[py][py][XAxis][YAxis] = -1.0*y*pow(r,-3.0) + 3.0*pow(x,2.0)*y*pow(r,-5.0);
   rotMat2ndDerivatives[py][pz][XAxis][YAxis] = -1.0*x*pow(R,-3.0) + 3.0*x*pow(y,2.0)*pow(R,-5.0);  
   rotMat2ndDerivatives[py][px][XAxis][YAxis] = -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*z
                                                   +(2.0*pow(r*R,-3.0) + 3.0/(pow(r,5.0)*R) + 3.0/(r*pow(R,5.0)))*x*pow(y,2.0)*z;
                                                
   rotMat2ndDerivatives[pz][py][XAxis][YAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][XAxis][YAxis] = 3.0*x*y*z*pow(R,-5.0);
   rotMat2ndDerivatives[pz][px][XAxis][YAxis] = (1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*y + x*y*pow(r,-1.0)*pow(R,-3.0) - 3.0*x*y*r*pow(R,-5.0);
                                                
   rotMat2ndDerivatives[px][py][XAxis][YAxis] = x*pow(r,-3.0) - 3.0*x*pow(y,2.0)*pow(R,-5.0);
   rotMat2ndDerivatives[px][pz][XAxis][YAxis] = rotMat2ndDerivatives[py][pz][XAxis][XAxis];
   rotMat2ndDerivatives[px][px][XAxis][YAxis] = rotMat2ndDerivatives[py][px][XAxis][XAxis];

   // for p-function, yx-derivatives
   for(int i=py; i<=px; i++){
      for(int j=py; j<=px; j++){
         rotMat2ndDerivatives[i][j][YAxis][XAxis] = rotMat2ndDerivatives[i][j][XAxis][YAxis];
      }
   }

   // for p-function, xz-derivatives
   rotMat2ndDerivatives[py][py][XAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[py][pz][XAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][XAxis][YAxis];
   rotMat2ndDerivatives[py][px][XAxis][ZAxis] = -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*y 
                                                   +(pow(r*R,-3.0) + 3.0/(r*pow(R,5.0)))*x*y*pow(z,2.0);
                                                
   rotMat2ndDerivatives[pz][py][XAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][XAxis][ZAxis] = -1.0*x*pow(R,-3.0) + 3.0*x*pow(z,2.0)*pow(R,-5.0); 
   rotMat2ndDerivatives[pz][px][XAxis][ZAxis] = x*z*pow(r,-1.0)*pow(R,-3.0) - 3.0*x*z*r*pow(R,-5.0);
                                                
   rotMat2ndDerivatives[px][py][XAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[px][pz][XAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][XAxis][XAxis];
   rotMat2ndDerivatives[px][px][XAxis][ZAxis] = pow(r*R,-1.0) - pow(z,2.0)*pow(r,-1.0)*pow(R,-3.0)
                                                   -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*pow(x,2.0)
                                                   +(pow(r*R,-3.0) + 3.0/(r*pow(R,5.0)))*pow(x*z,2.0);


   // for p-function, zx-derivatives
   for(int i=py; i<=px; i++){
      for(int j=py; j<=px; j++){
         rotMat2ndDerivatives[i][j][ZAxis][XAxis] = rotMat2ndDerivatives[i][j][XAxis][ZAxis];
      }
   }

   // for p-function, yy-derivatives
   rotMat2ndDerivatives[py][py][YAxis][YAxis] = -1.0*x*pow(r,-3.0) + 3.0*x*pow(y,2.0)*pow(r,-5.0); 
   rotMat2ndDerivatives[py][pz][YAxis][YAxis] = -3.0*y*pow(R,-3.0) + 3.0*pow(y,3.0)*pow(R,-5.0);
   rotMat2ndDerivatives[py][px][YAxis][YAxis] = -3.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*y*z
                                                   +(2.0*pow(r*R,-3.0) + 3.0/(pow(r,5.0)*R) + 3.0/(r*pow(R,5.0)))*pow(y,3.0)*z;
                                                
   rotMat2ndDerivatives[pz][py][YAxis][YAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][YAxis][YAxis] = -1.0*z*pow(R,-3.0) + 3.0*pow(y,2.0)*z*pow(R,-5.0);
   rotMat2ndDerivatives[pz][px][YAxis][YAxis] = -1.0*pow(r*R,-1.0) + (1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*pow(y,2.0)
                                                   +r*pow(R,-3.0) - 3.0*pow(y,2.0)*r*pow(R,-5.0) + pow(y,2.0)*pow(r,-1.0)*pow(R,-3.0);
                                                
   rotMat2ndDerivatives[px][py][YAxis][YAxis] = 3.0*y*pow(r,-3.0) - 3.0*pow(y,3.0)*pow(r,-5.0);
   rotMat2ndDerivatives[px][pz][YAxis][YAxis] = rotMat2ndDerivatives[py][pz][XAxis][YAxis];
   rotMat2ndDerivatives[px][px][YAxis][YAxis] = -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*z
                                                   +(2.0*pow(r*R,-3.0) + 3.0/(pow(r,5.0)*R) + 3.0/(r*pow(R,5.0)))*x*pow(y,2.0)*z;
               
   // for p-function, yz-derivatives
   rotMat2ndDerivatives[py][py][YAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[py][pz][YAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][YAxis][YAxis];
   rotMat2ndDerivatives[py][px][YAxis][ZAxis] = pow(r*R,-1.0) - pow(z,2.0)*pow(r,-1.0)*pow(R,-3.0)
                                                   -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*pow(y,2.0)
                                                   +(pow(r*R,-3.0) + 3.0/(r*pow(R,5.0)))*pow(y*z,2.0);
                                                
   rotMat2ndDerivatives[pz][py][YAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][YAxis][ZAxis] = -1.0*y*pow(R,-3.0) + 3.0*y*pow(z,2.0)*pow(R,-5.0);
   rotMat2ndDerivatives[pz][px][YAxis][ZAxis] = y*z*pow(r,-1.0)*pow(R,-3.0) - 3.0*y*z*r*pow(R,-5.0);
                                                
   rotMat2ndDerivatives[px][py][YAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[px][pz][YAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][XAxis][YAxis];
   rotMat2ndDerivatives[px][px][YAxis][ZAxis] = -1.0*(1.0/(pow(r,3.0)*R) + 1.0/(r*pow(R,3.0)))*x*y
                                                   +(pow(r*R,-3.0) + 3.0/(r*pow(R,5.0)))*x*y*pow(z,2.0);
                                          
               
   // for p-function, zy-derivatives
   for(int i=py; i<=px; i++){
      for(int j=py; j<=px; j++){
         rotMat2ndDerivatives[i][j][ZAxis][YAxis] = rotMat2ndDerivatives[i][j][YAxis][ZAxis];
      }
   }

   // for p-function, zz-derivatives
   rotMat2ndDerivatives[py][py][ZAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[py][pz][ZAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][YAxis][ZAxis];
   rotMat2ndDerivatives[py][px][ZAxis][ZAxis] = -3.0*y*z*pow(r,-1.0)*pow(R,-3.0) + 3.0*y*pow(z,3.0)*pow(r,-1.0)*pow(R,-5.0);
                                                
   rotMat2ndDerivatives[pz][py][ZAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[pz][pz][ZAxis][ZAxis] = -3.0*z*pow(R,-3.0) + 3.0*pow(z,3.0)*pow(R,-5.0); 
   rotMat2ndDerivatives[pz][px][ZAxis][ZAxis] = -3.0*pow(z,2.0)*r*pow(R,-5.0) + r*pow(R,-3.0);
                                                
   rotMat2ndDerivatives[px][py][ZAxis][ZAxis] = 0.0;
   rotMat2ndDerivatives[px][pz][ZAxis][ZAxis] = rotMat2ndDerivatives[pz][pz][XAxis][ZAxis];
   rotMat2ndDerivatives[px][px][ZAxis][ZAxis] = -3.0*x*z*pow(r,-1.0)*pow(R,-3.0) + 3.0*x*pow(z,3.0)*pow(r,-1.0)*pow(R,-5.0);

   // for d-function
   // ToDo: Second derivative of rotating matrix for d-orbital...
}

// see (B.40) in J. A. Pople book.
void Cndo2::CalcDiatomicOverlapInDiatomicFrame(double** diatomicOverlap, 
                                               const Atom& atomA, 
                                               const Atom& atomB) const{
   if(diatomicOverlap==NULL){
      stringstream ss;
      ss << this->errorMessageCalDiaOverlapDiaFrameNullMatrix;
      throw MolDSException(ss.str());
   }
   int na = atomA.GetValenceShellType() + 1;
   int nb = atomB.GetValenceShellType() + 1;
   int m = 0;
   double alpha = 0.0;
   double beta = 0.0;
   double pre = 0.0;
   double reducedOverlap = 0.0;
   double orbitalExponentA = 0.0;
   double orbitalExponentB = 0.0;
   double R = 0.0; // Inter nuclear distance between aton A and B.

   MallocerFreer::GetInstance()->Initialize<double>(diatomicOverlap, OrbitalType_end, OrbitalType_end);
   R = sqrt( 
            pow( atomA.GetXyz()[0] - atomB.GetXyz()[0], 2.0)
           +pow( atomA.GetXyz()[1] - atomB.GetXyz()[1], 2.0)
           +pow( atomA.GetXyz()[2] - atomB.GetXyz()[2], 2.0)
           );

   for(int a=0; a<atomA.GetValenceSize(); a++){
      OrbitalType valenceOrbitalA = atomA.GetValence(a);
      RealSphericalHarmonicsIndex realShpericalHarmonicsA(valenceOrbitalA);
      orbitalExponentA = atomA.GetOrbitalExponent(
                               atomA.GetValenceShellType(), 
                               valenceOrbitalA, 
                               this->theory);

      for(int b=0; b<atomB.GetValenceSize(); b++){
         OrbitalType valenceOrbitalB = atomB.GetValence(b);
         RealSphericalHarmonicsIndex realShpericalHarmonicsB(valenceOrbitalB);
         orbitalExponentB = atomB.GetOrbitalExponent(
                                  atomB.GetValenceShellType(), 
                                  valenceOrbitalB, 
                                  this->theory);

         if(realShpericalHarmonicsA.GetM() == realShpericalHarmonicsB.GetM()){
            m = abs(realShpericalHarmonicsA.GetM());
            alpha = orbitalExponentA * R;
            beta =  orbitalExponentB * R;

            reducedOverlap = this->GetReducedOverlap
                                   (na, realShpericalHarmonicsA.GetL(), m,
                                    nb, realShpericalHarmonicsB.GetL(), alpha, beta);


            pre =  pow(2.0*orbitalExponentA, na+0.5);
            pre *= pow(2.0*orbitalExponentB, nb+0.5);
            double factorials = Factorial(2*na)*Factorial(2*nb);
            pre /= sqrt(factorials);
            pre *= pow(R/2.0, na+nb+1.0);

            diatomicOverlap[valenceOrbitalA][valenceOrbitalB] = pre*reducedOverlap;
         }
         
      }
   }

   /*
   for(int i=0;i<OrbitalType_end;i++){
      for(int j=0;j<OrbitalType_end;j++){
         this->OutputLog(boost::format("diatomicOverlap[%d][%d]=%lf\n") % i % j % diatomicOverlap[i][j]);
      }
   }
   */
}

// First derivative of (B.40) in J. A. Pople book.
void Cndo2::CalcDiatomicOverlap1stDerivativeInDiatomicFrame(double** diatomicOverlapDeri, 
                                                            const Atom& atomA, 
                                                            const Atom& atomB) const{

   int na = atomA.GetValenceShellType() + 1;
   int nb = atomB.GetValenceShellType() + 1;
   int m = 0;
   double alpha = 0.0;
   double beta = 0.0;
   double pre = 0.0;
   double reducedOverlap = 0.0;
   double reducedOverlap1stDerivAlpha = 0.0;
   double reducedOverlap1stDerivBeta = 0.0;
   double orbitalExponentA = 0.0;
   double orbitalExponentB = 0.0;
   double temp1=0.0;
   double temp2=0.0;

   MallocerFreer::GetInstance()->Initialize<double>(diatomicOverlapDeri, 
                                                    OrbitalType_end, 
                                                    OrbitalType_end);
   double R = this->molecule->GetDistanceAtoms(atomA, atomB);

   for(int a=0; a<atomA.GetValenceSize(); a++){
      OrbitalType valenceOrbitalA = atomA.GetValence(a);
      RealSphericalHarmonicsIndex realShpericalHarmonicsA(valenceOrbitalA);
      orbitalExponentA = atomA.GetOrbitalExponent(
                               atomA.GetValenceShellType(), 
                               valenceOrbitalA,
                               this->theory);

      for(int b=0; b<atomB.GetValenceSize(); b++){
         OrbitalType valenceOrbitalB = atomB.GetValence(b);
         RealSphericalHarmonicsIndex realShpericalHarmonicsB(valenceOrbitalB);
         orbitalExponentB = atomB.GetOrbitalExponent(
                                  atomB.GetValenceShellType(), 
                                  valenceOrbitalB,
                                  this->theory);

         if(realShpericalHarmonicsA.GetM() == realShpericalHarmonicsB.GetM()){
            m = abs(realShpericalHarmonicsA.GetM());
            alpha = orbitalExponentA * R;
            beta =  orbitalExponentB * R;

            reducedOverlap = this->GetReducedOverlap
                                   (na, realShpericalHarmonicsA.GetL(), m,
                                    nb, realShpericalHarmonicsB.GetL(), alpha, beta);
            reducedOverlap1stDerivAlpha = this->GetReducedOverlap1stDerivativeAlpha(
                                                  na, 
                                                  realShpericalHarmonicsA.GetL(), 
                                                  m,
                                                  nb, 
                                                  realShpericalHarmonicsB.GetL(), 
                                                  alpha, 
                                                  beta);
            reducedOverlap1stDerivBeta  = this->GetReducedOverlap1stDerivativeBeta(
                                                  na, 
                                                  realShpericalHarmonicsA.GetL(), 
                                                  m,
                                                  nb, 
                                                  realShpericalHarmonicsB.GetL(), 
                                                  alpha, 
                                                  beta);

            temp1 = static_cast<double>(na+nb+1)*pow(R,na+nb)*reducedOverlap;
            temp2 = pow(R,na+nb+1)*(orbitalExponentA*reducedOverlap1stDerivAlpha
                                   +orbitalExponentB*reducedOverlap1stDerivBeta);

            pre =  pow(2.0*orbitalExponentA, na+0.5);
            pre *= pow(2.0*orbitalExponentB, nb+0.5);
            double factorials = Factorial(2*na)*Factorial(2*nb);
            pre /= sqrt(factorials);
            pre /= pow(2.0, na+nb+1.0);

            diatomicOverlapDeri[valenceOrbitalA][valenceOrbitalB] = pre*(temp1+temp2);
         }
         
      }
   }

   /*
   for(int i=0;i<OrbitalType_end;i++){
      for(int j=0;j<OrbitalType_end;j++){
         this->OutputLog(boost::format("diatomicOverlap[%d][%d]=%lf\n") % i % j % diatomicOverlap[i][j]);
      }
   }
   */

}

// Second derivative of (B.40) in J. A. Pople book.
void Cndo2::CalcDiatomicOverlap2ndDerivativeInDiatomicFrame(double** diatomicOverlap2ndDeri, 
                                                               const Atom& atomA, 
                                                               const Atom& atomB) const{

   int na = atomA.GetValenceShellType() + 1;
   int nb = atomB.GetValenceShellType() + 1;
   int m = 0;
   double alpha = 0.0;
   double beta = 0.0;
   double pre = 0.0;
   double reducedOverlap = 0.0;
   double reducedOverlap1stDerivAlpha = 0.0;
   double reducedOverlap1stDerivBeta = 0.0;
   double reducedOverlap2ndDerivAlpha = 0.0;
   double reducedOverlap2ndDerivBeta = 0.0;
   double reducedOverlap2ndDerivAlphaBeta = 0.0;
   double orbitalExponentA = 0.0;
   double orbitalExponentB = 0.0;
   double temp1=0.0;
   double temp2=0.0;
   double temp3=0.0;

   MallocerFreer::GetInstance()->Initialize<double>(diatomicOverlap2ndDeri, 
                                                    OrbitalType_end, 
                                                    OrbitalType_end);
   double R = this->molecule->GetDistanceAtoms(atomA, atomB);

   for(int a=0; a<atomA.GetValenceSize(); a++){
      OrbitalType valenceOrbitalA = atomA.GetValence(a);
      RealSphericalHarmonicsIndex realShpericalHarmonicsA(valenceOrbitalA);
      orbitalExponentA = atomA.GetOrbitalExponent(atomA.GetValenceShellType(),
                                                  valenceOrbitalA,
                                                  this->theory);

      for(int b=0; b<atomB.GetValenceSize(); b++){
         OrbitalType valenceOrbitalB = atomB.GetValence(b);
         RealSphericalHarmonicsIndex realShpericalHarmonicsB(valenceOrbitalB);
         orbitalExponentB = atomB.GetOrbitalExponent(atomB.GetValenceShellType(),
                                                     valenceOrbitalB,
                                                     this->theory);

         if(realShpericalHarmonicsA.GetM() == realShpericalHarmonicsB.GetM()){
            m = abs(realShpericalHarmonicsA.GetM());
            alpha = orbitalExponentA * R;
            beta =  orbitalExponentB * R;

            reducedOverlap = this->GetReducedOverlap(na,
                                                     realShpericalHarmonicsA.GetL(),
                                                     m,
                                                     nb,
                                                     realShpericalHarmonicsB.GetL(),
                                                     alpha,
                                                     beta);
            reducedOverlap1stDerivAlpha
               = this->GetReducedOverlap1stDerivativeAlpha(na,
                                                           realShpericalHarmonicsA.GetL(),
                                                           m,
                                                           nb,
                                                           realShpericalHarmonicsB.GetL(),
                                                           alpha,
                                                           beta);
            reducedOverlap1stDerivBeta
               = this->GetReducedOverlap1stDerivativeBeta(na,
                                                          realShpericalHarmonicsA.GetL(),
                                                          m,
                                                          nb,
                                                          realShpericalHarmonicsB.GetL(),
                                                          alpha,
                                                          beta);
            reducedOverlap2ndDerivAlpha
               = this->GetReducedOverlap2ndDerivativeAlpha(na,
                                                           realShpericalHarmonicsA.GetL(),
                                                           m,
                                                           nb,
                                                           realShpericalHarmonicsB.GetL(),
                                                           alpha,
                                                           beta);
            reducedOverlap2ndDerivBeta
               = this->GetReducedOverlap2ndDerivativeBeta(na,
                                                          realShpericalHarmonicsA.GetL(),
                                                          m,
                                                          nb,
                                                          realShpericalHarmonicsB.GetL(),
                                                          alpha,
                                                          beta);
            reducedOverlap2ndDerivAlphaBeta
               = this->GetReducedOverlap2ndDerivativeAlphaBeta(na,
                                                               realShpericalHarmonicsA.GetL(),
                                                               m,
                                                               nb,
                                                               realShpericalHarmonicsB.GetL(),
                                                               alpha,
                                                               beta);

            temp1 = static_cast<double>(na+nb+1)
                   *static_cast<double>(na+nb)
                   *pow(R,na+nb-1)*reducedOverlap;
            temp2 = 2.0*static_cast<double>(na+nb+1)*pow(R,na+nb)
                       *(orbitalExponentA*reducedOverlap1stDerivAlpha
                        +orbitalExponentB*reducedOverlap1stDerivBeta);
            temp3 = pow(R,na+nb+1)
                   *(pow(orbitalExponentA,2.0)*reducedOverlap2ndDerivAlpha
                    +pow(orbitalExponentB,2.0)*reducedOverlap2ndDerivBeta
                    +2.0*orbitalExponentA*orbitalExponentB*reducedOverlap2ndDerivAlphaBeta);

            pre =  pow(2.0*orbitalExponentA, na+0.5);
            pre *= pow(2.0*orbitalExponentB, nb+0.5);
            double factorials = Factorial(2*na)*Factorial(2*nb);
            pre /= sqrt(factorials);
            pre /= pow(2.0, na+nb+1.0);

            diatomicOverlap2ndDeri[valenceOrbitalA][valenceOrbitalB] = pre*(temp1+temp2+temp3);
         }
         
      }
   }

   /*
   for(int i=0;i<OrbitalType_end;i++){
      for(int j=0;j<OrbitalType_end;j++){
         this->OutputLog(boost::format("diatomicOverlap[%d][%d]=%lf\n") % i % j % diatomicOverlap[i][j]);
      }
   }
   */


}

// see (B.63) in Pople book.
void Cndo2::RotateDiatmicOverlapToSpaceFrame(double** diatomicOverlap, 
                                             double const* const* rotatingMatrix) const{
   if(diatomicOverlap==NULL){
      stringstream ss;
      ss << this->errorMessageRotDiaOverlapToSpaceFrameNullDiaMatrix;
      throw MolDSException(ss.str());
   }
   if(rotatingMatrix==NULL){
      stringstream ss;
      ss << this->errorMessageRotDiaOverlapToSpaceFrameNullRotMatrix;
      throw MolDSException(ss.str());
   }
   double** oldDiatomicOverlap = NULL;
   try{
      MallocerFreer::GetInstance()->Malloc<double>(&oldDiatomicOverlap, OrbitalType_end, OrbitalType_end);
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            oldDiatomicOverlap[i][j] = diatomicOverlap[i][j];
         }
      }
      // rotate
      for(int i=0; i<OrbitalType_end; i++){
         for(int j=0; j<OrbitalType_end; j++){
            diatomicOverlap[i][j] = 0.0;
            for(int k=0; k<OrbitalType_end; k++){
               for(int l=0; l<OrbitalType_end; l++){
                  diatomicOverlap[i][j] += oldDiatomicOverlap[k][l] 
                                          *rotatingMatrix[i][k] 
                                          *rotatingMatrix[j][l];
               }
            }
         }
      }
   }
   catch(MolDSException ex){
      MallocerFreer::GetInstance()->Free<double>(&oldDiatomicOverlap, OrbitalType_end, OrbitalType_end);
      throw ex;
   }
   MallocerFreer::GetInstance()->Free<double>(&oldDiatomicOverlap, OrbitalType_end, OrbitalType_end);

   /*
   for(int i=0;i<OrbitalType_end;i++){
      for(int j=0;j<OrbitalType_end;j++){
         this->OutputLog(boost::format("rotated diatomicOverlap[%d][%d]=%lf\n") % i % j % diatomicOverlap[i][j]);
         this->OutputLog(boost::format("rotating[%d][%d]=%lf\n") % i % j % rotating[i][j]);
      }
   }
   */

}

void Cndo2::SetOverlapElement(double** overlap, 
                              double const* const* diatomicOverlap, 
                              const Atom& atomA, 
                              const Atom& atomB) const{
   if(diatomicOverlap==NULL){
      stringstream ss;
      ss << this->errorMessageSetOverlapElementNullDiaMatrix;
      throw MolDSException(ss.str());
   }

   int firstAOIndexAtomA = atomA.GetFirstAOIndex();
   int firstAOIndexAtomB = atomB.GetFirstAOIndex();
   OrbitalType orbitalA;
   OrbitalType orbitalB;
   int mu=0;
   int nu=0;

   for(int i=0; i<atomA.GetValenceSize(); i++){
      orbitalA = atomA.GetValence(i);
      for(int j=0; j<atomB.GetValenceSize(); j++){
         orbitalB = atomB.GetValence(j);
         mu = firstAOIndexAtomA + i;      
         nu = firstAOIndexAtomB + j;      
         overlap[mu][nu] = diatomicOverlap[orbitalA][orbitalB];
         overlap[nu][mu] = diatomicOverlap[orbitalA][orbitalB];
      }
   }

}

// see (B.24) in J. A. Pople book.
double Cndo2::GetReducedOverlap(int na, int la, int m, int nb, int lb, double alpha, double beta) const{
   double value = 0.0;
   double temp = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp = Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j];
         temp *= this->GetAuxiliaryA(i, 0.5*(alpha+beta));
         temp *= this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         value += temp;
      }
   }
   value *= this->GetAuxiliaryD(la, lb, m);

   return value;
}

// see (B.30) in J. A. Pople book.
double Cndo2::GetReducedOverlap(int na, int nb, double alpha, double beta) const{
   double value = 0.0;
   double temp = 0.0;

   for(int k=0; k<=na+nb; k++){
      temp = Cndo2::ReducedOverlapParameters::Z[na][nb][k];
      temp *= this->GetAuxiliaryA(k, 0.5*(alpha+beta));
      temp *= this->GetAuxiliaryB(na+nb-k, 0.5*(alpha-beta));
      value += temp;
   }
   value *= 0.5;
   return value;
}

// First derivative of (B.24) in J. A. Pople book.
// This derivative is carried out by alpha.
double Cndo2::GetReducedOverlap1stDerivativeAlpha(int na, 
                                                  int la, 
                                                  int m, 
                                                  int nb, 
                                                  int lb, 
                                                  double alpha, 
                                                  double beta) const{
   double value = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp1 = this->GetAuxiliaryA1stDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         temp2 = this->GetAuxiliaryA(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB1stDerivative(j, 0.5*(alpha-beta));
         value += Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j]*(temp1 + temp2);
      }
   }
   value *= 0.5*this->GetAuxiliaryD(la, lb, m);

   return value;
}

// First derivative of (B.24) in J. A. Pople book.
// This derivative is carried out by Beta.
double Cndo2::GetReducedOverlap1stDerivativeBeta(int na, 
                                                 int la, 
                                                 int m, 
                                                 int nb, 
                                                 int lb, 
                                                 double alpha, 
                                                 double beta) const{
   double value = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp1 = this->GetAuxiliaryA1stDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         temp2 = this->GetAuxiliaryA(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB1stDerivative(j, 0.5*(alpha-beta));
         value += Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j]*(temp1 - temp2);
      }
   }
   value *= 0.5*this->GetAuxiliaryD(la, lb, m);

   return value;
}

// Second derivative of (B.24) in J. A. Pople book.
// This derivative is carried out by alpha twice.
double Cndo2::GetReducedOverlap2ndDerivativeAlpha(int na, 
                                                  int la, 
                                                  int m, 
                                                  int nb, 
                                                  int lb, 
                                                  double alpha, 
                                                  double beta) const{
   double value = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;
   double temp3 = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp1 = this->GetAuxiliaryA2ndDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         temp2 = this->GetAuxiliaryA(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB2ndDerivative(j, 0.5*(alpha-beta));
         temp3 = this->GetAuxiliaryA1stDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB1stDerivative(j, 0.5*(alpha-beta));
         value += Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j]*(temp1 + temp2 + 2.0*temp3);
      }
   }
   value *= 0.25*this->GetAuxiliaryD(la, lb, m);

   return value;
}

// Second derivative of (B.24) in J. A. Pople book.
// This derivative is carried out by beta twice.
double Cndo2::GetReducedOverlap2ndDerivativeBeta(int na, 
                                                    int la, 
                                                    int m, 
                                                    int nb, 
                                                    int lb, 
                                                    double alpha, 
                                                    double beta) const{
   double value = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;
   double temp3 = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp1 = this->GetAuxiliaryA2ndDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         temp2 = this->GetAuxiliaryA(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB2ndDerivative(j, 0.5*(alpha-beta));
         temp3 = this->GetAuxiliaryA1stDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB1stDerivative(j, 0.5*(alpha-beta));
         value += Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j]*(temp1 + temp2 - 2.0*temp3);
      }
   }
   value *= 0.25*this->GetAuxiliaryD(la, lb, m);

   return value;
}

// Second derivative of (B.24) in J. A. Pople book.
// This derivative is carried out by alpha and beta.
double Cndo2::GetReducedOverlap2ndDerivativeAlphaBeta(int na, 
                                                      int la, 
                                                      int m, 
                                                      int nb, 
                                                      int lb, 
                                                      double alpha, 
                                                      double beta) const{
   double value = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;
   int I = 2*ShellType_end+1;
   int J = 2*ShellType_end+1;

   for(int i=0; i<I; i++){
      for(int j=0; j<J; j++){
         temp1 = this->GetAuxiliaryA2ndDerivative(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB(j, 0.5*(alpha-beta));
         temp2 = this->GetAuxiliaryA(i, 0.5*(alpha+beta))
                *this->GetAuxiliaryB2ndDerivative(j, 0.5*(alpha-beta));
         value += Cndo2::ReducedOverlapParameters::Y[na][nb][la][lb][m][i][j]*(temp1 - temp2);
      }
   }
   value *= 0.25*this->GetAuxiliaryD(la, lb, m);

   return value;
}

// see (B.22) in J. A. Pople book.
double Cndo2::GetAuxiliaryA(int k, double rho) const{
   double value = 0.0;
   double temp = 0.0;

   value = exp(-1.0*rho)*static_cast<double>(Factorial(k));
   for(int mu=1; mu<=k+1; mu++){
      temp += pow(rho,-1.0*mu)/static_cast<double>(Factorial(k-mu+1));
   }
   value *= temp;

   return value;
}

// First derivative of (B.22) in J. A. Pople book.
double Cndo2::GetAuxiliaryA1stDerivative(int k, double rho) const{
   return -1.0*this->GetAuxiliaryA(k+1, rho);
}

// Second derivative of (B.22) in J. A. Pople book.
double Cndo2::GetAuxiliaryA2ndDerivative(int k, double rho) const{
   return this->GetAuxiliaryA(k+2, rho);
}

// see (B.23) in J. A. Pople book.
double Cndo2::GetAuxiliaryB(int k, double rho) const{
   double value = 0.0;
   double pre1 = 0.0;
   double pre2 = 0.0;
   double temp1 = 0.0;
   double temp2 = 0.0;

   if(fabs(rho)>0){
      pre1 = -1.0*exp(-1.0*rho);
      pre2 = -1.0*exp(rho);
      
      for(int mu=1; mu<=k+1; mu++){
         temp1 += pow(rho,-1.0*mu)  *static_cast<double>(Factorial(k)/Factorial(k-mu+1)) ;
         temp2 += pow(rho,-1.0*mu)  *static_cast<double>(Factorial(k)/Factorial(k-mu+1)) *pow(-1.0,k-mu);
      }
      value = pre1*temp1 + pre2*temp2;
   }
   else{
      if(k%2 == 0){
         value = 2.0/(1.0+static_cast<double>(k));
      }
      else{
         value = 0;
      }
   }

   return value;
}

// First derivative of (B.23) in J. A. Pople book.
double Cndo2::GetAuxiliaryB1stDerivative(int k, double rho) const{
   return -1.0*this->GetAuxiliaryB(k+1, rho);
}

// Second derivative of (B.23) in J. A. Pople book.
double Cndo2::GetAuxiliaryB2ndDerivative(int k, double rho) const{
   return this->GetAuxiliaryB(k+2, rho);
}

// see (B.16) in J. A. Pople book.
double Cndo2::GetAuxiliaryD(int la, int lb, int m) const{
   string errorMessageAuxiliaryDNegativeM = "Error in cndo::Cndo2::GetAuxiliaryD: m<0\n";
   double value = 0.0;

   if(m<0){
      stringstream ss;
      ss << errorMessageAuxiliaryDNegativeM;
      throw MolDSException(ss.str());
   }

   double pre = pow(Factorial(m+1)/8.0, 2.0);
   double termA = ( (2.0*la+1.0)*Factorial(la-m) ) / ( 2.0*Factorial(la+m) );
   double termB = ( (2.0*lb+1.0)*Factorial(lb-m) ) / ( 2.0*Factorial(lb+m) );
   value = pre*sqrt(termA)*sqrt(termB);
   //this->OutputLog(boost::format("pre=%lf, termA=%lf, termB=%lf\n") % pre % termA % termB);
   
   return value;
}
}



