/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.c.pre;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.c.pre.directive.CpreDefine;
import net.morilib.c.pre.directive.CpreDirective;
import net.morilib.c.pre.directive.CpreElif;
import net.morilib.c.pre.directive.CpreElse;
import net.morilib.c.pre.directive.CpreEndif;
import net.morilib.c.pre.directive.CpreError;
import net.morilib.c.pre.directive.CpreIf;
import net.morilib.c.pre.directive.CpreIfState;
import net.morilib.c.pre.directive.CpreIfdef;
import net.morilib.c.pre.directive.CpreIfndef;
import net.morilib.c.pre.directive.CpreNull;
import net.morilib.c.pre.directive.CprePragma;

public class CPreprocessorReader extends Reader {

	private static enum S0 { INI, SH1, EX1, EX2, AR1, AR2 };
	private static enum S1 { INI, SH1, SH2, DQ1, DQ2 };

	static final Map<String, CpreDirective> DIRS;
	static final Pattern PI1 = Pattern.compile("#include *<(.+)>");
	static final Pattern PI2 = Pattern.compile("#include *\"(.+)\"");

	static {
		Map<String, CpreDirective> d;

		d = new HashMap<String, CpreDirective>();
		d.put("#", new CpreNull());
		d.put("#define", new CpreDefine());
		d.put("#elif", new CpreElif());
		d.put("#else", new CpreElse());
		d.put("#endif", new CpreEndif());
		d.put("#error", new CpreError());
		d.put("#if", new CpreIf());
		d.put("#ifdef", new CpreIfdef());
		d.put("#ifndef", new CpreIfndef());
		d.put("#pragma", new CprePragma());
		DIRS = Collections.unmodifiableMap(d);
	}

	private Stack<LineNumberReader> rd =
		new Stack<LineNumberReader>();
	private Stack<CpreIfState> ifstat = new Stack<CpreIfState>();
	private Stack<String> fns = new Stack<String>();
	private CpreMacros macros;
	private String line = null;
	private int ptr = 0;

	/**
	 * 
	 * @param fname
	 * @param in
	 */
	public CPreprocessorReader(String fname, Reader in) {
		rd.push(new LineNumberReader(in));
		fns.push(fname != null ? fname : "<no name>");
		ifstat.add(CpreIfState.TOP);

		// initialize macros
		macros = new CpreMacros();
		macros.define("__STDC__", "0");
	}

	static String substitute(Reader rd,
			CpreMacros macros) throws IOException {
		List<String> l0 = null;
		StringBuffer b0, b1 = null, b2 = null;
		String s, nm = null, s1;
		S0 st = S0.INI;
		int c;

		b0 = new StringBuffer();
		while(true) {
			c = rd.read();
			switch(st) {
			case INI:
				if(CpreUtils.isCMacroIdentifierStart(c)) {
					b1 = new StringBuffer().appendCodePoint(c);
					st = S0.EX1;
				} else if(c == '#') {
					st = S0.SH1;
				} else if(c < 0) {
					return b0.toString();
				} else {
					b0.appendCodePoint(c);
				}
				break;
			case SH1:
				if(CpreUtils.isCMacroIdentifierStart(c)) {
					b1 = new StringBuffer().appendCodePoint(c);
					st = S0.EX2;
				} else if(c == '#') {
					b0.append('#').append('#');
					st = S0.INI;
				} else if(c < 0) {
					return b0.append('#').toString();
				} else {
					b0.append('#').appendCodePoint(c);
					st = S0.INI;
				}
				break;
			case EX1:
				if(CpreUtils.isCMacroIdentifierPart(c)) {
					b1.append((char)c);
				} else if(c == '(') {
					nm = b1.toString();
					l0 = new ArrayList<String>();
					b2 = new StringBuffer(b1).append('(');
					b1 = new StringBuffer();
					st = S0.AR1;
				} else {
					s = macros.substitute(b1.toString());
					b0.append(s != null ? s : b1);
					if(c < 0)  return b0.toString();
					if(c == '#') {
						st = S0.SH1;
					} else {
						b0.appendCodePoint(c);
						st = S0.INI;
					}
				}
				break;
			case EX2:
				if(CpreUtils.isCMacroIdentifierPart(c)) {
					b1.append((char)c);
				} else if(c == '(') {
					nm = b1.toString();
					l0 = new ArrayList<String>();
					b2 = new StringBuffer(b1).append('(');
					b1 = new StringBuffer();
					st = S0.AR1;
				} else {
					b0.append('\"');
					s = macros.substitute(b1.toString());
					b0.append(s != null ? s : b1);
					if(c < 0)  return b0.toString();
					if(c == '#') {
						st = S0.SH1;
					} else {
						b0.appendCodePoint(c).append('\"');
						st = S0.INI;
					}
				}
				break;
			case AR1:
				if(c == ')') {
					l0.add(b1.toString().trim());
					s1 = macros.substitute(nm, l0);
					b0.append(s1 != null ? s1 : b2.append(')'));
					b1 = new StringBuffer();
					st = S0.INI;
				} else if(c == ',') {
					l0.add(b1.toString().trim());
					b1 = new StringBuffer();
					b2.appendCodePoint(c);
				} else if(c < 0) {
					throw new CpreSyntaxException();
				} else {
					b1.appendCodePoint(c);
					b2.appendCodePoint(c);
				}
				break;
			case AR2:
				if(c == ')') {
					l0.add(b1.toString().trim());
					b0.append('\"');
					s1 = macros.substitute(nm, l0);
					b0.append(s1 != null ? s1 : b2.append(')'));
					b0.append('\"');
					b1 = new StringBuffer();
					st = S0.INI;
				} else if(c == ',') {
					l0.add(b1.toString().trim());
					b1 = new StringBuffer();
					b2.appendCodePoint(c);
				} else if(c < 0) {
					throw new CpreSyntaxException();
				} else {
					b1.appendCodePoint(c);
					b2.appendCodePoint(c);
				}
				break;
			}
		}
	}

	static String substitutep2(Reader rd) throws IOException {
		StringBuffer b0, b1 = null;
		S1 st = S1.INI;
		int c, d;

		b0 = new StringBuffer();
		while(true) {
			c = rd.read();
			switch(st) {
			case INI:
				if(c == '#') {
					st = S1.SH1;
				} else if(c == '"') {
					b0.appendCodePoint(c);
					st = S1.DQ1;
				} else if(c < 0) {
					return b0.toString();
				} else {
					b0.appendCodePoint(c);
				}
				break;
			case SH1:
				if(c == '#') {
					b1 = new StringBuffer();
					st = S1.SH2;
				} else if(c == '"') {
					b0.append('#').append('"');
					st = S1.DQ1;
				} else if(c < 0) {
					return b0.append('#').toString();
				} else {
					b0.append('#').appendCodePoint(c);
					st = S1.INI;
				}
				break;
			case SH2:
				if(c == ' ' || c == '\t') {
					b1.append((char)c);
				} else if(c < 0) {
					b0.append('#').append('#').append(b1);
					return b0.toString();
				} else {
					for(int i = b0.length() - 1; i >= 0; i--) {
						if((d = b0.charAt(i)) == ' ' || d == '\t') {
							b0.deleteCharAt(i);
						} else {
							break;
						}
					}
					b0.appendCodePoint(c);
					st = S1.INI;
				}
				break;
			case DQ1:
				if(c == '"') {
					st = S1.DQ2;
				} else if(c < 0) {
					return b0.toString();
				}
				b0.appendCodePoint(c);
				break;
			case DQ2:
				if(c == ' ' || c == '\t') {
					b0.append((char)c);
				} else if(c == '"') {
					for(int i = b0.length() - 1; i >= 0; i--) {
						if((d = b0.charAt(i)) == ' ' || d == '\t') {
							b0.deleteCharAt(i);
						} else {
							break;
						}
					}
					b0.deleteCharAt(b0.length() - 1);
					st = S1.DQ1;
				} else if(c < 0) {
					return b0.toString();
				} else {
					b0.appendCodePoint(c);
					st = S1.INI;
				}
				break;
			}
		}
	}

	/**
	 * 
	 * @param s
	 * @param macros
	 * @return
	 */
	public static String substitute(String s, CpreMacros macros) {
		StringReader r;
		String t = null;

		try {
			while(!s.equals(t)) {
				r = new StringReader(t = s);
				s = substitute(r, macros);  r.close();
				r = new StringReader(s);
				s = substitutep2(r);  r.close();
			}
			return s;
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
	}

	String readLine0() throws IOException {
		String s, t;

		while((s = rd.peek().readLine()) == null) {
			if(rd.size() > 1) {
				rd.pop().close();
			} else {
				rd.pop();
				return null;
			}
		}

		if(!s.endsWith("\\")) {
			return s;
		} else if((t = readLine0()) == null) {
			return s;
		} else {
			return s + "\n" + t;
		}
	}

	String[] getincpath() {
		String[] r;
		String ps = System.getProperty("path.separator");
		String s0 = System.getenv("INCLUDE");

		s0 = s0 != null ? s0 : "";
		r  = s0.split(ps);
		return r;
	}

	void openinc1(File fn) throws IOException {
		rd.push(new LineNumberReader(new InputStreamReader(
				new FileInputStream(fn))));
		fns.push(fn.toString());
	}

	boolean openinc(String fn) throws IOException {
		String[] pt = getincpath();
		File f;

		for(String d : pt) {
			if((f = new File(d, fn)).isFile()) {
				openinc1(f);
				return true;
			}
		}
		return false;
	}

	String[] split2(String s) {
		String[] r = new String[2];

		r[1] = "";
		for(int i = 0; i < s.length(); i++) {
			if(s.charAt(i) == ' ' || s.charAt(i) == '\t') {
				r[0] = s.substring(0, i);
				for(int j = i; j < s.length(); j++) {
					if(s.charAt(j) != ' ' && s.charAt(j) != '\t') {
						r[1] = s.substring(j);
						return r;
					}
				}
				return r;
			}
		}
		r[0] = s;
		return r;
	}

	String processLine() throws IOException {
		String ps = System.getProperty("path.separator");
		String s = readLine0(), fn;
		CpreDirective d;
		CpreIfState ifs;
		String[] s2;
		Matcher m;
		File f;

		if(s == null)  return null;
		s2 = split2(s);

		// #include
		if((m = PI1.matcher(s)).matches()) {
			fn = m.group(1);
			if(ps.equals(":")) {
				// UNIX
				f = new File("/usr/include", fn);
				if(f.isFile()) {
					openinc1(f);
				} else {
					throw new CpreSyntaxException();
				}
			} else if(!openinc(m.group(1))) {
				throw new CpreSyntaxException();
			}
			return processLine();
		} else if((m = PI2.matcher(s)).matches()) {
			fn = m.group(1);
			if((f = new File(fn)).isFile()) {
				openinc1(f);
			} else if(!openinc(m.group(1))) {
				throw new CpreSyntaxException();
			}
			return processLine();
		}

		// other directives
		if((d = DIRS.get(s2[0])) != null) {
			switch(ifs = d.execute(macros, s2[1], ifstat.peek())) {
			case BEGIN:
				ifstat.push(CpreIfState.PROCESSING);  return "";
			case READY:
				ifstat.push(CpreIfState.UNPROCESSED);  return "";
			case UNPROCESSED:
			case PROCESSING:
			case PROCESSED:
			case PROCESSING_ELSE:
			case DONE:
				ifstat.pop();  ifstat.push(ifs);  return "";
			case RETURN:
				if(ifstat.size() <= 1) {
					throw new CpreSyntaxException();
				}
				ifstat.pop();  return "";
			case STAY:  return "";
			case TOP:  throw new RuntimeException();
			}
		}

		// other lines
		return ifstat.peek().isProcessing() ?
				substitute(s, macros) : "";
	}

	/**
	 * 
	 * @return
	 * @throws IOException
	 */
	public String readLine() throws IOException {
		String r = line;

		ptr = 0;
		if(line != null) {
			line = null;
		} else {
			while(r == null && rd.size() > 0)  r = processLine();
		}

		if(r == null && ifstat.size() > 1) {
			throw new CpreSyntaxException();
		}
		return r;
	}

	/**
	 * 
	 * @return
	 */
	public int getLineNumber() {
		return rd.peek().getLineNumber();
	}

	@Override
	public int read() throws IOException {
		if(line == null)  ptr = 0;
		while(line == null)  line = processLine();
		if(ptr < line.length()) {
			return line.charAt(ptr++);
		} else {
			line = null;
			return '\n';
		}
	}

	@Override
	public int read(char[] cbuf, int off, int len) throws IOException {
		int i = off;

		for(; i < off + len; i++) {
			cbuf[i] = (char)read();
			if(!ready())  break;
		}
		return i - off;
	}

	@Override
	public void close() throws IOException {
		while(!rd.isEmpty())  rd.pop().close();
	}

}
