<?php
// $Id: user.php,v 1.4 2008/10/26 04:00:40 minahito Exp $
//  ------------------------------------------------------------------------ //
//                XOOPS - PHP Content Management System                      //
//                    Copyright (c) 2000 XOOPS.org                           //
//                       <http://www.xoops.org/>                             //
//  ------------------------------------------------------------------------ //
//  This program is free software; you can redistribute it and/or modify     //
//  it under the terms of the GNU General Public License as published by     //
//  the Free Software Foundation; either version 2 of the License, or        //
//  (at your option) any later version.                                      //
//                                                                           //
//  You may not change or alter any portion of this comment or credits       //
//  of supporting developers from this source code or any supporting         //
//  source code which is considered copyrighted (c) material of the          //
//  original comment or credit authors.                                      //
//                                                                           //
//  This program is distributed in the hope that it will be useful,          //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of           //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            //
//  GNU General Public License for more details.                             //
//                                                                           //
//  You should have received a copy of the GNU General Public License        //
//  along with this program; if not, write to the Free Software              //
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA //
//  ------------------------------------------------------------------------ //
// Author: Kazumi Ono (AKA onokazu)                                          //
// URL: http://www.myweb.ne.jp/, http://www.xoops.org/, http://jp.xoops.org/ //
// Project: The XOOPS Project                                                //
// ------------------------------------------------------------------------- //
if (!defined('XOOPS_ROOT_PATH')) {
	exit();
}
/**
 * Class for users 
 * @author Kazumi Ono <onokazu@xoops.org>
 * @copyright copyright (c) 2000-2003 XOOPS.org
 * @package kernel
 */
class XoopsUser extends XoopsObject
{
    /**
     * Array of groups that user belongs to 
     * @var array
	 * @access private
     */
    var $_groups = array();
    /**
     * @var bool is the user admin? 
	 * @access private
     */
    var $_isAdmin = null;

    /**
     * constructor 
     * @param array $id Array of key-value-pairs to be assigned to the user. (for backward compatibility only)
     * @param int $id ID of the user to be loaded from the database.
     */
    function __construct($id = null)
    {
        $this->initVar('uid', XOBJ_DTYPE_INT, null, false);
        $this->initVar('name', XOBJ_DTYPE_TXTBOX, null, false, 60);
        $this->initVar('uname', XOBJ_DTYPE_TXTBOX, null, true, 25);
        $this->initVar('email', XOBJ_DTYPE_TXTBOX, null, true, 60);
        $this->initVar('user_regdate', XOBJ_DTYPE_INT, null, false);
        $this->initVar('user_viewemail', XOBJ_DTYPE_INT, 0, false);
        $this->initVar('actkey', XOBJ_DTYPE_OTHER, null, false);
        $this->initVar('pass', XOBJ_DTYPE_TXTBOX, null, false, 32);
        $this->initVar('level', XOBJ_DTYPE_INT, 0, false);
        $this->initVar('timezone_offset', XOBJ_DTYPE_OTHER, null, false);
        $this->initVar('last_login', XOBJ_DTYPE_INT, 0, false);
        $this->initVar('user_mailok', XOBJ_DTYPE_INT, 1, false);
    }
    
	/**
	 * set the groups for the user
	 * 
	 * @param array $groupsArr Array of groups that user belongs to
	 */
	function setGroups($groupsArr)
	{
		if (is_array($groupsArr)) {
			$this->_groups =& $groupsArr;
		}
	}
    /**
     * get the groups that the user belongs to
	 * 
	 * @param $bReget When this is true, this object gets group informations from DB again.
	 *                This is a special method for the BASE(CMS core) functions, you should
	 *                not use this proactivity.
	 * @return array array of groups 
     */
    function getGroups($bReget = false)
    {
    	if ($bReget) {
    		unset($this->_groups);
    	}
    	
        if (empty($this->_groups)) {
            $member_handler =& xoops_gethandler('member');
            $this->_groups = $member_handler->getGroupsByUser($this->getVar('uid'));
        }
        return $this->_groups;
    }
    
    function getNumGroups()
	{
		if (empty($this->_groups)) {
			$this->getGroups();
		}
		return count($this->_groups);
	}
    
    /**
     * Is the user admin ?
     *
     * This method will return true if this user has admin rights for the specified module.<br />
     * - If you don't specify any module ID, the current module will be checked.<br />
     * - If you set the module_id to -1, it will return true if the user has admin rights for at least one module
     *
     * @param int $module_id check if user is admin of this module
	 * @return bool is the user admin of that module?
     */
    function isAdmin( $module_id = null ) {
		if ( is_null( $module_id ) ) {
			$module_id = is_object($module =& XCube_Root::getSingleton()->mContext->mXoopsModule) ? $module->getVar( 'mid', 'n' ) : 1;
		} elseif ( intval($module_id) < 1 ) {
			$module_id = 0;
		}
        $moduleperm_handler =& xoops_gethandler('groupperm');
        return $moduleperm_handler->checkRight('module_admin', $module_id, $this->getGroups());
    }
    /**
     * is the user activated?
     * @return bool
     */
    function isActive()
    {
        if ($this->getVar('level') == 0) {
            return false;
        }
        return true;
    }
}

/**
 * Class that represents a guest user
 * @author Kazumi Ono <onokazu@xoops.org>
 * @copyright copyright (c) 2000-2003 XOOPS.org
 * @package kernel
 */
class XoopsGuestUser extends XoopsUser
{
	function getGroups()
	{
		return XOOPS_GROUP_ANONYMOUS;
	}
}


/**
* XOOPS user handler class.  
* This class is responsible for providing data access mechanisms to the data source 
* of XOOPS user class objects.
*
* @author  Kazumi Ono <onokazu@xoops.org>
* @copyright copyright (c) 2000-2003 XOOPS.org
* @package kernel
*/
class XoopsUserHandler extends XoopsObjectHandler
{
    /**
     * create a new user
     * 
     * @param bool $isNew flag the new objects as "new"?
     * @return object XoopsUser
     */
    function &create($isNew = true)
    {
        $user = new XoopsUser();
        if ($isNew) {
            $user->setNew();
        }
        return $user;
    }

    /**
     * retrieve a user
     * 
     * @param int $id UID of the user
     * @return mixed reference to the {@link XoopsUser} object, FALSE if failed
     */
    function &get($id)
    {
        $ret = false;
        if (intval($id) > 0) {
            $sql = 'SELECT * FROM '.$this->db->prefix('users').' WHERE uid='.$id;
            if ($result = $this->db->query($sql)) {
                $numrows = $this->db->getRowsNum($result);
                if ($numrows == 1) {
                        $user = new XoopsUser();
                    $user->assignVars($this->db->fetchArray($result));
                        $ret =& $user;
                }
            }
        }
        return $ret;
    }

    /**
     * insert a new user in the database
     * 
     * @param object $user reference to the {@link XoopsUser} object
     * @param bool $force
     * @return bool FALSE if failed, TRUE if already present and unchanged or successful
     */
    function insert(&$user, $force = false)
    {
        if (strtolower(get_class($user)) != 'xoopsuser') {
            return false;
        }
        if (!$user->isDirty()) {
            return true;
        }
        if (!$user->cleanVars()) {
            return false;
        }
        foreach ($user->cleanVars as $k => $v) {
            ${$k} = $v;
        }
        // RMV-NOTIFY
        // Added two fields, notify_method, notify_mode
        if ($user->isNew()) {
            $uid = $this->db->genId('users_uid_seq');
            $sql = sprintf("INSERT INTO %s (uid, uname, name, email, user_regdate, user_viewemail, actkey, pass, level, timezone_offset, last_login, user_mailok) VALUES (%u, %s, %s, %s, %u, %u, %s, %s, %u, %.2f, %u, %u)", $this->db->prefix('users'), $uid, $this->db->quoteString($uname), $this->db->quoteString($name), $this->db->quoteString($email), time(), $user_viewemail, $this->db->quoteString($actkey), $this->db->quoteString($pass), $level, $timezone_offset, 0, $user_mailok);
        } else {
            $sql = sprintf("UPDATE %s SET uname = %s, name = %s, email = %s, user_viewemail = %u, pass = %s, level= %u, timezone_offset = %.2f, last_login = %u, user_mailok = %u WHERE uid = %u", $this->db->prefix('users'), $this->db->quoteString($uname), $this->db->quoteString($name), $this->db->quoteString($email), $user_viewemail, $this->db->quoteString($pass), $level, $timezone_offset, $last_login, $user_mailok, $uid);
        }
        if (false != $force) {
            $result = $this->db->queryF($sql);
        } else {
            $result = $this->db->query($sql);
        }
        if (!$result) {
            return false;
        }
        if (empty($uid)) {
            $uid = $this->db->getInsertId();
        }
        $user->assignVar('uid', $uid);
        return true;
    }

    /**
     * delete a user from the database
     * 
     * @param object $user reference to the user to delete
     * @param bool $force
     * @return bool FALSE if failed.
     */
    function delete(&$user, $force = false)
    {
        if (strtolower(get_class($user)) != 'xoopsuser') {
            return false;
        }
        $sql = sprintf("DELETE FROM %s WHERE uid = %u", $this->db->prefix("users"), $user->getVar('uid'));
        if (false != $force) {
            $result = $this->db->queryF($sql);
        } else {
            $result = $this->db->query($sql);
        }
        if (!$result) {
            return false;
        }
        return true;
    }

    /**
     * retrieve users from the database
     * 
     * @param object $criteria {@link CriteriaElement} conditions to be met
     * @param bool $id_as_key use the UID as key for the array?
     * @return array array of {@link XoopsUser} objects
     */
    function &getObjects($criteria = null, $id_as_key = false)
    {
        $ret = array();
        $limit = $start = 0;
        $sql = 'SELECT * FROM '.$this->db->prefix('users');
        if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement')) {
            $sql .= ' '.$criteria->renderWhere();
            if ($criteria->getSort() != '') {
                $sql .= ' ORDER BY '.$criteria->getSort().' '.$criteria->getOrder();
            }
            $limit = $criteria->getLimit();
            $start = $criteria->getStart();
        }
        $result = $this->db->query($sql, $limit, $start);
        if (!$result) {
            return $ret;
        }
        while ($myrow = $this->db->fetchArray($result)) {
            $user = new XoopsUser();
            $user->assignVars($myrow);
            if (!$id_as_key) {
                $ret[] =& $user;
            } else {
                $ret[$myrow['uid']] =& $user;
            }
            unset($user);
        }
        return $ret;
    }
    
	/**
	 This method is called from pmlite.php. Wmm..
	 Type:expand (no using criteria).
	 @author minahito
	 */
    function &getObjectsByLevel($level=0)
    {
		$ret=array();
		$level=intval($level);
		$result = $this->db->query("SELECT * FROM ".$this->db->prefix("users")." WHERE level > $level ORDER BY uname");
		if(!$result)
			return $ret;

		while($myrow=$this->db->fetchArray($result)) {
			$user=new XoopsUser();
			$user->assignVars($myrow);
			$ret[]=&$user;
			unset($user);
		}
		
		return $ret;
	}

    /**
     * count users matching a condition
     * 
     * @param object $criteria {@link CriteriaElement} to match
     * @return int count of users
     */
    function getCount($criteria = null)
    {
        $sql = 'SELECT COUNT(*) FROM '.$this->db->prefix('users');
        if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement')) {
            $sql .= ' '.$criteria->renderWhere();
        }
        $result = $this->db->query($sql);
        if (!$result) {
            return 0;
        }
        list($count) = $this->db->fetchRow($result);
        return $count;
    }

    /**
     * delete users matching a set of conditions
     * 
     * @param object $criteria {@link CriteriaElement} 
     * @return bool FALSE if deletion failed
     */
    function deleteAll($criteria = null)
    {
        $sql = 'DELETE FROM '.$this->db->prefix('users');
        if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement')) {
            $sql .= ' '.$criteria->renderWhere();
        }
        if (!$result = $this->db->query($sql)) {
            return false;
        }
        return true;
    }

	/**
     * Change a value for users with a certain criteria
     * 
     * @param   string  $fieldname  Name of the field
     * @param   string  $fieldvalue Value to write
     * @param   object  $criteria   {@link CriteriaElement} 
     * 
     * @return  bool
     **/
    function updateAll($fieldname, $fieldvalue, $criteria = null)
    {
        $set_clause = is_numeric($fieldvalue) ? $fieldname.' = '.$fieldvalue : $fieldname.' = '.$this->db->quoteString($fieldvalue);
        $sql = 'UPDATE '.$this->db->prefix('users').' SET '.$set_clause;
        if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement')) {
            $sql .= ' '.$criteria->renderWhere();
        }
        if (!$result = $this->db->query($sql)) {
            return false;
        }
        return true;
    }
}
?>