/* javanet.c - Common internal functions for the java.net package
   Copyright (C) 1998, 2002, 2004, 2005, 2006  Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

/* do not move; needed here because of some macro definitions */
#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <jni.h>
#include <jcl.h>

#include "cpnative.h"
#include "cpnet.h"

#include "javanet.h"

#ifndef WITHOUT_NETWORK
/* Need to have some value for SO_TIMEOUT */
#ifndef SO_TIMEOUT
#ifndef SO_RCVTIMEO
#warning Neither SO_TIMEOUT or SO_RCVTIMEO are defined!
#warning This will cause all get/setOption calls with that value to throw an exception
#else
#define SO_TIMEOUT SO_RCVTIMEO
#endif /* not SO_RCVTIMEO */
#endif /* not SO_TIMEOUT */
#endif /* WITHOUT_NETWORK */

/*************************************************************************/

/*
 * Sets an integer field in the specified object.
 */
static void
_javanet_set_int_field (JNIEnv * env, jobject obj,
			const char *class, const char *field, int val)
{
  jclass cls;
  jfieldID fid;

  cls = (*env)->FindClass (env, class);
  if (cls == NULL)
    return;

  fid = (*env)->GetFieldID (env, cls, field, "I");
  if (fid == NULL)
    return;

  (*env)->SetIntField (env, obj, fid, val);

  return;
}

/*************************************************************************/

/*
 * Returns the value of the specified integer instance variable field or
 * -1 if an error occurs.
 */
int
_javanet_get_int_field (JNIEnv * env, jobject obj, const char *field)
{
  jclass cls = 0;
  jfieldID fid;
  int fd;

  DBG ("_javanet_get_int_field(): Entered _javanet_get_int_field\n");

  cls = (*env)->GetObjectClass (env, obj);
  if (cls == NULL)
    return -1;

  fid = (*env)->GetFieldID (env, cls, field, "I");
  if (fid == NULL)
    return -1;
  DBG ("_javanet_get_int_field(): Found field id\n");

  fd = (*env)->GetIntField (env, obj, fid);

  return fd;
}

/*************************************************************************/

/*
 * Creates a FileDescriptor object in the parent class.  It is not used
 * by this implementation, but the docs list it as a variable, so we
 * need to include it.
 */
static void
_javanet_create_localfd (JNIEnv * env, jobject this, jboolean stream)
{
  jclass this_cls, fd_cls;
  jfieldID fid;
  jmethodID mid;
  jobject fd_obj;

  DBG ("_javanet_create_localfd(): Entered _javanet_create_localfd\n");

  /* Look up the fd field */
  if (stream)
    this_cls = (*env)->FindClass(env, "java/net/SocketImpl");
  else
    this_cls = (*env)->FindClass(env, "java/net/DatagramSocketImpl");
  if (this_cls == NULL)
    return;

  fid = (*env)->GetFieldID (env, this_cls, "fd", "Ljava/io/FileDescriptor;");
  if (fid == NULL)
    return;

  DBG ("_javanet_create_localfd(): Found fd variable\n");

  /* Create a FileDescriptor */
  fd_cls = (*env)->FindClass (env, "java/io/FileDescriptor");
  if (fd_cls == NULL)
    return;

  DBG ("_javanet_create_localfd(): Found FileDescriptor class\n");

  mid = (*env)->GetMethodID (env, fd_cls, "<init>", "()V");
  if (mid == NULL)
    return;

  DBG ("_javanet_create_localfd(): Found FileDescriptor constructor\n");

  fd_obj = (*env)->NewObject (env, fd_cls, mid);
  if (fd_obj == NULL)
    return;

  DBG ("_javanet_create_localfd(): Created FileDescriptor\n");

  /* Now set the pointer to the new FileDescriptor */
  (*env)->SetObjectField (env, this, fid, fd_obj);
  DBG ("_javanet_create_localfd(): Set fd field\n");

  return;
}

/*************************************************************************/

/*
 * Returns a Boolean object with the specfied value
 */
static jobject
_javanet_create_boolean (JNIEnv * env, jboolean val)
{
  jclass cls;
  jmethodID mid;
  jobject obj;

  cls = (*env)->FindClass (env, "java/lang/Boolean");
  if (cls == NULL)
    return NULL;

  mid = (*env)->GetMethodID (env, cls, "<init>", "(Z)V");
  if (mid == NULL)
    return NULL;

  obj = (*env)->NewObject (env, cls, mid, val);
  if (obj == NULL)
    return NULL;

  return obj;
}

/*************************************************************************/

/*
 * Returns an Integer object with the specfied value
 */
static jobject
_javanet_create_integer (JNIEnv * env, jint val)
{
  jclass cls;
  jmethodID mid;
  jobject obj;

  cls = (*env)->FindClass (env, "java/lang/Integer");
  if (cls == NULL)
    return NULL;

  mid = (*env)->GetMethodID (env, cls, "<init>", "(I)V");
  if (mid == NULL)
    return NULL;

  obj = (*env)->NewObject (env, cls, mid, val);
  if (obj == NULL)
    return NULL;

  return obj;
}

/*************************************************************************/

/*
 * Builds an InetAddress object from a 32 bit address in host byte order
 */
jobject
_javanet_create_inetaddress (JNIEnv * env, cpnet_address *netaddr)
{
#ifndef WITHOUT_NETWORK
  jbyte octets[4];
  char buf[64];
  jclass ia_cls;
  jmethodID mid;
  jstring ip_str;
  jobject ia;

  /* Build a string IP address */
  cpnet_IPV4AddressToBytes(netaddr, octets);
  sprintf (buf, "%d.%d.%d.%d", (int) (unsigned char)octets[0], (int)(unsigned char)octets[1], (int)(unsigned char)octets[2], (int)(unsigned char)octets[3]);
  DBG ("_javanet_create_inetaddress(): Created ip addr string\n");

  /* Get an InetAddress object for this IP */
  ia_cls = (*env)->FindClass (env, "java/net/InetAddress");
  if (ia_cls == NULL)
    {
      return NULL;
    }

  DBG ("_javanet_create_inetaddress(): Found InetAddress class\n");

  mid = (*env)->GetStaticMethodID (env, ia_cls, "getByName",
				   "(Ljava/lang/String;)Ljava/net/InetAddress;");
  if (mid == NULL)
    {
      return NULL;
    }

  DBG ("_javanet_create_inetaddress(): Found getByName method\n");

  ip_str = (*env)->NewStringUTF (env, buf);
  if (ip_str == NULL)
    {
      return NULL;
    }

  ia = (*env)->CallStaticObjectMethod (env, ia_cls, mid, ip_str);
  if (ia == NULL)
    {
      return NULL;
    }

  DBG ("_javanet_create_inetaddress(): Called getByName method\n");

  return ia;
#else /* not WITHOUT_NETWORK */
  return NULL;
#endif /* not WITHOUT_NETWORK */
}

/*************************************************************************/

static void
_javanet_set_remhost_addr (JNIEnv * env, jobject this, jobject ia)
{
  jclass this_cls;
  jfieldID fid;

  /* Set the variable in the object */
  this_cls = (*env)->FindClass (env, "java/net/SocketImpl");
  if (this_cls == NULL)
    return;

  fid =
    (*env)->GetFieldID (env, this_cls, "address", "Ljava/net/InetAddress;");
  if (fid == NULL)
    return;

  DBG ("_javanet_set_remhost_addr(): Found address field\n");

  (*env)->SetObjectField (env, this, fid, ia);
  DBG ("_javanet_set_remhost_addr(): Set field\n");
}

/*
 * Set's the value of the "addr" field in PlainSocketImpl with a new
 * InetAddress for the specified addr
 */
static void
_javanet_set_remhost (JNIEnv * env, jobject this, cpnet_address *netaddr)
{
  jobject ia;

  DBG ("_javanet_set_remhost(): Entered _javanet_set_remhost\n");

  /* Get an InetAddress object */
  ia = _javanet_create_inetaddress (env, netaddr);
  if (ia == NULL)
    return;

  _javanet_set_remhost_addr (env, this, ia);
}


/*************************************************************************/

/*
 * Returns an Internet address for the passed in InetAddress object
 */
cpnet_address *
_javanet_get_ip_netaddr (JNIEnv * env, jobject addr)
{
#ifndef WITHOUT_NETWORK
  jclass cls = 0;
  jmethodID mid;
  jarray arr = 0;
  jbyte *octets;
  cpnet_address *netaddr;
  jint len;

  DBG ("_javanet_get_ip_netaddr(): Entered _javanet_get_ip_netaddr\n");

  if (addr == NULL)
    {
      JCL_ThrowException (env, "java/lang/NullPointerException",
			  "Null address");
      return 0;
    }

  /* Call the getAddress method on the object to retrieve the IP address */
  cls = (*env)->GetObjectClass (env, addr);
  if (cls == NULL)
    return 0;

  mid = (*env)->GetMethodID (env, cls, "getAddress", "()[B");
  if (mid == NULL)
    return 0;

  DBG ("_javanet_get_ip_netaddr(): Got getAddress method\n");

  arr = (*env)->CallObjectMethod (env, addr, mid);
  if (arr == NULL)
    return 0;

  DBG ("_javanet_get_ip_netaddr(): Got the address\n");

  /* Turn the IP address into a system cpnet address.
   * If the length is 4 then it is an IPV4 address, if it
   * is 16 then it is an IPV6 address else it is an InternError. */
  len = (*env)->GetArrayLength (env, arr);
  if (len != 4 && len != 16)
    {
      JCL_ThrowException (env, IO_EXCEPTION, "Internal Error");
      return 0;
    }
  DBG ("_javanet_get_ip_netaddr(): Length ok\n");

  octets = (*env)->GetByteArrayElements (env, arr, 0);
  if (octets == NULL)
    return 0;

  DBG ("_javanet_get_ip_netaddr(): Grabbed bytes\n");

  switch (len)
    {
    case 4:      
      netaddr = cpnet_newIPV4Address(env);
      cpnet_bytesToIPV4Address(netaddr, octets);
      break;
#ifdef HAVE_INET6
    case 16:
      netaddr = cpnet_newIPV6Address(env);
      cpnet_bytesToIPV6Address(netaddr, octets);
      break;
#endif
    default:
      /* This should not happen as we have checked before.
       * But that way we shut the compiler warnings */
      JCL_ThrowException (env, IO_EXCEPTION, "Internal Error");
      return 0;
      
    }

  (*env)->ReleaseByteArrayElements (env, arr, octets, 0);
  DBG ("_javanet_get_ip_netaddr(): Done getting addr\n");

  return netaddr;
#else /* not WITHOUT_NETWORK */
#endif /* not WITHOUT_NETWORK */
}

// ȍ~̊֐͎gpĂȂ悤Ȃ̂ŁAS폜

/* end of file */
