﻿// Copyright 2010 The Native Client Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can
// be found in the LICENSE file.
#include <stdio.h>

#include <ppapi/cpp/audio.h>
#include <ppapi/cpp/instance.h>
#include <ppapi/cpp/module.h>
#include <ppapi/cpp/completion_callback.h>
#include <ppapi/cpp/url_loader.h>
#include <ppapi/cpp/url_request_info.h>
#include <ppapi/cpp/dev/scriptable_object_deprecated.h>
#include <ppapi/cpp/var.h>
#include <pthread.h>

#include <cassert>
#include <cmath>
#include <limits>
#include <sstream>
#include <iostream>
#include <list>
#include "boost/shared_ptr.hpp"
#include "boost/format.hpp"

#include "_mod.h"
#include "geturl_handler.h"
//#include "zlib.h"

namespace {
  const int kPthreadMutexSuccess = 0;
  // XM ファイルのロード
  const char* const kLoadXmFile = "loadXmFile";
  // XM ファイルの再生コマンド文字列
  const char* const kPlaySoundId = "playSound";
  // XM ファイルの停止コマンド文字列
  const char* const kStopSoundId = "stopSound";
  // メインスレッド以外のログを取得する
  const char* const kGetLogId = "getLog";
 //
  const char* const kCheckLogId = "checkLog";
  const char * const kRecievedBytesId = "receivedBytes";
  const char * const kTotalBytesId = "totalBytes";

 // システムにリクエストするサンプルカウント数
const uint32_t kSampleFrameCount = 4096u;
// 現在サポートしているのはステレオオーディオのみである。
const uint32_t kChannels = 2u;
}  // namespace

namespace xm_player {
// A small helper RAII class that implementes a scoped pthread_mutex lock.
class ScopedMutexLock {
 public:
  explicit ScopedMutexLock(pthread_mutex_t* mutex) : mutex_(mutex) {
    if (pthread_mutex_lock(mutex_) != kPthreadMutexSuccess) {
      mutex_ = NULL;
    }
  }
  ~ScopedMutexLock() {
    if (mutex_)
      pthread_mutex_unlock(mutex_);
  }
  bool is_valid() const {
    return mutex_ != NULL;
  }
 private:
  pthread_mutex_t* mutex_;  // Weak reference.
};

class XmPlayerInstance : public pp::Instance {
 public:
  explicit XmPlayerInstance(PP_Instance instance)
      : pp::Instance(instance)/*, scriptable_object_(NULL)*/,
        sample_frame_count_(kSampleFrameCount)  , mod_( MOD__Create(this))
  {
    pthread_mutexattr_init(&mutex_attr_);
    pthread_mutexattr_settype(&mutex_attr_,PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&mutex_, NULL );
  }

  virtual ~XmPlayerInstance() 
  {
    pthread_mutex_destroy(&mutex_);
    pthread_mutexattr_destroy(&mutex_attr_);
    MOD__Destruct(mod_);
  }

  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]) {

    // Ask the device for an appropriate sample count size.
    sample_frame_count_ =
        pp::AudioConfig::RecommendSampleFrameCount(PP_AUDIOSAMPLERATE_44100,
                                                   kSampleFrameCount);
    audio_ = pp::Audio(
        this,
        pp::AudioConfig(this,
                        PP_AUDIOSAMPLERATE_44100,
                        sample_frame_count_),
        XmPlayerCallback, this);
    return true;
  }

  void play()
  {
      if(MOD__LoadSong(mod_,&(load_data_[0]),load_data_.size()) > 0 ){
          MOD__InitPlay(mod_);
          audio_.StartPlayback();
          SendLog("started.");
      } else {
        SendLog("Load Error.");
      };
  }

  void stop()
  {
    audio_.StopPlayback();
    SendLog("stopped.");
  }
  std::vector<char> & GetLoadData(){return load_data_;}
  void WriteLog(const char * log)
  {  
    PostMessage(log);
    //ScopedMutexLock lock(&mutex_);
    //  log_.push_back(log);
  }

  bool CheckLog()
  {
    return !log_.empty();
  }

  std::string GetLog()
  {
    if(log_.empty())
    {
        return "";
    }
    std::string ret(log_.front());
    {
     ScopedMutexLock lock(&mutex_);
      log_.pop_front();
    }
    return ret;
  };

  void SendLog(const std::string& log)
  {
    PostMessage(std::string("reportLog:") + log);
  }

  virtual void HandleMessage(const pp::Var& message)
  {
    SendLog("HandleMessage");
    const std::string& mes(message.AsString());
    if(mes.find(kLoadXmFile) == 0)
    {
      size_t sep_pos = mes.find_first_of(':');
      if (sep_pos != std::string::npos) {
        std::string url = mes.substr(sep_pos + 1);
         LoadXmFile(url);
      }
    }

    if(mes == kPlaySoundId)
    {
      play();
    }

    if(mes == kStopSoundId)
    {
      stop();
    }


  };

  bool LoadXmFile(const std::string& url)
  {
     SendLog("XmPlayerInstance::LoadXmFile()::CreateGetURLHandler");
    //MODファイルのロード
   { //GetURLHandler *handler_ = 0;
    if(!handler_  || handler_->GetStatus() == GetURLHandler::end){
       handler_.reset(GetURLHandler::Create(this,url,GetLoadData()));
      // handler_= GetURLHandler::Create(this,url,GetLoadData(),jsFuncName);
      if(!handler_)
      {
          return false;
      }
    } else {
      return false;
    }
    SendLog("XmPlayerInstance::LoadXmFile()::handler_->Start()");
    if(!handler_->Start()){
      return false;
    };
    SendLog("XmPlayerInstance::LoadXmFile() Exit");
    return true;
   }
  }

  int32_t GetReceivedBytes()
  {
    if(!handler_) return 0;
     return  handler_->GetReceivedBytes();
  }

  int32_t GetTotalBytes()
  {
    if(!handler_) return 0;
    return  handler_->GetTotalBytes();
  }

 private:
  static void XmPlayerCallback(void* samples, size_t buffer_size, void* data) {
    XmPlayerInstance* xm_player_instance =
        reinterpret_cast<XmPlayerInstance*>(data);
    //int16_t* buff = reinterpret_cast<int16_t*>(samples);

    // Make sure we can't write outside the buffer.
    //assert(buffer_size >= (sizeof(*buff) * kChannels *
    //                       xm_player_instance->sample_frame_count_));
    MOD__PlaySong(xm_player_instance->mod_,reinterpret_cast<U32*>(samples),buffer_size / (sizeof(U32)));
  }
  // Audio resource. Allocated in Init()
  pp::Audio audio_;

  //XmPlayerScriptableObject* scriptable_object_;

  // The count of sample frames per channel in an audio buffer.
  uint32_t sample_frame_count_;
 // MOD構造体へのポインタ
  ::MOD * mod_;
  std::vector<char> load_data_;
  std::list<std::string> log_;
  pthread_mutex_t mutex_;
  pthread_mutexattr_t mutex_attr_;
  boost::shared_ptr<GetURLHandler> handler_; 
};

// The Module class.  The browser calls the CreateInstance() method to create
// an instance of you NaCl module on the web page.  The browser creates a new
// instance for each <embed> tag with type="application/x-ppapi-nacl-srpc".
class XmPlayerModule : public pp::Module {
 public:
  XmPlayerModule() : pp::Module() {}
  ~XmPlayerModule() {}

  // Create and return a HelloWorldInstance object.
  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new XmPlayerInstance(instance);
  }
};


}  // namespace xm_player


void LogMsg (void* instance,const char* format,...)
{
  va_list ap;
  va_start(ap,format);
  char buf[4096];
  vsnprintf(buf,sizeof(buf),format,ap);
  va_end(ap);

  //pp::Instance * inst = pp::Module::Get()->InstanceForPPInstance(instance);
 xm_player::XmPlayerInstance * inst = static_cast<xm_player::XmPlayerInstance*>(instance);
 inst->SendLog(buf);
 
 //printf(buf);
 //inst->WriteLog(buf);
 // pp::Var window(inst->GetWindowObject());
  //window.is_int();
  //pp::Var exception;
  //window.Call("reportLog",std::string(buf), &exception);
};

// Factory function called by the browser when the module is first loaded.
// The browser keeps a singleton of this module.  It calls the
// CreateInstance() method on the object you return to make instances.  There
// is one instance per <embed> tag on the page.  This is the main binding
// point for your NaCl module with the browser.
namespace pp {
Module* CreateModule() {
  return new xm_player::XmPlayerModule();
}
}  // namespace pp


