package nicolib.comment;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.LinkedList;
import java.util.Queue;

import nicolib.util.Logger;

/**
 * コメントサーバー通信の基底クラス
 * サーバーから取得した文字列は
 * 1.このクラスで適切なブロックに分解され
 * 2.StreamPaserで対応するクラスのインスタンスに変換し
 * 3.CommentHandlerでイベント通知
 * という流れで処理される
 * 
 * データの受信とイベントの通知は独立したスレッドで行われる
 * @author hal
 *
 */
public abstract class CommentReceiver {

	private final static String threadStartMessageFormat = "<thread thread=\"%s\" res_from=\"%s\" version=\"20061206\" />\0";
	
	protected CommentHandler handler;
	protected Socket socket;
	protected StreamParser parser;
	
	private CommentBuffer buffer;
	
	
	private Thread recvThread;
	private Thread servThread;
	
	private String connectionString;
	
	private Object connectSyncObj = new Object();
	
	public CommentReceiver(){
		buffer = new CommentBuffer();
	}
	
	/**
	 * 接続状況通知を受け取るハンドラを設定します
	 * @param handler
	 */
	public void setHandler(CommentHandler handler) {
		this.handler = handler;
	} 
	
	/**
	 * データ解析オブジェクトを取得します
	 * パーサーを変更したい場合はこのメソッドをオーバーライドしてください。
	 * @return
	 */
	protected StreamParser getParser(){
		if(parser == null){
			parser = new StreamParser();
		}
		return parser;
	}	
	
	/**
	 * コメントサーバーに接続します
	 * @param addr サーバーのアドレス
	 * @param port サーバーのポート
	 * @param thread スレッド
	 * @param resFrom コメント取得開始位置
	 * @return 接続を開始できたか
	 */
	public boolean connectServer(String addr, int port, int thread, int resFrom){
		
		disconnect();
		
		synchronized(connectSyncObj){
		
			try {
				socket = new Socket(addr, port);
				connectionString = String.format(threadStartMessageFormat, thread, resFrom);
				
				recvThread = new Thread(new Runnable() {
					@Override
					public void run() {
						receive();
					}
				}, "recvThread");
				
				servThread = new Thread(new Runnable() {
					@Override
					public void run() {
						serve();
					}
				}, "servThread");
				
				recvThread.start();
				servThread.start();
							
				return true;
				
			} catch (UnknownHostException e) {
				Logger.writeException(e);
			} catch (IOException e) {
				Logger.writeException(e);
			}
			
			return false;
		}
	}
	
	/**
	 * コメントサーバーから切断します。
	 * 受信用のスレッドが終了するまで呼び出しスレッドは中断されます
	 * @return 正しく切断で来たか
	 */
	public boolean disconnect(){
		synchronized(connectSyncObj){
			if(recvThread != null && recvThread.isAlive()){
				recvThread.interrupt();
				
				try {
					socket.close();
				} catch (IOException e) {
					Logger.writeException(e, true);
					return false;
				}
				
				try {
					connectSyncObj.wait();
				} catch (InterruptedException e) {
					Logger.writeException(e, true);
					return false;
				}
				recvThread = null;
				servThread = null;
			}
		}
		
		return true;
	} 
	
	/**
	 * 受信処理
	 */
	private void receive(){
		
		if(socket == null || connectionString == null) return;
		
		try{
			
			DataOutputStream out = null;
			BufferedReader in = null;
		
			try{
			
				out = new DataOutputStream(socket.getOutputStream());
				in = new BufferedReader(new InputStreamReader(socket.getInputStream(), "UTF-8"));
				
				out.write(connectionString.getBytes());
				
				StringBuilder sb = new StringBuilder();
				while(true){
					int d = in.read();
					if(d == 0){
						commentRecv(sb.toString());
						sb.setLength(0);
						
						if(Thread.interrupted()) break;
						
					} else if(d == -1){
						break;
					} else {
						sb.append((char)d);
					}
				}
			}catch(InterruptedException e){
				System.out.println("receive end");
			}finally{
				in.close();
				out.close();
				socket.close();
			}
		}catch(SocketException e){
			// in.readで待ち状態にある時にdisconnectが呼び出されるとここで例外が補足される 
			// Logger.writeException(e);
		}catch(IOException e){
			Logger.writeException(e, true);
		}finally{
			if(servThread != null && servThread.isAlive()){
				servThread.interrupt();
			}
		}
		
		// 通信終了を通知
		synchronized(connectSyncObj){
			connectSyncObj.notifyAll();
		}
	}
	
	/**
	 * サーバーから受け取った文字列をオブジェクトに変換してバッファーに入れる
	 * @param text
	 * @throws InterruptedException
	 */
	protected void commentRecv(String text)
		throws InterruptedException
	{
		Object obj = getParser().parse(text);
		if(obj != null){
			buffer.set(obj);
		}
	}
	
	/**
	 * イベント通知処理
	 * バッファーを監視
	 */
	private void serve(){
		try {
			while(!Thread.interrupted()){
				Object obj = buffer.get();
				commentServ(obj);
			}
		} catch (InterruptedException e) {
		}
		
		handler.disconnectServer();
	}
	
	/**
	 * 変換されたデータをハンドラに渡します
	 * @param obj
	 */
	protected void commentServ(Object obj){
		
		if(obj instanceof ThreadHeader){
			handler.connectServer((ThreadHeader)obj);
		}
		
	}
		
}

class CommentBuffer {
	private Queue<Object> buf;
	
	public CommentBuffer(){
		buf = new LinkedList<Object>();
	}
	
	public synchronized void set(Object obj){
		boolean empty = buf.isEmpty();
		
		buf.add(obj);
		if(empty){
			notifyAll();
		}
	}
	
	public synchronized Object get() 
		throws InterruptedException
	{
		while(buf.isEmpty()){
			wait();
		}
		
		return buf.poll();
	}
}
