package jp.sourceforge.nicoro;

import static jp.sourceforge.nicoro.Log.LOG_TAG;
import static jp.sourceforge.nicoro.PlayerConstants.*;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.preference.PreferenceManager;
import android.support.v4.app.Fragment;
import android.text.TextUtils;
import android.view.ViewTreeObserver;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public abstract class AbstractPlayerFragment extends Fragment
implements ViewTreeObserver.OnGlobalLayoutListener,
PlayerInfoViews.PlayDataAppender, PlayerInfoViews.TimeAppender,
DestroyTask.Callback {
    private static final boolean DEBUG_LOGV = Release.IS_DEBUG & true;
    private static final boolean DEBUG_LOGD = Release.IS_DEBUG & true;

    protected static final String INFO_TIME_DEFAULT = "000:00";

    private static final String KEY_SAVE_CURRENT_PLAY_TIME = "KEY_SAVE_CURRENT_PLAY_TIME";
    private static final String KEY_SAVE_IS_PAUSE_PLAY = "KEY_SAVE_IS_PAUSE_PLAY";

    protected Context mContext;
    protected SharedPreferences mSharedPreferences;

    protected VideoLoader mVideoLoader;
    protected MessageLoader mMessageLoader;
    protected MessageLoaderFork mMessageLoaderFork;

    protected ThumbInfo mThumbInfo;

    protected MessageData mMessageData = new MessageData();
    protected MessageData mMessageDataFork = new MessageData();

    protected Rational mRationalCurrentPlayTime = new Rational();

    protected volatile MessageHandler mHandler;

    protected MessageChatController mMessageChatController = new MessageChatController();

    protected StateManager mStateManager = new StateManager();
    protected boolean mDidStartPlay = false;

    private Rational mSaveStateCurrentPlayTime;
    private boolean mSaveIsPausePlay;
    protected boolean mIsVideoDownloadOk;
    protected boolean mIsVideoCachedOk;

    protected boolean mShowHintToast;

    protected int mLastOrientation;

    protected abstract class MessageHandler extends Handler {
        @Override
        public void handleMessage(Message msg) {
            if (mHandler == null) {
                if (DEBUG_LOGD) {
                    Log.d(LOG_TAG, Log.buf().append("Fragment was destroyed. ignore message=")
                            .append(msg.toString()).toString());
                }
                return;
            }
            PlayerActivity playerActivity = getPlayerActivity();
            Handler parentHandler = (playerActivity == null) ? null
                    : playerActivity.getHandler();
            switch (msg.what) {
            case MSG_ID_MESSAGE_FINISHED:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_MESSAGE_FINISHED);
                }
                mMessageData.setChats(mMessageLoader.getChats());
                if (canStartPlay()) {
                    startPlay();
                }
                break;
            case MSG_ID_MESSAGE_OCCURRED_ERROR:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_MESSAGE_OCCURRED_ERROR,
                            msg.obj).sendToTarget();
                }
                break;
            case MSG_ID_THUMBINFO_FINISHED:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_THUMBINFO_FINISHED,
                            mIsVideoDownloadOk ? 1 : 0, 0, mThumbInfo).sendToTarget();
                }
                mMessageLoader.setVideoLength((String) msg.obj);
                mMessageLoader.startLoad();
                mMessageLoaderFork.setVideoLength((String) msg.obj);
                mMessageLoaderFork.startLoad();
                break;
            case MSG_ID_THUMBINFO_OCCURRED_ERROR:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_THUMBINFO_OCCURRED_ERROR,
                            msg.obj).sendToTarget();
                }
                break;
            case MSG_ID_VIDEO_OCCURRED_ERROR:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_VIDEO_OCCURRED_ERROR,
                            msg.obj).sendToTarget();
                }
                break;
            case MSG_ID_VIDEO_NOTIFY_PROGRESS:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_VIDEO_NOTIFY_PROGRESS,
                            msg.arg1, msg.arg2).sendToTarget();
                }
                break;
            case MSG_ID_PLAY_ERROR:
                if (parentHandler != null) {
                    parentHandler.obtainMessage(MSG_ID_PLAY_ERROR,
                            msg.obj).sendToTarget();
                }
                break;
            case MSG_ID_MESSAGE_FORK_FINISHED:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_MESSAGE_FORK_FINISHED);
                }
                mMessageDataFork.setChats(mMessageLoaderFork.getChats());
                if (canStartPlay()) {
                    startPlay();
                }
                break;
            case MSG_ID_MESSAGE_FORK_OCCURRED_ERROR:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_MESSAGE_FORK_OCCURRED_ERROR);
                }
                // TODO 投稿者コメントのエラーはとりあえず無視？
                Log.w(LOG_TAG, Log.buf().append("fork=\"1\" message load failed:")
                        .append((String) msg.obj).toString());
                // ぬるぽ出るのでメンバも設定
                mMessageDataFork.setChats(null);
                break;
            case MSG_ID_ENABLE_SEEK_BAR:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_ENABLE_SEEK_BAR);
                }
                break;
            case MSG_ID_INFO_PLAY_DATA_UPDATE:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_INFO_PLAY_DATA_UPDATE);
                }
                break;
            case MSG_ID_THUMBINFO_FINISHED_NEW:
                mThumbInfo = (ThumbInfo) msg.obj;
                mHandler.obtainMessage(MSG_ID_THUMBINFO_FINISHED,
                        mThumbInfo.getLength()).sendToTarget();
                break;
            case MSG_ID_THUMBINFO_OCCURRED_ERROR_NEW:
                String errorMessage = (String) msg.obj;
                mHandler.obtainMessage(MSG_ID_THUMBINFO_OCCURRED_ERROR,
                        errorMessage).sendToTarget();
                break;
            case MSG_ID_PLAY_FINISHED:
                // ダイアログ表示は少し遅らせる
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessageDelayed(MSG_ID_PLAY_FINISHED_DIALOG,
                            1000);
                }
                break;
            case MSG_ID_VIDEO_DOWNLOAD_FINISHED:
                if (parentHandler != null) {
                    parentHandler.sendEmptyMessage(MSG_ID_VIDEO_DOWNLOAD_FINISHED);
                }
                mIsVideoDownloadOk = true;
                break;
            default:
                assert false : msg.what;
                break;
            }
        }
    }

    /**
     * すべての再生準備が整ったか確認する（動画ファイルのダウンロード、コメントファイルのダウンロード等）
     * @return
     */
    protected abstract boolean canStartPlay();

    /**
     * 動画の再生位置を取得する
     * @param rational 位置情報（秒単位の分数）
     */
    public abstract void getCurrentPositionVideoPlay(Rational rational);
    /**
     * 音声の再生位置を取得する
     * @param rational 位置情報（秒単位の分数）
     */
    protected abstract void getCurrentPositionAudioPlay(Rational rational);
    /**
     * 動画のデコード位置を取得する
     * @param rational 位置情報（秒単位の分数）
     */
    protected abstract void getCurrentPositionVideoDecode(Rational rational);
    /**
     * 音声のデコード位置を取得する
     * @param rational 位置情報（秒単位の分数）
     */
    protected abstract void getCurrentPositionAudioDecode(Rational rational);

    /**
     * 再生状態と一時停止状態を切り替える
     * @return 成否
     */
    protected abstract boolean switchPausePlay();
    /**
     * 再生を一時停止する
     */
    protected abstract void pausePlay();
    /**
     * 一時停止していた再生を再開する
     */
    protected abstract void restartPlay();
    /**
     * 再生が一時停止中か確認する
     * @return
     */
    protected abstract boolean isPausePlay();

    /**
     * 秒単位でシークする
     * @param second
     */
    protected abstract void seekBySecond(int second);

    protected abstract VideoLoader.EventListener createVideoLoaderEventListener();

    @Override
    public StringBuilder appendMovieType(StringBuilder builder) {
        if (mVideoLoader.isLow()) {
            builder.append("low-")
                .append(mVideoLoader.getMovieType());
        } else {
            builder.append(mThumbInfo.getMovieType());
        }
        return builder;
    }

    @Override
    public StringBuilder appendTotalPlayTime(StringBuilder builder) {
        return builder.append(mThumbInfo.getFormattedLengthForPlayer());
    }

    protected abstract void startPlay();

    protected void startPlay(ThumbInfo thumbInfo) {
        PlayerActivity playerActivity = getPlayerActivity();
        if (playerActivity != null) {
            playerActivity.startPlay(thumbInfo);

            Handler parentHandler = playerActivity.getHandler();
            if (parentHandler != null) {
                parentHandler.sendEmptyMessage(MSG_ID_INFO_TIME_UPDATE);
//                parentHandler.sendEmptyMessage(MSG_ID_INFO_CLOCK_UPDATE);
            }
        }

        if (thumbInfo != null) {
            mMessageChatController.setDuration(thumbInfo.getLengthByVpos());
        }

        mDidStartPlay = true;
    }

    protected void postStartPlayIfIsRestored() {
        if (mStateManager.wasRestored()) {
            // 再開時は一時停止にしておく
            pausePlay();
            setButtonPauseImage();
            int second = -1;
            if (mSaveStateCurrentPlayTime != null) {
                second = (int) (mSaveStateCurrentPlayTime.num / mSaveStateCurrentPlayTime.den);
                if (second != 0) {  // 0のときはシーク不要
                    seekBySecond(second);
                }
            }
            PlayerActivity playerActivity = getPlayerActivity();
            if (playerActivity != null) {
                playerActivity.postStartPlay(second, mThumbInfo.getLengthBySecond());
            }
        }
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onCreate: savedInstanceState=")
                    .append(savedInstanceState).toString());
        }
        super.onCreate(savedInstanceState);
        mStateManager.onCreate(savedInstanceState);

        mContext = getActivity().getApplicationContext();

        if (savedInstanceState == null) {
        } else {
            mSaveStateCurrentPlayTime = savedInstanceState.getParcelable(
                    KEY_SAVE_CURRENT_PLAY_TIME);
            mSaveIsPausePlay = savedInstanceState.getBoolean(
                    KEY_SAVE_IS_PAUSE_PLAY);
        }

        mDidStartPlay = false;
        mIsVideoDownloadOk = false;
        mIsVideoCachedOk = false;

        mMessageData.clear();
        mMessageDataFork.clear();

        mSharedPreferences = PreferenceManager.getDefaultSharedPreferences(
                mContext);

        Bundle args = getArguments();
        loadThumbInfo(args);

        String userSession = mSharedPreferences.getString(NicoroConfig.COOKIE_USER_SESSION, null);
        if (mMessageLoader != null && mMessageLoader.getChats() != null) {
            mHandler.sendEmptyMessage(MSG_ID_MESSAGE_FINISHED);
        } else {
            mMessageLoader = createMessageLoader(args, mContext, userSession);
            if (mMessageLoader != null) {
                mMessageLoader.setEventListener(new MessageLoader.EventListener() {
                    @Override
                    public void onFinished(MessageLoader messageLoader) {
                        Handler handler = mHandler;
                        if (handler != null) {
                            handler.sendEmptyMessage(MSG_ID_MESSAGE_FINISHED);
                        }
                    }
                    @Override
                    public void onOccurredError(MessageLoader messageLoader,
                            String errorMessage) {
                        Handler handler = mHandler;
                        if (handler != null) {
                            handler.obtainMessage(MSG_ID_MESSAGE_OCCURRED_ERROR,
                                    errorMessage).sendToTarget();
                        }
                    }
                });
            }
        }
        if (mMessageLoaderFork != null && mMessageLoaderFork.getChats() != null) {
            mHandler.sendEmptyMessage(MSG_ID_MESSAGE_FORK_FINISHED);
        } else {
            mMessageLoaderFork = createMessageLoaderFork(args, mContext, userSession);
            if (mMessageLoaderFork != null) {
                mMessageLoaderFork.setEventListener(new MessageLoader.EventListener() {
                    @Override
                    public void onFinished(MessageLoader messageLoader) {
                        Handler handler = mHandler;
                        if (handler != null) {
                            handler.sendEmptyMessage(MSG_ID_MESSAGE_FORK_FINISHED);
                        }
                    }
                    @Override
                    public void onOccurredError(MessageLoader messageLoader,
                            String errorMessage) {
    //                  // TODO 投稿者コメントのエラーはとりあえず無視？
    //                  Log.w(LOG_TAG, Log.buf().append("fork=\"1\" message load failed:")
    //                          .append(errorMessage).toString());
    //                  mMessageDataFork.mIsMessageOk = true;
                        Handler handler = mHandler;
                        if (handler != null) {
                            handler.obtainMessage(MSG_ID_MESSAGE_FORK_OCCURRED_ERROR,
                                    errorMessage).sendToTarget();
                        }
                    }
                });
            }
        }

        boolean messageAntialias = mSharedPreferences.getBoolean(
                getString(R.string.pref_key_message_antialias), false);
        mMessageChatController.setAntiAlias(messageAntialias);

        mShowHintToast = mSharedPreferences.getBoolean(
                getString(R.string.pref_key_show_hint_toast), true);
        if (mShowHintToast) {
            Util.showInfoToast(mContext,
                    R.string.toast_explain_player_ctrl);
        }

        Configuration config = getResources().getConfiguration();
        mLastOrientation = config.orientation;
    }

    @Override
    public void onStart() {
        super.onStart();
        mStateManager.onStart();
    }


    @Override
    public void onResume() {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onResume").toString());
        }
        super.onResume();
        mStateManager.onResume();

        if (canStartPlay() && !mDidStartPlay) {
            startPlay();
        }
    }

    private void savePlayState() {
        if (mDidStartPlay) {
            if (mSaveStateCurrentPlayTime == null) {
                mSaveStateCurrentPlayTime = new Rational();
            }
            getCurrentPositionVideoPlay(mSaveStateCurrentPlayTime);
            mSaveIsPausePlay = isPausePlay();
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onSaveInstanceState: outState=")
                    .append(outState).toString());
        }
        super.onSaveInstanceState(outState);
        mStateManager.onSaveInstanceState();

        savePlayState();
        outState.putParcelable(KEY_SAVE_CURRENT_PLAY_TIME, mSaveStateCurrentPlayTime);
        outState.putBoolean(KEY_SAVE_IS_PAUSE_PLAY, mSaveIsPausePlay);
    }

    @Override
    public void onPause() {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onPause").toString());
        }
        super.onPause();
        mStateManager.onPause();
    }

    @Override
    public void onStop() {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onStop").toString());
        }
        super.onStop();
        mStateManager.onStop();
    }

    @Override
    public void onDestroy() {
        if (DEBUG_LOGV) {
            Log.v(LOG_TAG, Log.buf().append(getClass().getName())
                    .append("#onDestroy").toString());
        }
        mStateManager.onDestroy();

        super.onDestroy();
    }

    @Override
    public void onDestroyImplPre() {
        mHandler = null;
    }

    @Override
    public void onDestroyImpl() {
        releaseLoader();
    }

    @Override
    public void onDestroyImplPost() {
        mVideoLoader = null;
        mMessageLoader = null;
        mMessageLoaderFork = null;
        mThumbInfo = null;
        mMessageChatController = null;
    }

    @Override
    public void finishReally() {
        // 空実装
    }

    @Override
    public boolean wasDestroyed() {
        return mStateManager.wasDestroyed();
    }

    private void releaseLoader() {
        if (mVideoLoader == null && mMessageLoader == null
                && mMessageLoaderFork == null && mThumbInfo == null) {
            // すべて解放済みなら処理無し
            return;
        }

        // XXX 本当は***Loader類のabort対応が必要
        ExecutorService executorService = Executors.newCachedThreadPool();
        CountDownLatch latch = new CountDownLatch(4);
        if (mVideoLoader == null) {
            latch.countDown();
        } else {
            mVideoLoader.finishAsync(executorService, latch);
        }
        if (mMessageLoader == null) {
            latch.countDown();
        } else {
            mMessageLoader.finishAsync(executorService, latch);
        }
        if (mMessageLoaderFork == null) {
            latch.countDown();
        } else {
            mMessageLoaderFork.finishAsync(executorService, latch);
        }
        if (mThumbInfo == null) {
            latch.countDown();
        } else {
            mThumbInfo.finishAsync(executorService, latch);
        }
        while (true) {
            try {
                latch.await();
                break;
            } catch (InterruptedException e) {
                Log.d(LOG_TAG, e.toString(), e);
            }
        }
        executorService.shutdown();
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (mLastOrientation != newConfig.orientation) {
            onOrientationChanged(newConfig.orientation);
            mLastOrientation = newConfig.orientation;
        }
    }

    abstract protected void onOrientationChanged(int orientation);

    @Override
    public void onGlobalLayout() {
    }

    protected static VideoLoader createVideoLoader(Bundle args, Context context) {
        VideoLoader videoLoader = null;
        String url = args.getString(INTENT_NAME_VIDEO_URL);
        String cookie = args.getString(INTENT_NAME_COOKIE);
        String videoNumber = args.getString(INTENT_NAME_VIDEO_NUMBER);
        String cookieUserSession = args.getString(INTENT_NAME_COOKIE_USER_SESSION);

        if (DEBUG_LOGD) {
            Log.d(LOG_TAG, Log.buf().append(" url=").append(url).append(" cookie=")
                    .append(cookie).append(" videoNumber=").append(videoNumber)
                    .append(" cookieUserSession=").append(cookieUserSession)
                    .toString());
        }
        if (url != null && cookie != null) {
            videoLoader = new VideoLoader(url, cookie, videoNumber,
                    context, cookieUserSession,
//                  android.os.Process.THREAD_PRIORITY_BACKGROUND);
                    android.os.Process.THREAD_PRIORITY_BACKGROUND
                    + android.os.Process.THREAD_PRIORITY_MORE_FAVORABLE * 2);
        }
        return videoLoader;
    }

    protected static MessageLoader createMessageLoader(Bundle args, Context context,
            String userSession) {
        MessageLoader messageLoader = null;
        String cookie = args.getString(
                INTENT_NAME_COOKIE);
        String messageUrl = args.getString(
                INTENT_NAME_MESSAGE_URL);
        String threadId = args.getString(
                INTENT_NAME_THREAD_ID);
        String userId = args.getString(
                INTENT_NAME_USER_ID);
        String threadKey = args.getString(
                INTENT_NAME_THREAD_KEY);
        String force184 = args.getString(
                INTENT_NAME_FORCE_184);
        if (DEBUG_LOGD) {
            Log.d(LOG_TAG, Log.buf().append(" cookie=").append(cookie)
                    .append(" messageUrl=").append(messageUrl).append(" threadId=")
                    .append(threadId).append(" userId=").append(userId).append(" threadKey=")
                    .append(threadKey).append(" force184=").append(force184).toString());
        }
        if (messageUrl != null && threadId != null && cookie != null) {
            messageLoader = new MessageLoader(messageUrl, threadId, userSession,
                    userId, threadKey, force184,
                    context);
        }
        return messageLoader;
    }

    protected static MessageLoaderFork createMessageLoaderFork(Bundle args, Context context,
            String userSession) {
        MessageLoaderFork messageLoader = null;
        String cookie = args.getString(
                INTENT_NAME_COOKIE);
        String messageUrl = args.getString(
                INTENT_NAME_MESSAGE_URL);
        String threadId = args.getString(
                INTENT_NAME_THREAD_ID);
        String userId = args.getString(
                INTENT_NAME_USER_ID);
        String threadKey = args.getString(
                INTENT_NAME_THREAD_KEY);
        String force184 = args.getString(
                INTENT_NAME_FORCE_184);
        if (DEBUG_LOGD) {
            Log.d(LOG_TAG, Log.buf().append(" cookie=").append(cookie)
                    .append(" messageUrl=").append(messageUrl).append(" threadId=")
                    .append(threadId).append(" userId=").append(userId).append(" threadKey=")
                    .append(threadKey).append(" force184=").append(force184).toString());
        }
        if (messageUrl != null && threadId != null && cookie != null) {
            messageLoader = new MessageLoaderFork(messageUrl, threadId, userSession,
                    userId, threadKey, force184,
                    context);
        }
        return messageLoader;
    }

    protected void loadThumbInfo(Bundle args) {
        String videoNumber = args.getString(INTENT_NAME_VIDEO_NUMBER);
        if (DEBUG_LOGD) {
            Log.d(LOG_TAG, Log.buf().append(" videoNumber=").append(videoNumber).toString());
        }
        loadThumbInfo(videoNumber);
    }
    protected void loadThumbInfo(String videoNumber) {
        if (videoNumber == null) {
            return;
        }
        Handler handler = mHandler;
        if (mThumbInfo != null && videoNumber.equals(mThumbInfo.getVideoNumber())) {
            // ThumbInfo is set already
            if (handler != null) {
                handler.obtainMessage(MSG_ID_THUMBINFO_FINISHED,
                        mThumbInfo.getLength()).sendToTarget();
            }
            return;
        }
        Activity activity = getActivity();
        if (activity == null) {
            return;
        }
        ThumbInfoCacher cacher = NicoroApplication.getInstance(activity
                ).getThumbInfoCacher();
        ThumbInfo thumbInfo = cacher.getThumbInfo(videoNumber);
        if (thumbInfo == null) {
            if (handler != null) {
                cacher.loadThumbInfo(videoNumber,
                        new CallbackMessage<ThumbInfo, String>(
                                handler,
                                MSG_ID_THUMBINFO_FINISHED_NEW,
                                MSG_ID_THUMBINFO_OCCURRED_ERROR_NEW));
            }
        } else {
            mThumbInfo = thumbInfo;
            if (handler != null) {
                handler.obtainMessage(MSG_ID_THUMBINFO_FINISHED,
                        thumbInfo.getLength()).sendToTarget();
            }
        }
    }

    protected void seekBySecondCommon(int second) {
        // コメントデータ初期化
        // TODO 排他制御どうするか？
        mMessageData.resetChats(mMessageLoader.getChats());
        mMessageDataFork.resetChats(mMessageLoaderFork.getChats());
    }

    protected StringBuilder appendCurrentPlayTimeCommon(StringBuilder builder,
            final long posNum, final int posDen) {
        if (posDen != 0) {
            int minutes = (int) (posNum / ((long) posDen * 60));
            int seconds = (int) (posNum / posDen % 60);
            return Util.appendPlayTime(builder, minutes, seconds);
        } else {
            return builder.append(INFO_TIME_DEFAULT);
        }
    }

    protected void setButtonPauseImage() {
        // TODO 直接PlayerActivity使って良いかどうか
        PlayerActivity activity = getPlayerActivity();
        if (activity != null) {
            activity.setButtonPauseImage();
        }
    }

    protected PlayerActivity getPlayerActivity() {
        Activity activity = getActivity();
        if (activity == null) {
            Log.w(LOG_TAG, "AbstractPlayerFragment#getPlayerActivity() is called when activity is null");
            if (DEBUG_LOGD) {
                Log.d(LOG_TAG, "stack trace", new Throwable());
            }
        }
        return (PlayerActivity) activity;
    }

    public ThumbInfo getThumbInfo() {
        return mThumbInfo;
    }

    protected boolean getMessageDisable() {
        PlayerActivity activity = getPlayerActivity();
        if (activity == null) {
            return false;
        } else {
            return activity.getMessageDisable();
        }
    }

    public void moveThumbInfoFrom(AbstractPlayerFragment src) {
        mThumbInfo = src.mThumbInfo;
        src.mThumbInfo = null;
    }

    public void moveVideoLoaderFrom(AbstractPlayerFragment src) {
        mVideoLoader = src.mVideoLoader;
        src.mVideoLoader = null;
    }

    public void moveMessagesFrom(AbstractPlayerFragment src) {
        mMessageLoader = src.mMessageLoader;
        src.mMessageLoader = null;
        mMessageLoaderFork = src.mMessageLoaderFork;
        src.mMessageLoaderFork = null;
    }

    protected void initializeVideoLoader() {
        Bundle args = getArguments();
        if (mVideoLoader != null) {
            if (TextUtils.equals(args.getString(INTENT_NAME_VIDEO_URL),
                    mVideoLoader.getUrl())) {
                // VideoLoader is set already
                VideoLoader.EventListener listener = createVideoLoaderEventListener();
                mVideoLoader.setEventListener(listener);
                Handler handler = mHandler;
                if (handler != null) {
                    if (mVideoLoader.hasCache()) {
                        listener.onCached(mVideoLoader);
                    }
                    if (mVideoLoader.isCacheCompleted()) {
                        listener.onFinished(mVideoLoader);
                    }
                }
                return;
            }
        }

        mVideoLoader = createVideoLoader(args, mContext);
        if (mVideoLoader != null) {
            mVideoLoader.setEventListener(createVideoLoaderEventListener());
            // 最初にダウンロード要求
            mVideoLoader.startLoad();
        }
    }
}
