package jp.sourceforge.nicoro;

import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.test.AndroidTestCase;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.SeekBar;
import android.widget.ToggleButton;

public class PlayerControllerViewsTest extends AndroidTestCase {
    private LayoutInflater mInflater;
    private View mRootView;
    private Resources mRes;
    private PlayerControllerViews mViews;

    @Override
    protected void setUp() {
        mInflater = LayoutInflater.from(getContext());
        assertNotNull(mInflater);
        mRootView = createRootView(mInflater);
        assertNotNull(mRootView);
        mRes = getContext().getResources();
        mViews = new PlayerControllerViews();
        mViews.initializeView(mRootView);
    }

    @Override
    protected void tearDown() {
        mInflater = null;
        mRootView = null;
        mRes = null;
        mViews = null;
    }

    public void testSetTime() {
        VariableLabelView controllerTime = Util.findViewById(mRootView,
                R.id.controller_time);
        final String time = "02:05/25:25";
        mViews.setTime(time);
        if (controllerTime != null) {
            assertEquals(time, controllerTime.getText());
        }
    }

    public void testCopyLayoutParamsFrom() {
        Configuration configOrg = mRes.getConfiguration();
        try {
            Configuration config = new Configuration(configOrg);
            if (config.orientation == Configuration.ORIENTATION_PORTRAIT) {
                config.orientation = Configuration.ORIENTATION_LANDSCAPE;
            } else if (config.orientation == Configuration.ORIENTATION_LANDSCAPE) {
                config.orientation = Configuration.ORIENTATION_PORTRAIT;
            }
            mRes.updateConfiguration(config, mRes.getDisplayMetrics());
            ViewGroup anotherRootView =
                (ViewGroup) createRootView(LayoutInflater.from(getContext()));
            assertNotNull(anotherRootView);

            mViews.copyLayoutParamsFrom(anotherRootView);

            TestUtil.checkLayoutParams(mRootView, anotherRootView);
        } finally {
            mRes.updateConfiguration(configOrg, mRes.getDisplayMetrics());
        }
    }

    public void testSetButtonPauseImage() {
        Bitmap bitmapPlay = ((BitmapDrawable) mRes.getDrawable(
                android.R.drawable.ic_media_play)).getBitmap();
        Bitmap bitmapPause = ((BitmapDrawable) mRes.getDrawable(
                android.R.drawable.ic_media_pause)).getBitmap();

        ImageView buttonPause = Util.findViewById(mRootView, R.id.button_pause);
        if (buttonPause == null) {
            mViews.setButtonPauseImage(true);
            mViews.setButtonPauseImage(false);
        } else {
            Bitmap bitmap = ((BitmapDrawable) buttonPause.getDrawable()).getBitmap();
            TestUtil.checkBitmap(bitmapPause, bitmap);

            mViews.setButtonPauseImage(true);
            bitmap = ((BitmapDrawable) buttonPause.getDrawable()).getBitmap();
            TestUtil.checkBitmap(bitmapPlay, bitmap);

            mViews.setButtonPauseImage(false);
            bitmap = ((BitmapDrawable) buttonPause.getDrawable()).getBitmap();
            TestUtil.checkBitmap(bitmapPause, bitmap);
        }
    }

    public void testSetEnabledSeekController() {
        View seekBar = mRootView.findViewById(R.id.seek_bar);
        View buttonFromBegin = mRootView.findViewById(R.id.button_from_begin);
        if (seekBar == null) {
            assertNull(buttonFromBegin);
            mViews.setEnabledSeekController(true);
            mViews.setEnabledSeekController(false);
        } else {
            assertNotNull(buttonFromBegin);
            mViews.setEnabledSeekController(true);
            assertTrue(seekBar.isEnabled());
            assertTrue(buttonFromBegin.isEnabled());
            mViews.setEnabledSeekController(false);
            assertFalse(seekBar.isEnabled());
            assertFalse(buttonFromBegin.isEnabled());
        }
    }

    public void testGetControllerView() {
        ViewGroup controllerView = mViews.getControllerView();
        assertNotNull(controllerView);
        assertEquals(R.id.player_controller, controllerView.getId());
    }

    public void testSetSeekBarMax() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);
        final int second = 12345;
        mViews.setSeekBarMax(second);
        if (seekBar != null) {
            assertEquals(second, seekBar.getMax());
        }
    }

    public void testSetSeekBarProgress() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);
        mViews.setSeekBarProgress(98);
        if (seekBar != null) {
            assertEquals(98, seekBar.getProgress());
        }
    }

    public void testSetSeekBarSecondaryProgress() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);
        mViews.setSeekBarMax(1000);
        mViews.setSeekBarSecondaryProgress(2, 3);
        if (seekBar != null) {
            assertEquals(666, seekBar.getSecondaryProgress());
        }
    }

    public void testSetSeekBarSecondaryMax() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);
        mViews.setSeekBarMax(12345);
        mViews.setSeekBarSecondaryMax();
        if (seekBar != null) {
            assertEquals(12345, seekBar.getSecondaryProgress());
        }
    }

    private class FakeSetOnSeekBarChangeListener implements SeekBar.OnSeekBarChangeListener {
        boolean onProgressChangedCalled = false;
        boolean onStartTrackingTouchCalled = false;
        boolean onStopTrackingTouchCalled = false;

        public void clearFlag() {
            onProgressChangedCalled = false;
            onStartTrackingTouchCalled = false;
            onStopTrackingTouchCalled = false;
        }
        @Override
        public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
            onProgressChangedCalled = true;
        }
        @Override
        public void onStartTrackingTouch(SeekBar seekBar) {
            onStartTrackingTouchCalled = true;
        }
        @Override
        public void onStopTrackingTouch(SeekBar seekBar) {
            onStopTrackingTouchCalled = true;
        }
    }
    public void testSetOnSeekBarChangeListener() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);

        FakeSetOnSeekBarChangeListener listener = new FakeSetOnSeekBarChangeListener();
        mViews.setOnSeekBarChangeListener(listener);
        assertEquals(listener, mViews.mOnSeekBarChangeListener);
        // TODO 中のコールバックのテストは実装面倒なので省略
    }

    public void testCanUpdateSeekBarProgress() {
        SeekBar seekBar = Util.findViewById(mRootView, R.id.seek_bar);

        if (seekBar == null) {
            assertFalse(mViews.canUpdateSeekBarProgress());
        } else {
            seekBar.setEnabled(true);
            mViews.mIsTrackingSeekBar = false;
            assertTrue(mViews.canUpdateSeekBarProgress());

            mViews.mIsTrackingSeekBar = true;
            assertFalse(mViews.canUpdateSeekBarProgress());

            seekBar.setEnabled(false);
            mViews.mIsTrackingSeekBar = false;
            assertFalse(mViews.canUpdateSeekBarProgress());
        }
    }

    public void testSetCheckedButtonCommentOnOff() {
        ToggleButton buttonCommentOnOff = Util.findViewById(mRootView, R.id.button_comment_onoff);
        assertNotNull(buttonCommentOnOff);

        mViews.setCheckedButtonCommentOnOff(false);
        assertFalse(buttonCommentOnOff.isChecked());

        mViews.setCheckedButtonCommentOnOff(true);
        assertTrue(buttonCommentOnOff.isChecked());
    }

    public void testHasControllerTime() {
        VariableLabelView controllerTime = Util.findViewById(mRootView,
                R.id.controller_time);
        if (controllerTime == null) {
            assertFalse(mViews.hasControllerTime());
        } else {
            assertTrue(mViews.hasControllerTime());
        }
    }

    protected View createRootView(LayoutInflater inflater) {
        FrameLayout temp = new FrameLayout(getContext());
        return inflater.inflate(R.layout.player_controller, temp, false);
    }
}
