/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;

import jp.ossc.nimbus.core.*;

/**
 * LbVT[rXۃNXB<p>
 *
 * @author M.Takata
 */
public abstract class AbstractCacheService<E> extends ServiceBase
 implements CacheRemoveListener<E>, AbstractCacheServiceMBean<E>{
    
    private static final long serialVersionUID = 4327482025283418963L;
    
    /**
     * LbVQƂ̏WB<p>
     */
    protected Set<CachedReference<E>> references;
    
    /**
     * OverflowControllerT[rX̃T[rXzB<p>
     */
    protected ServiceName[] overflowControllerServiceNames;
    
    /**
     * OverflowControllerT[rX̃XgB<p>
     */
    protected List<OverflowController<E>> overflowControllers;
    
    /**
     * T[rX~ɁALbVNA邩ǂ̃tOB<p>
     * ftHǵAfalseB<br>
     */
    protected boolean isClearOnStop;
    
    /**
     * T[rXjɁALbVNA邩ǂ̃tOB<p>
     * ftHǵAtrueB<br>
     */
    protected boolean isClearOnDestroy = true;
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public void setOverflowControllerServiceNames(ServiceName[] names){
        overflowControllerServiceNames = names;
    }
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public ServiceName[] getOverflowControllerServiceNames(){
        return overflowControllerServiceNames;
    }
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public void setClearOnStop(boolean isClear){
        isClearOnStop = isClear;
    }
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public boolean isClearOnStop(){
        return isClearOnStop;
    }
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public void setClearOnDestroy(boolean isClear){
        isClearOnDestroy = isClear;
    }
    
    // AbstractCacheServiceMBeanJavaDoc
    @Override
    public boolean isClearOnDestroy(){
        return isClearOnDestroy;
    }
    
    /**
     * OverflowControllerݒ肷B
     */
    public void setOverflowControllers(List<OverflowController<E>> overflowControllers) {
		this.overflowControllers = overflowControllers;
	}

	/**
     * T[rX̐OsB<p>
     * CX^Xϐ̏sB<br>
     *
     * @exception OɎsꍇ
     */
    @Override
    public void preCreateService() throws Exception{
        super.preCreateService();
        references = Collections.synchronizedSet(new HashSet<CachedReference<E>>());
        overflowControllers = new ArrayList<OverflowController<E>>();
    }
    
    /**
     * T[rX̊JnOsB<p>
     * QverflowControllerT[rX̎擾sB<br>
     *
     * @exception JnOɎsꍇ
     */
    @Override
    public void preStartService() throws Exception{
        super.preStartService();
        if(overflowControllerServiceNames != null){
            for(int i = 0; i < overflowControllerServiceNames.length; i++){
                overflowControllers.add(
                    ServiceManagerFactory
                        .<OverflowController<E>>getServiceObject(overflowControllerServiceNames[i])
                );
            }
        }
    }
    
    /**
     * T[rX̒~㏈sB<p>
     * QverflowControllerT[rX̎QƂjB<br>
     *
     * @exception ~㏈Ɏsꍇ
     */
    @Override
    public void postStopService() throws Exception{
        if(isClearOnStop()){
            clear();
        }
        if(overflowControllers != null){
            overflowControllers.clear();
        }
        super.postStopService();
    }
    
    /**
     * T[rX̔j㏈sB<p>
     * CX^Xϐ̎QƂjB<br>
     *
     * @exception j㏈Ɏsꍇ
     */
    @Override
    public void postDestroyService() throws Exception{
        if(isClearOnDestroy()){
            clear();
        }
        references = null;
        overflowControllers = null;
        super.postDestroyService();
    }
    
    // CacheJavaDoc
    @Override
    public CachedReference<E> add(E obj){
        final CachedReference<E> ref = createCachedReference(obj);
        add(ref);
        return ref;
    }
    
    /**
     * w肳ꂽLbVQƂǉB<p>
     *
     * @param ref LbVQ
     */
    protected void add(CachedReference<E> ref){
        if(ref == null || references == null || getState().compareTo(State.STOPPED) > 0){
            return;
        }
        synchronized(references){
            ref.addCacheRemoveListener(this);
            references.add(ref);
            if(overflowControllers.size() != 0){
                for(OverflowController<E> controller : overflowControllers){
                    controller.control(ref);
                }
            }
        }
        return;
    }
    
    /**
     * w肳ꂽIuWFNg̃LbVQƂ𐶐B<p>
     *
     * @param obj LbVIuWFNg
     * @return LbVQ
     */
    protected abstract CachedReference<E> createCachedReference(E obj);
    
    // CacheJavaDoc
    @Override
    public Iterator<CachedReference<E>> iterator(){
        return new CachedReferenceIterator();
    }
    
    // CacheJavaDoc
    @Override
    public boolean contains(Object ref){
        if(references == null){
            return false;
        }
        return references.contains(ref);
    }
    
    // CacheJavaDoc
    @Override
    public boolean containsAll(Collection<?> c){
        if(references == null){
            return false;
        }
        return references.containsAll(c);
    }
    
    // CacheJavaDoc
    @Override
    public boolean isEmpty(){
        if(references == null){
            return true;
        }
        return references.isEmpty();
    }
    
    // CacheJavaDoc
    @SuppressWarnings("unchecked")
    @Override
    public boolean remove(Object ref){
        boolean result = false;
        if(references != null){
            synchronized(references){
                result = references.remove(ref);
                if(result){
                    ((CachedReference<E>)ref).remove(this);
                }
            }
        }
        return result;
    }
    
    // CacheJavaDoc
    @Override
    public boolean removeAll(Collection<?> c){
        boolean result = false;
        for(Object obj : c){
            result |= remove(obj);
        }
        return result;
    }
    
    // CacheJavaDoc
    @Override
    public boolean retainAll(Collection<?> c){
        if(references == null){
            return false;
        }
        boolean result = false;
        synchronized(references){
            for(CachedReference<E> ref : references){
                if(!c.contains(ref)){
                    result |= remove(ref);
                }
            }
        }
        return result;
    }
    
    // CacheJavaDoc
    @Override
    public int size(){
        if(references == null){
            return 0;
        }else{
            return references.size();
        }
    }
    
    // CacheJavaDoc
    @Override
    public void clear(){
        if(references != null){
            synchronized(references){
                final Object[] refs = references.toArray();
                for(int i = 0; i < refs.length; i++){
                    remove(refs[i]);
                }
            }
        }
    }
    
    // CacheJavaDoc
    @Override
    public Object[] toArray(){
        if(references == null){
            return new CachedReference[0];
        }
        return references.toArray(
            new CachedReference[references.size()]
        );
    }
    
    // CacheJavaDoc
    @SuppressWarnings("unchecked")
    @Override
    public <T> CachedReference<T>[] toArray(CachedReference<T>[] refs){
        if(references == null){
            return new CachedReference[0];
        }
        return references.toArray(refs);
    }
    
    // CacheRemoveListenerJavaDoc
    @Override
    public void removed(CachedReference<E> ref){
        if(references != null && references.contains(ref)){
            references.remove(ref);
        }
    }
    
    /**
     * LbṼLbVQƌJԂB<p>
     *
     * @author M.Takata
     * @see AbstractCacheService#iterator()
     */
    private class CachedReferenceIterator
     implements Iterator<CachedReference<E>>, java.io.Serializable{
        
        private static final long serialVersionUID = -6125109804226184416L;
        
        private Iterator<CachedReference<E>> iterator;
        private CachedReference<E> current;
        
        /**
         * CX^X𐶐B<p>
         */
        public CachedReferenceIterator(){
            if(references != null){
                iterator = new HashSet<CachedReference<E>>(references).iterator();
            }
        }
        
        // IteratorJavaDoc
        @Override
        public boolean hasNext(){
            return iterator == null ? false : iterator.hasNext();
        }
        
        // IteratorJavaDoc
        @Override
        public CachedReference<E> next(){
            if(iterator == null){
                throw new NoSuchElementException();
            }
            current = null;
            current = iterator.next();
            return current;
        }
        
        // IteratorJavaDoc
        @Override
        public void remove(){
            if(current == null){
                throw new IllegalStateException();
            }
            iterator.remove();
            AbstractCacheService.this.remove(current);
            if(!iterator.hasNext()){
                current = null;
            }
        }
    }
}
