/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import java.io.*;

/**
 * Ro[^[eBeBB<p>
 * 
 * @author M.Takata
 */
public class Converters{
    
    private static final StringConverter ALPHABET_HANKAKU_TO_ZENKAKU_STRING
         = new StringConverterImpl(
               new AlphabetStringConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final StringConverter ALPHABET_ZENKAKU_TO_HANKAKU_STRING
         = new StringConverterImpl(
               new AlphabetStringConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final StringConverter KATAKANA_HANKAKU_TO_ZENKAKU_STRING
         = new StringConverterImpl(
               new KatakanaStringConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final StringConverter KATAKANA_ZENKAKU_TO_HANKAKU_STRING
         = new StringConverterImpl(
               new KatakanaStringConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final StringConverter NUMBER_HANKAKU_TO_ZENKAKU_STRING
         = new StringConverterImpl(
               new NumberStringConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final StringConverter NUMBER_ZENKAKU_TO_HANKAKU_STRING
         = new StringConverterImpl(
               new NumberStringConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final StringConverter SYMBOL_HANKAKU_TO_ZENKAKU_STRING
         = new StringConverterImpl(
               new SymbolStringConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final StringConverter SYMBOL_ZENKAKU_TO_HANKAKU_STRING
         = new StringConverterImpl(
               new SymbolStringConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final StringConverter KATAKANA_TO_HIRAGANA_STRING
         = new StringConverterImpl(
               new KatakanaHiraganaStringConverter(
                   KatakanaHiraganaStringConverter.KATAKANA_TO_HIRAGANA
               )
           );
    private static final StringConverter HIRAGANA_TO_KATAKANA_STRING
         = new StringConverterImpl(
               new KatakanaHiraganaStringConverter(
                   KatakanaHiraganaStringConverter.HIRAGANA_TO_KATAKANA
               )
           );
    private static final StringConverter HANKAKU_TO_ZENKAKU_STRING
         = new StringConverterImpl(
               new CustomConverter(
                   new Converter[]{
                       new AlphabetStringConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new KatakanaStringConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new NumberStringConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new SymbolStringConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       )
                   }
               )
           );
    private static final StringConverter ZENKAKU_TO_HANKAKU_STRING
         = new StringConverterImpl(
               new CustomConverter(
                   new Converter[]{
                       new AlphabetStringConverter(
                           HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                       ),
                       new KatakanaStringConverter(
                           HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                       ),
                       new NumberStringConverter(
                           HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                       ),
                       new SymbolStringConverter(
                           HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                       )
                   }
               )
           );
    
    private static final CharacterConverter ALPHABET_HANKAKU_TO_ZENKAKU_CHAR
         = new CharacterConverterImpl(
               new AlphabetCharacterConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final CharacterConverter ALPHABET_ZENKAKU_TO_HANKAKU_CHAR
         = new CharacterConverterImpl(
               new AlphabetCharacterConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final CharacterConverter KATAKANA_HANKAKU_TO_ZENKAKU_CHAR
         = new CharacterConverterImpl(
               new KatakanaCharacterConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final CharacterConverter KATAKANA_ZENKAKU_TO_HANKAKU_CHAR
         = new CharacterConverterImpl(
               new KatakanaCharacterConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final CharacterConverter NUMBER_HANKAKU_TO_ZENKAKU_CHAR
         = new CharacterConverterImpl(
               new NumberCharacterConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final CharacterConverter NUMBER_ZENKAKU_TO_HANKAKU_CHAR
         = new CharacterConverterImpl(
               new NumberCharacterConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final CharacterConverter SYMBOL_HANKAKU_TO_ZENKAKU_CHAR
         = new CharacterConverterImpl(
               new SymbolCharacterConverter(
                   HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
               )
           );
    private static final CharacterConverter SYMBOL_ZENKAKU_TO_HANKAKU_CHAR
         = new CharacterConverterImpl(
               new SymbolCharacterConverter(
                   HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
               )
           );
    private static final CharacterConverter KATAKANA_TO_HIRAGANA_CHAR
         = new CharacterConverterImpl(
               new KatakanaHiraganaCharacterConverter(
                   KatakanaHiraganaCharacterConverter.KATAKANA_TO_HIRAGANA
               )
           );
    private static final CharacterConverter HIRAGANA_TO_KATAKANA_CHAR
         = new CharacterConverterImpl(
               new KatakanaHiraganaCharacterConverter(
                   KatakanaHiraganaCharacterConverter.HIRAGANA_TO_KATAKANA
               )
           );
    private static final CharacterConverter HANKAKU_TO_ZENKAKU_CHAR
         = new CharacterConverterImpl(
               new CustomConverter(
                   new Converter[]{
                       new AlphabetCharacterConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new KatakanaCharacterConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new NumberCharacterConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       ),
                       new SymbolCharacterConverter(
                           HankakuZenkakuConverter.HANKAKU_TO_ZENKAKU
                       )
                   }
               )
           );
    private static final CharacterConverter ZENKAKU_TO_HANKAKU_CHAR
         = new CharacterConverterImpl(
              new CustomConverter(
                  new Converter[]{
                      new AlphabetCharacterConverter(
                          HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                      ),
                      new KatakanaCharacterConverter(
                          HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                      ),
                      new NumberCharacterConverter(
                          HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                      ),
                       new SymbolCharacterConverter(
                          HankakuZenkakuConverter.ZENKAKU_TO_HANKAKU
                      )
                  }
              )
         );
    
    private Converters(){
    }
    
    /**
     * ppSppRo[^擾B<p>
     *
     * @return ppSppRo[^
     * @see AlphabetStringConverter
     */
    public static StringConverter getAlphabetHankakuToZenkakuStringConverter(){
        return (StringConverter)ALPHABET_HANKAKU_TO_ZENKAKU_STRING;
    }
    
    /**
     * SppppRo[^擾B<p>
     *
     * @return SppppRo[^
     * @see AlphabetStringConverter
     */
    public static StringConverter getAlphabetZenkakuToHankakuStringConverter(){
        return (StringConverter)ALPHABET_ZENKAKU_TO_HANKAKU_STRING;
    }
    
    /**
     * pJ^JiSpJ^JiRo[^擾B<p>
     *
     * @return pJ^JiSpJ^JiRo[^
     * @see KatakanaStringConverter
     */
    public static StringConverter getKatakanaHankakuToZenkakuStringConverter(){
        return (StringConverter)KATAKANA_HANKAKU_TO_ZENKAKU_STRING;
    }
    
    /**
     * SpJ^JipJ^JiRo[^擾B<p>
     *
     * @return SpJ^JipJ^JiRo[^
     * @see KatakanaStringConverter
     */
    public static StringConverter getKatakanaZenkakuToHankakuStringConverter(){
        return (StringConverter)KATAKANA_ZENKAKU_TO_HANKAKU_STRING;
    }
    
    /**
     * pSpRo[^擾B<p>
     *
     * @return pSpRo[^
     * @see NumberStringConverter
     */
    public static StringConverter getNumberHankakuToZenkakuStringConverter(){
        return (StringConverter)NUMBER_HANKAKU_TO_ZENKAKU_STRING;
    }
    
    /**
     * SppRo[^擾B<p>
     *
     * @return SppRo[^
     * @see NumberStringConverter
     */
    public static StringConverter getNumberZenkakuToHankakuStringConverter(){
        return (StringConverter)NUMBER_ZENKAKU_TO_HANKAKU_STRING;
    }
    
    /**
     * pLSpLRo[^擾B<p>
     *
     * @return pLSpLRo[^
     * @see SymbolStringConverter
     */
    public static StringConverter getSymbolHankakuToZenkakuStringConverter(){
        return (StringConverter)SYMBOL_HANKAKU_TO_ZENKAKU_STRING;
    }
    
    /**
     * SpLpLRo[^擾B<p>
     *
     * @return SpLpLRo[^
     * @see SymbolStringConverter
     */
    public static StringConverter getSymbolZenkakuToHankakuStringConverter(){
        return (StringConverter)SYMBOL_ZENKAKU_TO_HANKAKU_STRING;
    }
    
    /**
     * SpJ^JiSpЂ炪ȃRo[^擾B<p>
     *
     * @return SpJ^JiSpЂ炪ȃRo[^
     * @see KatakanaHiraganaStringConverter
     */
    public static StringConverter getKatakanaToHiraganaStringConverter(){
        return (StringConverter)KATAKANA_TO_HIRAGANA_STRING;
    }
    
    /**
     * SpЂ炪ȁSpJ^JiRo[^擾B<p>
     *
     * @return SpЂ炪ȁSpJ^JiRo[^
     * @see KatakanaHiraganaStringConverter
     */
    public static StringConverter getHiraganaToKatakanaStringConverter(){
        return (StringConverter)HIRAGANA_TO_KATAKANA_STRING;
    }
    
    /**
     * pSpRo[^擾B<p>
     * pAJ^JiAAL̔pSpϊsRo[^擾B
     *
     * @return pSpRo[^
     * @see AlphabetStringConverter
     * @see KatakanaStringConverter
     * @see NumberStringConverter
     * @see SymbolStringConverter
     */
    public static StringConverter getHankakuToZenkakuStringConverter(){
        return (StringConverter)HANKAKU_TO_ZENKAKU_STRING;
    }
    
    /**
     * SppRo[^擾B<p>
     * pAJ^JiAAL̑SppϊsRo[^擾B
     *
     * @return SppRo[^
     * @see AlphabetStringConverter
     * @see KatakanaStringConverter
     * @see NumberStringConverter
     * @see SymbolStringConverter
     */
    public static StringConverter getZenkakuToHankakuStringConverter(){
        return (StringConverter)ZENKAKU_TO_HANKAKU_STRING;
    }
    
    /**
     * ppSppRo[^擾B<p>
     *
     * @return ppSppRo[^
     * @see AlphabetCharacterConverter
     */
    public static CharacterConverter getAlphabetHankakuToZenkakuCharacterConverter(){
        return (CharacterConverter)ALPHABET_HANKAKU_TO_ZENKAKU_CHAR;
    }
    /**
     * SppppRo[^擾B<p>
     *
     * @return SppppRo[^
     * @see AlphabetCharacterConverter
     */
    public static CharacterConverter getAlphabetZenkakuToHankakuCharacterConverter(){
        return (CharacterConverter)ALPHABET_ZENKAKU_TO_HANKAKU_CHAR;
    }
    /**
     * pJ^JiSpJ^JiRo[^擾B<p>
     *
     * @return pJ^JiSpJ^JiRo[^
     * @see KatakanaCharacterConverter
     */
    public static CharacterConverter getKatakanaHankakuToZenkakuCharacterConverter(){
        return (CharacterConverter)KATAKANA_HANKAKU_TO_ZENKAKU_CHAR;
    }
    /**
     * SpJ^JipJ^JiRo[^擾B<p>
     *
     * @return SpJ^JipJ^JiRo[^
     * @see KatakanaCharacterConverter
     */
    public static CharacterConverter getKatakanaZenkakuToHankakuCharacterConverter(){
        return (CharacterConverter)KATAKANA_ZENKAKU_TO_HANKAKU_CHAR;
    }
    /**
     * pSpRo[^擾B<p>
     *
     * @return pSpRo[^
     * @see NumberCharacterConverter
     */
    public static CharacterConverter getNumberHankakuToZenkakuCharacterConverter(){
        return (CharacterConverter)NUMBER_HANKAKU_TO_ZENKAKU_CHAR;
    }
    /**
     * SppRo[^擾B<p>
     *
     * @return SppRo[^
     * @see NumberCharacterConverter
     */
    public static CharacterConverter getNumberZenkakuToHankakuCharacterConverter(){
        return (CharacterConverter)NUMBER_ZENKAKU_TO_HANKAKU_CHAR;
    }
    /**
     * pLSpLRo[^擾B<p>
     *
     * @return pLSpLRo[^
     * @see SymbolCharacterConverter
     */
    public static CharacterConverter getSymbolHankakuToZenkakuCharacterConverter(){
        return (CharacterConverter)SYMBOL_HANKAKU_TO_ZENKAKU_CHAR;
    }
    /**
     * SpLpLRo[^擾B<p>
     *
     * @return SpLpLRo[^
     * @see SymbolCharacterConverter
     */
    public static CharacterConverter getSymbolZenkakuToHankakuCharacterConverter(){
        return (CharacterConverter)SYMBOL_ZENKAKU_TO_HANKAKU_CHAR;
    }
    /**
     * SpJ^JiSpЂ炪ȃRo[^擾B<p>
     *
     * @return SpJ^JiSpЂ炪ȃRo[^
     * @see KatakanaHiraganaCharacterConverter
     */
    public static CharacterConverter getKatakanaToHiraganaCharacterConverter(){
        return (CharacterConverter)KATAKANA_TO_HIRAGANA_CHAR;
    }
    /**
     * SpЂ炪ȁSpJ^JiRo[^擾B<p>
     *
     * @return SpЂ炪ȁSpJ^JiRo[^
     * @see KatakanaHiraganaCharacterConverter
     */
    public static CharacterConverter getHiraganaToKatakanaCharacterConverter(){
        return (CharacterConverter)HIRAGANA_TO_KATAKANA_CHAR;
    }
    /**
     * pSpRo[^擾B<p>
     * pAJ^JiAAL̔pSpϊsRo[^擾B
     *
     * @return pSpRo[^
     * @see AlphabetCharacterConverter
     * @see KatakanaCharacterConverter
     * @see NumberCharacterConverter
     * @see SymbolCharacterConverter
     */
    public static CharacterConverter getHankakuToZenkakuCharacterConverter(){
        return (CharacterConverter)HANKAKU_TO_ZENKAKU_CHAR;
    }
    /**
     * SppRo[^擾B<p>
     * pAJ^JiAAL̑SppϊsRo[^擾B
     *
     * @return SppRo[^
     * @see AlphabetCharacterConverter
     * @see KatakanaCharacterConverter
     * @see NumberCharacterConverter
     * @see SymbolCharacterConverter
     */
    public static CharacterConverter getZenkakuToHankakuCharacterConverter(){
        return (CharacterConverter)ZENKAKU_TO_HANKAKU_CHAR;
    }
    
    /**
     * JX^Ro[^𐶐B<p>
     *
     * @param type ϊ
     * @param fromStrs ϊ㕶z
     * @param toStrs ϊΏەz
     * @return JX^Ro[^
     * @see ReversibleConverter#POSITIVE_CONVERT
     * @see ReversibleConverter#REVERSE_CONVERT
     * @see CustomStringConverter
     */
    public static  StringConverter newCustomStringConverter(
        int type,
        String[] fromStrs,
        String[] toStrs
    ){
        return new CustomStringConverter(type, fromStrs, toStrs);
    }
    
    /**
     * JX^Ro[^𐶐B<p>
     *
     * @param type ϊ
     * @param fromChars ϊ㕶z
     * @param toChars ϊΏەz
     * @return JX^Ro[^
     * @see ReversibleConverter#POSITIVE_CONVERT
     * @see ReversibleConverter#REVERSE_CONVERT
     * @see CustomStringConverter
     */
    public static  StringConverter newCustomStringConverter(
        int type,
        char[] fromChars,
        char[] toChars
    ){
        return new CustomStringConverter(type, fromChars, toChars);
    }
    
    /**
     * JX^Ro[^𐶐B<p>
     *
     * @param type ϊ
     * @param fromStrs ϊ㕶z
     * @param toStrs ϊΏەz
     * @param fromChars ϊ㕶z
     * @param toChars ϊΏەz
     * @return JX^Ro[^
     * @see ReversibleConverter#POSITIVE_CONVERT
     * @see ReversibleConverter#REVERSE_CONVERT
     * @see CustomStringConverter
     */
    public static  StringConverter newCustomStringConverter(
        int type,
        String[] fromStrs,
        String[] toStrs,
        char[] fromChars,
        char[] toChars
    ){
        return new CustomStringConverter(
            type,
            fromStrs,
            toStrs,
            fromChars,
            toChars
        );
    }
    
    /**
     * ̃JX^Ro[^𐶐B<p>
     * 
     * @return ̃JX^Ro[^
     */
    public static  CustomConverter newCustomConverter(){
        return new CustomConverter();
    }
    
    /**
     * w肵Ro[^AJX^Ro[^𐶐B<p>
     * 
     * @param convs Ro[^z
     * @return JX^Ro[^
     */
    public static  CustomConverter newCustomConverter(Converter[] convs){
        return new CustomConverter(convs);
    }
    
    /**
     * w肵K\̃Ro[^𐶐B<p>
     *
     * @return K\Ro[^
     */
    public static  PatternStringConverter patternStringConverter(){
        return new PatternStringConverter();
    }
    
    /**
     * w肵K\̃Ro[^𐶐B<p>
     *
     * @param flags }b`OtO
     * @return K\Ro[^
     */
    public static  PatternStringConverter patternStringConverter(int flags){
        return new PatternStringConverter(flags);
    }
    
    /**
     * w肵K\̃Ro[^𐶐B<p>
     *
     * @param flags }b`OtO
     * @param fromStrs ϊ鐳K\z
     * @param toStrs ϊΏەz
     * @return K\Ro[^
     */
    public static  PatternStringConverter patternStringConverter(
        int flags,
        String[] fromStrs,
        String[] toStrs
    ){
        return new PatternStringConverter(flags,fromStrs,toStrs);
    }
    
    private static final class StringConverterImpl
     implements StringConverter, Serializable{
        
        private static final long serialVersionUID = 1416548061709103644L;
        
        private Converter converter;
        @SuppressWarnings("unused")
		public StringConverterImpl(Converter conv){
            converter = conv;
        }
        public StringConverterImpl(StringConverter conv){
            converter = conv;
        }
        @Override
        public Object convert(Object obj) throws ConvertException{
            return converter.convert(obj);
        }
        @Override
        public String convert(String obj) throws ConvertException{
            return (String)converter.convert(obj);
        }
    }
    
    private static final class CharacterConverterImpl
     implements CharacterConverter, Serializable{
        
        private static final long serialVersionUID = -3076044124853526944L;
        
        private Converter converter;
        @SuppressWarnings("unused")
		public CharacterConverterImpl(Converter conv){
            converter = conv;
        }
        public CharacterConverterImpl(CharacterConverter conv){
            converter = conv;
        }
        @Override
        public Object convert(Object obj) throws ConvertException{
            return converter.convert(obj);
        }
        @Override
        public Character convert(Character c) throws ConvertException{
            return (Character)converter.convert(c);
        }
        @Override
        public char convert(char c) throws ConvertException{
            if(converter instanceof CharacterConverter){
                return ((CharacterConverter)converter).convert(c);
            }else{
                return c;
            }
        }
    }
}
